/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ametys.cms.data.File;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItemGroup;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Provide all the necessary method to add an illustration to a question type
 */
public interface IllustrableFormQuestionType
{
    /** Constant for picture data attribute. */
    public static final String ATTRIBUTE_PICTURE = "picture";
    /** Constant for picture alternative attribute. */
    public static final String ATTRIBUTE_PICTURE_ALTERNATIVE = "picture-alternative";
    
    /**
     * Provide a list of model items that must be included in the model items return by the method FormQuestionType.getModel
     * @return a list of model items
     */
    public default List<ModelItem> getIllustrationModelItems()
    {
        ElementDefinition illustration = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_PICTURE, org.ametys.cms.data.type.ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE_DESC", null);
        illustration.setWidget("edition.file");
        Map<String, I18nizableText> widgetParams = new HashMap<>();
        widgetParams.put("allowSources", new I18nizableText("external"));
        widgetParams.put("filter", new I18nizableText("image"));
        illustration.setWidgetParameters(widgetParams);
        
        ElementDefinition illustrationAlternative = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_PICTURE_ALTERNATIVE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE_ALT", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE_ALT_DESC", null);
        
        return List.of(illustration, illustrationAlternative);
    }
    
    /**
     * Get the view's common illustration tab 
     * @param model the model of the question type
     * @return the illustration tab as SimpleViewItemGroup
     */
    public default SimpleViewItemGroup getIllustrationTab(Model model) 
    {
        SimpleViewItemGroup pictureFieldset = new SimpleViewItemGroup();
        pictureFieldset.setName("illustration");
        pictureFieldset.setLabel(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_IMAGE_FIELDSET"));
        pictureFieldset.setRole(ViewItemGroup.TAB_ROLE);
        
        ViewElement picture = new ViewElement();
        picture.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_PICTURE));
        pictureFieldset.addViewItem(picture);
        
        ViewElement pictureAlternative = new ViewElement();
        pictureAlternative.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_PICTURE_ALTERNATIVE));
        pictureFieldset.addViewItem(pictureAlternative);
        
        return pictureFieldset;
    }
    
    /**
     * Get the question illustration
     * @param question the question
     * @return the illustration
     */
    public default File getIllustration(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_PICTURE);
    }
    
    /**
     * Get the alternative of illustration
     * @param question the question
     * @return the alternative of illustration
     */
    public default String getIllustrationAlternative(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_PICTURE_ALTERNATIVE);
    }
}
