/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.ListUtils;

import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItemGroup;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Provide all the necessary method to add workflow restriction to a question type
 */
public interface RestrictiveQuestionType
{
    /** Constant for reading workflows data attribute. */
    public static final String ATTRIBUTE_READING_CHECKBOX = "readingCheckbox";
    /** Constant for reading workflows data attribute. */
    public static final String ATTRIBUTE_WRITING_CHECKBOX = "writingCheckbox";
    /** Constant for reading workflows data attribute. */
    public static final String ATTRIBUTE_READING = "reading";
    /** Constant for writing workflows data attribute. */
    public static final String ATTRIBUTE_WRITING = "writing";
    /** Constant for writing workflows data attribute. */
    public static final Long INITIAL_WORKFLOW_ID = 0L;
    
    /**
     * Provide a list of model items that must be included in the model items return by the method FormQuestionType.getModel
     * @return a list of model items
     */
    public default List<ModelItem> getRestrictiveModelItems()
    {
        ElementDefinition readingCheckbox = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_READING_CHECKBOX, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING_CHECKBOX", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING_CHECKBOX_DESC", null);
        ElementDefinition<Long> reading = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_READING, ModelItemTypeConstants.LONG_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING_DESC", null);
        reading.setMultiple(true);
        
        Map<String, I18nizableText> readingWidgetParameters  = new HashMap<>();
        readingWidgetParameters.put("naturalOrder", new I18nizableText("true"));
        reading.setWidgetParameters(readingWidgetParameters);
        Map<String, I18nizableText> widgetParameters  = new HashMap<>();
        widgetParameters.put("emptyText", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_READING_PLACEHOLDER"));
        reading.setWidgetParameters(widgetParameters);
        
        DisableConditions disableConditions = new DisableConditions();
        DisableCondition condition = new DisableCondition(ATTRIBUTE_READING_CHECKBOX, OPERATOR.NEQ, "true"); 
        disableConditions.getConditions().add(condition);
        reading.setDisableConditions(disableConditions);
        
        ElementDefinition writingCheckbox = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_WRITING_CHECKBOX, ModelItemTypeConstants.BOOLEAN_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_WRITING_CHECKBOX", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_WRITING_CHECKBOX_DESC", null);
        ElementDefinition<Long> writing = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_WRITING, ModelItemTypeConstants.LONG_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_WRITING", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_WRITING_DESC", new DefaultValidator(null, true));
        writing.setMultiple(true);
        
        Map<String, I18nizableText> writingWidgetParameters  = new HashMap<>();
        writingWidgetParameters.put("naturalOrder", new I18nizableText("true"));
        writing.setWidgetParameters(writingWidgetParameters);

        DisableConditions writingDisableConditions = new DisableConditions();
        DisableCondition writingCondition = new DisableCondition(ATTRIBUTE_WRITING_CHECKBOX, OPERATOR.NEQ, "true"); 
        writingDisableConditions.getConditions().add(writingCondition);
        writing.setDisableConditions(writingDisableConditions);
        
        return List.of(readingCheckbox, reading, writingCheckbox, writing);
    }
    
    /**
     * Get the workflow descriptor of the form
     * @param form the form
     * @return he workflow descriptor
     */
    public WorkflowDescriptor getWorkflowDescriptor(Form form); 
    
    /**
     * Get the view's common illustration tab 
     * @param model the model of the question type
     * @param form the form
     * @return the illustration tab as SimpleViewItemGroup
     */
    public default SimpleViewItemGroup getRestrictiveTab(Model model, Form form) 
    {
        SimpleViewItemGroup restrictiveFieldset = new SimpleViewItemGroup();
        restrictiveFieldset.setName("restriction");
        restrictiveFieldset.setLabel(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_FIELDSET"));
        restrictiveFieldset.setDescription(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RESTRICTIVE_FIELDSET_DESCRIPTION"));
        restrictiveFieldset.setRole(ViewItemGroup.TAB_ROLE);

        ViewElement readingCheckbox = new ViewElement();
        readingCheckbox.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_READING_CHECKBOX));
        restrictiveFieldset.addViewItem(readingCheckbox);
        
        ViewElement reading = new ViewElement();
        @SuppressWarnings("unchecked")
        ElementDefinition<Long> readingElementDefinition = (ElementDefinition<Long>) model.getModelItem(ATTRIBUTE_READING);
        readingElementDefinition.setEnumerator(_getEnumerator(form, true));
        reading.setDefinition(readingElementDefinition);

        restrictiveFieldset.addViewItem(reading);

        ViewElement writingCheckbox = new ViewElement();
        writingCheckbox.setDefinition((ElementDefinition< ? >) model.getModelItem(ATTRIBUTE_WRITING_CHECKBOX));
        restrictiveFieldset.addViewItem(writingCheckbox);

        ViewElement writing = new ViewElement();
        @SuppressWarnings("unchecked")
        ElementDefinition<Long> writingElementDefinition = (ElementDefinition<Long>) model.getModelItem(ATTRIBUTE_WRITING);
        writingElementDefinition.setEnumerator(_getEnumerator(form, false));
        writing.setDefinition(writingElementDefinition);
        
        restrictiveFieldset.addViewItem(writing);

        
        return restrictiveFieldset;
    }
    
    private StaticEnumerator<Long> _getEnumerator(Form form, boolean forReading)
    {
        StaticEnumerator<Long> extensionEnum = new StaticEnumerator<>();
        
        WorkflowDescriptor workflowDesc = getWorkflowDescriptor(form);
        List<StepDescriptor> steps = workflowDesc.getSteps();

        for (StepDescriptor step : steps)
        {
            if (forReading || _hasEditAction(step))
            {
                extensionEnum.add(new I18nizableText("plugin.forms", step.getName()), Long.valueOf(step.getId()));
            }
        }
        
        return extensionEnum;
    }
    
    @SuppressWarnings("unchecked")
    private boolean _hasEditAction(StepDescriptor step)
    {
        List<ActionDescriptor> actions = step.getActions();
        return actions.stream()
                .map(ActionDescriptor::getMetaAttributes)
                .filter(m -> "edit".equals(m.getOrDefault("action-type", null)))
                .findAny()
                .isPresent();
    }
    
    /**
     * <code>true</code> if the question is read restricted
     * @param question the question
     * @return <code>true</code> if the question is read restricted
     */
    public default boolean isReadRestricted(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_READING_CHECKBOX, false, false);
    }

    /**
     * <code>true</code> if the question is modifiable
     * @param question the question
     * @return <code>true</code> if the question is  modifiable
     */
    public default boolean isModifiable(FormQuestion question)
    {
        return question.getValue(ATTRIBUTE_WRITING_CHECKBOX, false, false);
    }

    /**
     * Get the steps that allows reading
     * @param question the question
     * @return the steps that allows reading
     */
    public default List<Long> getReadingSteps(FormQuestion question)
    {
        if (question.hasValue(ATTRIBUTE_READING))
        {
            return Arrays.asList(question.getValue(ATTRIBUTE_READING));
        }
        else
        {
            return List.of();
        }
    }

    /**
     * Get the steps that allows writing
     * @param question the question
     * @return the steps that allows writing
     */
    public default List<Long> getWritingSteps(FormQuestion question)
    {
        if (question.hasValue(ATTRIBUTE_WRITING))
        {
            return ListUtils.sum(Arrays.asList(question.getValue(ATTRIBUTE_WRITING)), List.of(INITIAL_WORKFLOW_ID));
        }
        else
        {
            return List.of(INITIAL_WORKFLOW_ID);
        }
    }
}
