/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository.type;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.type.RepositoryElementType;
import org.ametys.runtime.model.exception.BadItemTypeException;

/**
 * Class for rule element types
 */
public class RuleRepositoryElementType extends AbstractRuleElementType implements RepositoryElementType<Rule>
{
    public String getRepositoryDataType()
    {
        return RepositoryData.STRING_REPOSITORY_DATA_TYPE;
    }

    public Object read(RepositoryData parentData, String name) throws BadItemTypeException
    {
        if (!parentData.hasValue(name))
        {
            return null;
        }
        
        if (!isCompatible(parentData, name))
        {
            throw new BadItemTypeException("Try to get " + getId() + " value from the non " + getId() + " data '" + name + "' on '" + parentData + "'");
        }
        
        if (parentData.isMultiple(name))
        {
            String[] values = parentData.getStrings(name);
            return Arrays.stream(values)
                        .map(this::_string2Rule)
                        .toArray(Rule[]::new);
        }
        else
        {
            String value = parentData.getString(name);
            return _string2Rule(value);
        }
    }
    
    public boolean hasNonEmptyValue(RepositoryData parentData, String name) throws BadItemTypeException
    {
        if (!parentData.hasValue(name))
        {
            return false;
        }
        
        if (!isCompatible(parentData, name))
        {
            throw new BadItemTypeException("Try to check rule value from the non " + getId() + " data '" + name + "' on '" + parentData + "'");
        }
        
        if (parentData.isMultiple(name))
        {
            return parentData.getStrings(name).length > 0;
        }
        else
        {
            return StringUtils.isNotEmpty(parentData.getString(name));
        }
    }

    public void write(ModifiableRepositoryData parentData, String name, Object value) throws BadItemTypeException
    {
        if (value == null)
        {
            if (parentData.hasValue(name) && parentData.isMultiple(name))
            {
                parentData.setValues(name, new String[0]);
            }
            else
            {
                parentData.setValue(name, StringUtils.EMPTY);
            }
        }
        else if (value instanceof String)
        {
            parentData.setValue(name, (String) value);
        }
        else if (value instanceof String[])
        {
            Arrays.stream((String[]) value)
                  .forEach(v -> Optional.ofNullable(v)
                                        .orElseThrow(() -> new IllegalArgumentException("Try to set a null value into the multiple " + getId() + " data '" + name + "' on '" + parentData + "'")));
            
            parentData.setValues(name, (String[]) value);
        }
        else if (value instanceof Rule)
        {
            Rule rule = (Rule) value;
            parentData.setValue(name, _jsonUtils.convertObjectToJson(rule.toJson()));
        }
        else if (value instanceof Rule[])
        {
            String[] values = Stream.of((Rule[]) value)
                .map(r -> _jsonUtils.convertObjectToJson(r.toJson()))
                .toArray(String[]::new);
            
            parentData.setValues(name, values);
        }
        else if (value instanceof List)
        {
            @SuppressWarnings("unchecked")
            List<Rule> ruleList = (List<Rule>) value;
            String[] ruleListAsJson = ruleList.stream()
                .map(r -> _jsonUtils.convertObjectToJson(r.toJson()))
                .toArray(String[]::new);
            
            parentData.setValues(name, ruleListAsJson);
        }
        else
        {
            throw new BadItemTypeException("Try to set the non " + getId() + " value '" + value + "' to the " + getId() + " data '" + name + "' on '" + parentData + "'");
        }
    }
}
