/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;

import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * This class creates a button for each workflow step on form entries
 * For each button, an action is created for each action in the step
 */
public class DefaultFormEntryWorkflowStepsClientSideElement extends FormEntryWorkflowStepsClientSideElement
{
    @SuppressWarnings("unchecked")
    @Override
    protected List<Integer> _getAllowedActions(WorkflowDescriptor workflowDescriptor, List<Integer> stepIds, boolean actionsIncludeMode, List<Integer> configuredActions)
    {
        return stepIds.stream()
                .map(stepId -> workflowDescriptor.getStep(stepId).getActions())
                .flatMap(actionsLists -> ((List<ActionDescriptor>) actionsLists).stream())
                .distinct()
                .mapToInt(ActionDescriptor::getId)
                .boxed()
                .collect(Collectors.toList());
    }
    
    @Override
    protected List<Integer> _getAllowedSteps(WorkflowDescriptor workflowDescriptor, boolean stepsIncludeMode, List<Integer> configuredSteps)
    {
        return workflowDescriptor.getSteps().stream()
                .mapToInt(step -> ((StepDescriptor) step).getId())
                .boxed()
                .collect(Collectors.toList());
    }
    
    @Override
    protected void _configureParameters(Map<String, Object> parameters) throws ConfigurationException
    {
        super._configureParameters(parameters);
        
        String workflowStepName = (String) parameters.get("workflow-step-name");
        
        // Description is the label for default implementation
        parameters.put("description", new I18nizableText("application", workflowStepName));
        
        String[] icons = new String[]{"-small", "-medium", "-large"};
        for (String icon : icons)
        {
            parameters.put("icon" + icon, new I18nizableText("/plugins/forms/resources/img/workflow/FORMS_DEFAULT_WORKFLOW_STATE" + icon + ".png"));
        }
    }
    
    @Override
    protected void _addActionDescriptionConfiguration(Configuration configuration, DefaultConfiguration classConf, int actionId, String actionName)
    {
        DefaultConfiguration descConf = new DefaultConfiguration("description");
        descConf.setAttribute("i18n", configuration.getChild("menu-" + actionId + "-description").getAttribute("i18n", "true"));
        descConf.setValue(configuration.getChild("menu-" + actionId + "-label").getValue(actionName));
        classConf.addChild(descConf);
    }
    
    @Override
    protected void _addActionSendMailConfiguration(Configuration itemConf, DefaultConfiguration classConf, int actionId, Map<String, Object> parameters)
    {
        // No send mail for default configuration
    }
    
    @Override
    protected void _addActionCommentConfiguration(Configuration configuration, DefaultConfiguration classConf, int actionId, String actionName, Map<String, Object> parameters)
    {
        // No comment for default configuration
    }
}
