/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Helper for grids of {@link Ametys.plugins.forms.tool.EntriesSearchTool}s
 * @private
 */
Ext.define('Ametys.plugins.forms.helper.SearchEntriesGridHelper', {
    singleton: true,

    /**
     * Function to render matrix in result grid
     * @param {Object} value The data value
     * @param {Object} metaData A collection of metadata about the current cell
     * @param {Ext.data.Model} record The record
     */
    renderMatrix: function (value, metaData, record)
    {
        if (!Ext.Object.isEmpty(value))
        {
            var htmlTooltip = "";
            var colId = metaData.column.headerId;
            var labels = record.get(colId + "matrice-labels");
            for(var key in value)
            {
                var vals = value[key];
                htmlTooltip += labels.rows[key] + ": ";
                for (var j in vals)
                {
                    if (j != 0)
                    {
                        htmlTooltip += ", ";
                    }
                    htmlTooltip += labels.columns[vals[j]];
                }
                htmlTooltip += "\n";
            }
            return "<span title='" + htmlTooltip + "' >[...]</span>";
            
        }
        return null;
    },
    
    /**
     * Function to render string choice list in result grid
     * @param {Object} value The data value
     * @param {Object} metaData A collection of metadata about the current cell
     * @param {Ext.data.Model} record The record
     */
    renderStringChoiceList: function (value, metaData, record)
    {
        if (!Ext.Object.isEmpty(value))
        {
            return Ext.String.escapeHtml(value.label);
        }
        return null;
    },
    
    /**
     * Function to render user choice list in result grid
     * @param {Object} value The data value
     * @param {Object} metaData A collection of metadata about the current cell
     * @param {Ext.data.Model} record The record
     * @param {Number} rowIndex The index of the current row
     * @param {Number} colIndex The index of the current column
     * @param {Ext.data.Store} store The store
     * @param {Ext.view.View} view The current view
     * @param {String} dataIndex concatenated with '_content' will be get the content additional informations. 
     */
    renderUserChoiceList: function (value, metaData, record, rowIndex, colIndex, store, view, dataIndex)
    {
        if (value == null)
        {
            return null;
        }
        
        if (value.isOther)
        {
            return value.value;
        }
        else
        {
            return Ametys.plugins.cms.search.SearchGridHelper.renderUser(value, metaData, record, rowIndex, colIndex, store, view, dataIndex)
        }
    },
    
    /**
     * Convert a content value for choice list.
     * Content values can be of two kinds
     * @param {String/String[]/Object/Object[]} value If it is an object, it will return the 'id' property.  
     * @param {String} value.id The id
     * @param {Ext.data.Model} record The record
     * @param {String} dataIndex The model data index
     * @return {String/String[]} The ids only
     */
    convertContent: function(value, record, dataIndex)
    {
        var computedValue = null;
        var otherValue = null;
        
        if (Ext.isArray(value) && value.length > 0 && Ext.isObject(value[0]))
        {
            computedValue = [];
            for (var i = 0; i < value.length; i++)
            {
                if (!value[i].isOther)
                {
                    computedValue.push(value[i]);
                }
                else
                {
                    otherValue = value[i];
                }
            }
            
        }
        else if (Ext.isObject(value))
        {
            if (!value.isOther)
            {
                computedValue = value;
            }
            else
            {
                otherValue = value;
            }
        }
        
        var convertValue = Ametys.plugins.cms.search.SearchGridHelper.convertContent(computedValue, record, dataIndex);
        if (otherValue != null)
        {
            if (convertValue != null)
            {
                // If convertValue and otherValue is not null, convertValue is an array
                convertValue.push(otherValue);
            }
            else
            {
                convertValue = otherValue;
            }
        }
        
        return convertValue;
    },
    
    /**
     * Function to render content choice list in result grid
     * @param {Object} value The data value
     * @param {Object} metaData A collection of metadata about the current cell
     * @param {Ext.data.Model} record The record
     * @param {Number} rowIndex The index of the current row
     * @param {Number} colIndex The index of the current column
     * @param {Ext.data.Store} store The store
     * @param {Ext.view.View} view The current view
     * @param {String} dataIndex concatenated with '_content' will be get the content additional informations. 
     */
    renderContentChoiceList: function (value, metaData, record, rowIndex, colIndex, store, view, dataIndex)
    {
        if (value == null)
        {
            return null;
        }
        
        if (value.isOther)
        {
            return value.value;
        }
        else
        {
            return Ametys.plugins.cms.search.SearchGridHelper.renderContent(value, metaData, record, rowIndex, colIndex, store, view, dataIndex)
        }
    },
    
    /**
     * Function to render page in result grid
     * @param {Object} value The data value
     * @param {Object} metaData A collection of metadata about the current cell
     * @param {Ext.data.Model} record The record
     * @param {Number} rowIndex The index of the current row
     * @param {Number} colIndex The index of the current column
     * @param {Ext.data.Store} store The store
     * @param {Ext.view.View} view The current view
     * @param {String} dataIndex concatenated with '_content' will be get the content additional informations. 
     */
    renderPage: function (value, metaData, record, rowIndex, colIndex, store, view, dataIndex)
    {
        if (!Ext.Object.isEmpty(value))
        {
            return '<a href="javascript:(function(){Ametys.tool.ToolsManager.openTool(\'uitool-page\', {id:\'' + value.id + '\'});})()">' + value.title + '</a>';
        }
        
        return null;
    }
    
});
