/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.actions;

import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang.StringUtils;

import org.ametys.core.captcha.CaptchaHelper;
import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.dao.FormEntryDAO.Sort;
import org.ametys.plugins.forms.dao.FormQuestionDAO.FormEntryValues;
import org.ametys.plugins.forms.helper.FormMailHelper;
import org.ametys.plugins.forms.helper.FormMailHelper.LimitationMailType;
import org.ametys.plugins.forms.helper.FormWorkflowHelper;
import org.ametys.plugins.forms.helper.LimitedEntriesHelper;
import org.ametys.plugins.forms.helper.ScheduleOpeningHelper;
import org.ametys.plugins.forms.helper.ScheduleOpeningHelper.FormStatus;
import org.ametys.plugins.forms.question.types.RestrictiveQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;
import org.ametys.web.cache.PageHelper;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.Site;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
/**
 * Process the user entries to the form.
 */
public class ProcessFormAction extends AbstractProcessFormAction
{
    /** The catpcha key */
    public static final String CAPTCHA_KEY = "ametys-captcha";
    
    /** The form workflow helper */
    protected FormWorkflowHelper _formWorkflowHelper;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The users manager */
    protected UserManager _userManager;
    /** the Handle Limited Entries Helper */
    protected LimitedEntriesHelper _limitedEntriesHelper;
    /** The form mail helper */
    protected FormMailHelper _formMailHelper;
    /** The schedule opening helper */
    protected ScheduleOpeningHelper _scheduleOpeningHelper;
    /** The page helper */
    protected PageHelper _pageHelper;
    /** The form entry DAO */
    protected FormEntryDAO _formEntryDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _formWorkflowHelper = (FormWorkflowHelper) serviceManager.lookup(FormWorkflowHelper.ROLE);
        _limitedEntriesHelper = (LimitedEntriesHelper) serviceManager.lookup(LimitedEntriesHelper.ROLE);
        _formMailHelper = (FormMailHelper) serviceManager.lookup(FormMailHelper.ROLE);
        _scheduleOpeningHelper = (ScheduleOpeningHelper) serviceManager.lookup(ScheduleOpeningHelper.ROLE);
        _pageHelper = (PageHelper) serviceManager.lookup(PageHelper.ROLE);
        _formEntryDAO = (FormEntryDAO) serviceManager.lookup(FormEntryDAO.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    { 
        Request request = ObjectModelHelper.getRequest(objectModel);
        Map<String, String> result = _processForm(request);
        if (result == null)
        {
            return null;
        }
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        return EMPTY_MAP;
    }

    @Override
    protected List<FormQuestion> _getRuleFilteredQuestions(Request request, Form form, FormEntryValues entryValues, Optional<Long> currentStepId)
    {
        // Get only readable questions 
        return _formQuestionDAO.getRuleFilteredQuestions(form, entryValues, currentStepId, false, true);
    }
    
    /**
     * Process form
     * @param request the request
     * @return the results
     */
    protected Map<String, String> _processForm(Request request)
    {
        Map<String, String> result = new HashMap<>();
        
        String formId = request.getParameter("formId");
        if (StringUtils.isNotEmpty(formId))
        {
            // Retrieve the current workspace.
            String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
            try
            {
                // Force the workspace.
                RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
                // Get the form object.
                Form form = (Form) _resolver.resolveById(formId);
                if (!_rightManager.currentUserHasReadAccess(form))
                {
                    throw new IllegalAccessError("Can't answer to the form data without convenient right");
                }
                
                UserIdentity user = _currentUserProvider.getUser();
                String clientIp = _limitedEntriesHelper.getClientIp(request);
                
                Multimap<String, I18nizableText> formErrors = ArrayListMultimap.create();
                boolean canUserSubmit = _limitedEntriesHelper.canUserSubmit(form, user, clientIp);
                boolean isOpen = _scheduleOpeningHelper.getStatus(form) == FormStatus.OPEN;
                boolean isConfigured = _formDAO.isFormConfigured(form);
                if (canUserSubmit && isOpen && isConfigured)
                {
                    if (!_checkCaptcha(request, form, formErrors))
                    {
                        request.setAttribute("form", form);
                        request.setAttribute("form-errors", formErrors);
                        return null;
                    }
                    
                    // Add the user entries into jcr.
                    FormEntry entry = _entryDAO.createEntry(form);
                    try
                    {
                        Optional<Long> currentStepId = form.hasWorkflow() ? Optional.of(RestrictiveQuestionType.INITIAL_WORKFLOW_ID) : Optional.empty();
                        
                        View entryView = View.of(entry.getModel());
                        Map<String, Object> formInputValues = _foAmetysObjectCreationHelper.getFormValues(request, entryView, "", formErrors);
                        _adaptFormValuesForChoiceList(form, formInputValues);
                        View filteredEntryView = _getRuleFilteredEntryView(request, form, entryView, new FormEntryValues(formInputValues, null), currentStepId);
                        formErrors.putAll(_foAmetysObjectCreationHelper.validateValues(formInputValues, filteredEntryView));
                        for (FormQuestion question : form.getQuestions())
                        {
                            if (filteredEntryView.hasModelViewItem(question.getNameForForm()))
                            {
                                question.getType().validateEntryValues(question, formInputValues, formErrors, currentStepId, new HashMap<>());
                            }
                        }
                        
                        if (!formErrors.isEmpty())
                        {
                            // If there were errors in the input, store it as a request attribute and stop.
                            request.setAttribute("form", form);
                            request.setAttribute("form-errors", formErrors);
                            return null;
                        }
                        
                        entry.synchronizeValues(filteredEntryView, formInputValues);
                        
                        _handleComputedValues(form.getQuestions(), entry, false);
                        entry.setUser(_currentUserProvider.getUser());
                        entry.setIP(clientIp);
                        entry.setSubmitDate(ZonedDateTime.now());
                        entry.setActive(true);
                        _setEntryId(entry);
                        
                        _formWorkflowHelper.initializeWorkflow(entry);
                        
                        form.saveChanges();
                        
                        // send mail
                        _sendEmails(entry);
                        
                        if (form.isQueueEnabled())
                        {
                            int totalSubmissions = form.getActiveEntries().size();
                            long rankInQueue = totalSubmissions - form.getMaxEntries().get();
                            result.put("isInQueue", String.valueOf(rankInQueue > 0));
                            if (rankInQueue > 0)
                            {
                                result.put("rankInQueue", String.valueOf(rankInQueue));
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        request.setAttribute("form", form);
                        request.setAttribute("form-errors", formErrors);
                        getLogger().error("An error occured while storing entry", e);
                        return null;
                    }
                }
                else
                {
                    if (!canUserSubmit)
                    {
                        formErrors.put("entries-limit-reached", new I18nizableText("plugin.forms", "PLUGINS_FORMS_ENTRIES_LIMIT_REACHED_ERROR"));
                        request.setAttribute("form", form);
                        request.setAttribute("form-errors", formErrors);
                    }
                    
                    if (!isOpen)
                    {
                        formErrors.put("scheduled-not-open", new I18nizableText("plugin.forms", "PLUGINS_FORMS_OPENING_SCHEDULE_PROCESS_ERROR"));
                        request.setAttribute("form", form);
                        request.setAttribute("form-errors", formErrors);
                    }
                    
                    return null;
                }
            }
            finally 
            {
                // Restore context
                RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
            }
        }
        
        return result;
    }
    
    /**
     * Check the captcha if needed
     * @param request the request
     * @param form the form
     * @param formErrors the form errors
     * @return <code>true</code> if the captcha is good
     */
    protected boolean _checkCaptcha(Request request, Form form, Multimap<String, I18nizableText> formErrors)
    {
        String zoneItemId = request.getParameter("ametys-zone-item-id");
        ZoneItem zoneItem = _resolver.resolveById(zoneItemId);
        
        if (!_isFormOnZoneItem(form, zoneItemId))
        {
            throw new IllegalAccessError("The form '" + form.getId() + "' doesn't belong to the zone item '" + zoneItemId + "'");
        }
        
        SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
        Site site = form.getSite();
        String captchaPolicy = site.getValue("display-captcha-policy");
        
        if (_pageHelper.isCaptchaRequired(sitemapElement))
        {
            String captchaValue = request.getParameter(CAPTCHA_KEY);
            String captchaKey = request.getParameter(CAPTCHA_KEY + "-key");
            if (!CaptchaHelper.checkAndInvalidate(captchaKey, captchaValue))
            {
                formErrors.put(CAPTCHA_KEY, new I18nizableText("plugin.forms", "PLUGINS_FORMS_ERROR_CAPTCHA_INVALID"));
                return false;
            }
        }
        else if (captchaPolicy == null || "restricted".equals(captchaPolicy)) 
        {
            if (!_rightManager.currentUserHasReadAccess(sitemapElement))
            {
                throw new IllegalAccessError("The user try to answer to form '" + form.getId() + "' which belong to an other zone item '" + zoneItemId + "'");
            }
        }
        
        return true;
    }
    
    private boolean _isFormOnZoneItem(Form form, String zoneItemId)
    {
        AmetysObjectIterable<ModifiableZoneItem> zoneItems = _formDAO.getFormZoneItems(form.getId(), form.getSiteName());
        
        return zoneItems.stream()
                    .filter(z -> z.getId().equals(zoneItemId))
                    .findAny()
                    .isPresent();
    }
    
    /**
     * Set entry id (auto-incremental id)
     * @param entry the entry
     */
    protected void _setEntryId(FormEntry entry)
    {
        List<FormEntry> formEntries = _formEntryDAO.getFormEntries(entry.getForm(), false, List.of(new Sort(FormEntry.ATTRIBUTE_ID, "descending")));
        Long entryId = formEntries.isEmpty() ? 1L : formEntries.get(0).getEntryId() + 1;
        entry.setEntryId(entryId);
    }
    
    /**
     * Send the receipt and notification emails.
     * @param entry the current entry
     */
    protected void _sendEmails(FormEntry entry)
    {
        Form form = entry.getForm();
        
        Optional<String[]> adminEmails = form.getAdminEmails();
        Optional<String> otherAdminEmails = form.getOtherAdminEmails();
        if (adminEmails.isPresent() || otherAdminEmails.isPresent())
        {
            String[] emailsAsArray = _formMailHelper.getAdminEmails(form, entry);
            
            _formMailHelper.sendEmailsForAdmin(form, entry, emailsAsArray);
            
            if (form.isEntriesLimited())
            {
                int totalSubmissions = form.getActiveEntries().size();
                Long maxEntries = form.getMaxEntries().get();
                if (maxEntries == totalSubmissions)
                {
                    _formMailHelper.sendLimitationReachedMailForAdmin(entry, emailsAsArray, LimitationMailType.LIMIT);
                }
                else if (form.isQueueEnabled() && form.getQueueSize().isPresent() && form.getQueueSize().get() + maxEntries == totalSubmissions)
                {
                    _formMailHelper.sendLimitationReachedMailForAdmin(entry, emailsAsArray, LimitationMailType.QUEUE);
                }
            }
        }

        _formMailHelper.sendReceiptEmail(form, entry);
    }
}
