/*
 * Copyright 2016 Anyware Services
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.workflow;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.Field;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.Field.FieldType;
import org.ametys.plugins.forms.content.data.FieldValue;
import org.ametys.plugins.forms.content.data.UserEntry;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper component allowing to retrieve information in order to send emails
 */
public class MailInformationHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role */
    public static final String ROLE = MailInformationHelper.class.getName();

    /** The email validation pattern. */
    protected static final Pattern _EMAIL_PATTERN = SendMailHelper.EMAIL_VALIDATION;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The manager for the JCR representation of forms */
    protected FormPropertiesManager _formPropertiesManager;
    
    /** The manager for the form entries */
    protected FormTableManager _formTableManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _formPropertiesManager = (FormPropertiesManager) serviceManager.lookup(FormPropertiesManager.ROLE);
        _formTableManager = (FormTableManager) serviceManager.lookup(FormTableManager.ROLE);
    }
    
    /**
     * Get information on the given entry of the given form
     * @param formId the id of the form
     * @param entryId the id of the entry
     * @return a map of information on the current user and on the email fields filled in the entry
     */
    @Callable
    public Map<String, Object> getMailInfo(String formId, Integer entryId)
    {
        Map<String, Object> sendMail = new HashMap<> ();
        
        sendMail.put("current-user", _getSender());
        
        UserEntry formEntry = _getFormEntry(formId, entryId);
        sendMail.put("emails", _getAvailableRecipients(formEntry));

        Map<String, Object> result = new HashMap<> ();
        result.put("send-mail", sendMail);
        return result;
    }
    
    /**
     * Get the sender filled into the dialog to send informations to claimant by email about his form entry status.
     * @return the sender
     */
    protected Map<String, Object> _getSender()
    {
        // Retrieve the current user name and email
        UserIdentity userIdentity = _currentUserProvider.getUser();
        User user = _userManager.getUser(userIdentity);
        
        Map<String, Object> currentUser = new HashMap<> ();
        currentUser.put("email", user.getEmail());
        currentUser.put("fullname", user.getFullName());
        
        return currentUser;
    }
    
    /**
     * Get the available recipients from the form entries to propose recipients.
     * @param formEntry The form entry
     * @return the list of available recipients
     */
    protected List<Map<String, String>> _getAvailableRecipients(UserEntry formEntry)
    {
        // Retrieve the valid email fields
        List<Map<String, String>> emails = new ArrayList<>();
        for (FieldValue fieldValue : formEntry.getValues())
        {
            Field field = fieldValue.getField();
            if (FieldType.TEXT.equals(field.getType()))
            {
                Map<String, String> properties = field.getProperties();
                String value = (String) fieldValue.getValue();
                if (value != null && "email".equals(properties.get("regexptype")) && _EMAIL_PATTERN.matcher(value).matches())
                {
                    Map<String, String> email = new HashMap<> ();
                    email.put("displayValue", field.getLabel() + ": " + value);
                    email.put("value", value);
                    emails.add(email);
                }
            }
        }
        
        return emails;
    }
    
    /**
     * Get the form entry from form identifier and entry identifier.
     * @param formId The form id
     * @param entryId The entry id
     * @return the form entry or throw a {@link IllegalArgumentException} if it doesn't exist.
     */
    protected UserEntry _getFormEntry(String formId, Integer entryId)
    {
        try
        {
            Form form = _formPropertiesManager.getForm(formId);
            Map<String, FieldValue> columns = _formTableManager.getColumns(form);
            
            return Optional.ofNullable(_formTableManager.getSubmission(form, columns, entryId))
                    .orElseThrow(() -> new IllegalArgumentException("Unable to retrieve the workflow id for the entry '" + entryId + "' of the form '" + formId + "'."));
        }
        catch (FormsException e)
        {
            throw new IllegalArgumentException("An error occured while trying to recover the user entry '" + entryId + "' of the form of id '" + formId + "'.");
        }
    }
}
