/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.plugins.forms.question.FormQuestionDataTypeExtensionPoint;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.model.type.ModelItemType;
import org.ametys.runtime.parameter.Validator;

/**
 * Helper to create element definition for form
 */
public class FormElementDefinitionHelper implements Serviceable
{
    /** The form question data type extension point */
    protected static FormQuestionDataTypeExtensionPoint _formQuestionDataTypeExtensionPoint;
    
    /** The logger */
    protected static Logger __logger = LoggerFactory.getLogger(FormElementDefinitionHelper.class);
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _formQuestionDataTypeExtensionPoint = (FormQuestionDataTypeExtensionPoint) manager.lookup(FormQuestionDataTypeExtensionPoint.ROLE);
    }
    
    /**
     * Create element definition
     * @param name the name
     * @param typeId the type id
     * @param labelKey the label i18n key. Can be null.
     * @param descKey the desc i18n key. Can be null.
     * @param validator the validator. Can be null.
     * @return the element definition
     */
    @SuppressWarnings("unchecked")
    public static ElementDefinition getElementDefinition(String name, String typeId, String labelKey, String descKey, Validator validator)
    {
        ModelItemType type = _formQuestionDataTypeExtensionPoint.getExtension(typeId);
        if (!(type instanceof ElementType))
        {
            throw new BadItemTypeException("The type '" + typeId + "' (used for data '" + name + "') can not be used for an element definition.");
        }
        else
        {
            ElementDefinition elementDefinition = new DefaultElementDefinition(name, false, (ElementType) type);
            if (StringUtils.isNotBlank(labelKey))
            {
                elementDefinition.setLabel(new I18nizableText("plugin.forms", labelKey)); 
            }

            if (StringUtils.isNotBlank(descKey))
            {
                elementDefinition.setDescription(new I18nizableText("plugin.forms", descKey)); 
            }
            
            if (validator != null)
            {
                elementDefinition.setValidator(validator);
            }
            
            return elementDefinition;
        }
    }
    
    /**
     * Create repeater definition 
     * @param datapath path of the parent element
     * @param childElements the repeater's children
     * @param labelKey the label i18n key. Can be null.
     * @param descKey the desc i18n key. Can be null.
     * @param addKey the add i18n key. Can be null.
     * @param deleteKey the delete i18n key. Can be null.
     * @param maxSize number maximum of entries
     * @return the created repeater
     */
    public static RepeaterDefinition getRepeaterDefinition(String datapath, ElementDefinition[] childElements, String labelKey, String descKey, String addKey, String deleteKey, Integer maxSize)
    {
        try
        {
            RepeaterDefinition repeater = RepeaterDefinition.of(datapath, FormQuestionDataTypeExtensionPoint.ROLE, childElements);
            repeater.setLabel(new I18nizableText("plugin.forms", labelKey));
            repeater.setDescription(new I18nizableText("plugin.forms", descKey));
            repeater.setAddLabel(new I18nizableText("plugin.forms", addKey));
            repeater.setDeleteLabel(new I18nizableText("plugin.forms", deleteKey));
            repeater.setMaxSize(maxSize);
            
            return repeater;
        }
        catch (Exception e)
        {
            __logger.error("An error occured while creating repeater definition", e);
        }
        return null;
    }
}
