/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.io.IOUtils;

import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;

/**
 * Extension point for questions types
 */
public class FormQuestionTypeExtensionPoint extends AbstractThreadSafeComponentExtensionPoint<FormQuestionType>
{
    /** The form question type extension point role */
    public static final String ROLE = FormQuestionTypeExtensionPoint.class.getName();
    
    private String _questions2htmlXSLT;
    
    /**
     * Get the content of the questions XSLT file
     * @return A valid XSLT file content
     */
    public String getQuestions2htmlXSLT()
    {
        return _questions2htmlXSLT;
    }
    
    @Override
    public void initializeExtensions() throws Exception
    {
        String template = _readTemplate();
        
        Set<String> questions2htmlXSLTs = _getDocbook2htmlXSLTs();
        String questions2htmlXSLTsIncludes = _createIncludes(questions2htmlXSLTs);
        _questions2htmlXSLT = template.replace("@INCLUDE@", questions2htmlXSLTsIncludes);
    }
    
    private String _readTemplate() throws IOException
    {
        try (InputStream is = this.getClass().getResourceAsStream("template.xsl"))
        {
            return IOUtils.toString(is, "UTF-8");
        }
    }
    
    private String _createIncludes(Set<String> includes)
    {
        StringBuffer sb = new StringBuffer("");
        
        for (String file : includes)
        {
            sb.append("<xsl:include href=\"");
            sb.append(file);
            sb.append("\"/>");
        }
        
        return sb.toString();
    }
    
    private Set<String> _getDocbook2htmlXSLTs()
    {
        Set<String> question2htmlXSLTs = new HashSet<>();

        for (String id : getExtensionsIds())
        {
            FormQuestionType formQuestionType = getExtension(id);
            question2htmlXSLTs.add(formQuestionType.getDisplayXSLT());
        }
        
        return question2htmlXSLTs;
    }
}
