/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.computing;

import java.util.ArrayList;
import java.util.List;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.forms.question.FormQuestionDataTypeExtensionPoint;
import org.ametys.plugins.forms.question.sources.ChoiceSourceTypeExtensionPoint;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Static class for creating {@link ComputingType} from xml configuration
 */
public abstract class AbstractStaticComputingType extends AbstractLogEnabled implements ComputingType, Serviceable, Configurable, PluginAware
{
    /** Forms */
    protected String _pluginName;
    /** Id of computing type */
    protected String _id;
    /** Label of computing type */
    protected I18nizableText _label;
    /** Description of computing type */
    protected I18nizableText _description;
    /** Xslt associated with computing type */
    protected String _xslt;
    
    /** The computing type extension point */
    protected ComputingTypeExtensionPoint _computingTypeExtensionPoint;
    /** The form question data type extension point */
    protected FormQuestionDataTypeExtensionPoint _formQuestionDataTypeExtensionPoint;
    /** The choice source type extension point */
    protected ChoiceSourceTypeExtensionPoint _choiceSourceTypeExtensionPoint;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _computingTypeExtensionPoint = (ComputingTypeExtensionPoint) manager.lookup(ComputingTypeExtensionPoint.ROLE);
        _formQuestionDataTypeExtensionPoint = (FormQuestionDataTypeExtensionPoint) manager.lookup(FormQuestionDataTypeExtensionPoint.ROLE);
        _choiceSourceTypeExtensionPoint = (ChoiceSourceTypeExtensionPoint) manager.lookup(ChoiceSourceTypeExtensionPoint.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }

    public void configure(Configuration configuration) throws ConfigurationException
    {
        _id = configuration.getAttribute("id");
        
        Configuration childLabel = configuration.getChild("label");
        _label = I18nizableText.getI18nizableTextValue(childLabel, "plugin." + _pluginName, childLabel.getValue());
        
        Configuration childDesc = configuration.getChild("description");
        _description = I18nizableText.getI18nizableTextValue(childDesc, "plugin." + _pluginName, childDesc.getValue());
        
        _xslt = configuration.getChild("xslt").getValue(null);
    }

    public String getId()
    {
        return _id;
    }

    public I18nizableText getLabel()
    {
        return _label;
    }
    
    public I18nizableText getDescription()
    {
        return _description;
    }

    public String getXSLT()
    {
        return _xslt;
    }
    
    public boolean hasComputedValue()
    {
        return true;
    }
    
    public List<String> getFieldToDisableIfFormPublished()
    {
        return new ArrayList<>();
    }
    
    public void saxAdditionalValue(ContentHandler contentHandler, FormQuestion question, FormEntry entry) throws SAXException
    {
        // Do nothing by default
    }
    
    public String getJSRenderer()
    {
        return null;
    }
}
