/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.util.DateUtils;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.question.validators.DateTimeIntervalFormValidator;
import org.ametys.plugins.forms.question.validators.LocalDateIntervalFormValidator;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;

/**
 * Class for creating date time questions
 */
public class DateTimeQuestionType extends AbstractFormQuestionType
{
    /** Constant for minimum Date attribute. */
    public static final String ATTRIBUTE_MIN_DATE = "min-date";

    /** Constant for maximum Date attribute. */
    public static final String ATTRIBUTE_MAX_DATE = "max-date";

    /** Constant for minimum Date attribute. */
    public static final String ATTRIBUTE_MIN_DATETIME = "min-datetime";

    /** Constant for maximum Date attribute. */
    public static final String ATTRIBUTE_MAX_DATETIME = "max-datetime";

    /** Constant for date format attribute. */
    public static final String ATTRIBUTE_DATE_FORMAT = "date-format";

    /** Name of datetimeStaticEnumerator entry that enable min and max datetime fields */
    public static final String DATE_TIME_FORMAT_VALUE = "datetime";

    /** Name of datetimeStaticEnumerator entry that enable min and max date fields */
    public static final String DATE_FORMAT_VALUE = "date";

    /** Constant for default title */
    public static final String DEFAULT_TITLE = "PLUGIN_FORMS_QUESTION_DEFAULT_TITLE_DATE";
    
    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        // DATETIME Conditions
        DisableConditions datetimeDisableConditions = new DisableConditions();
        DisableCondition conditionDatetime = new DisableCondition(ATTRIBUTE_DATE_FORMAT, OPERATOR.NEQ, DATE_TIME_FORMAT_VALUE);
        datetimeDisableConditions.getConditions().add(conditionDatetime);

        // MIN datetime
        ElementDefinition minDateTime = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MIN_DATETIME, ModelItemTypeConstants.DATETIME_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_MIN_DATE", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_MIN_DATE_DESC", null);
        minDateTime.setDisableConditions(datetimeDisableConditions);
        modelItems.add(minDateTime);

        // MAX datetime
        ElementDefinition maxDateTime = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MAX_DATETIME, ModelItemTypeConstants.DATETIME_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_MAX_DATE", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_MAX_DATE_DESC", null);
        maxDateTime.setDisableConditions(datetimeDisableConditions);
        modelItems.add(maxDateTime);

        // DATE Conditions
        DisableConditions dateDisableConditions = new DisableConditions();
        DisableCondition conditionDate = new DisableCondition(ATTRIBUTE_DATE_FORMAT, OPERATOR.NEQ, DATE_FORMAT_VALUE);
        dateDisableConditions.getConditions().add(conditionDate);

        // MIN date
        ElementDefinition minDate = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MIN_DATE, ModelItemTypeConstants.DATE_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_MIN_DATE", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_MIN_DATE_DESC", null);
        minDate.setDisableConditions(dateDisableConditions);
        modelItems.add(minDate);

        // MAX date
        ElementDefinition maxDate = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MAX_DATE, ModelItemTypeConstants.DATE_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_MAX_DATE", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_MAX_DATE_DESC", null);
        maxDate.setDisableConditions(dateDisableConditions);
        modelItems.add(maxDate);

        // FORMAT
        ElementDefinition<String> format = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_DATE_FORMAT, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_FORMAT", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_FORMAT_DESC", new DefaultValidator(null, true));
        StaticEnumerator<String> datetimeStaticEnumerator = new StaticEnumerator<>();
        datetimeStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_FORMAT_DATETIME"), DATE_TIME_FORMAT_VALUE);
        datetimeStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATETIME_FORMAT_DATE"), DATE_FORMAT_VALUE);
        format.setEnumerator(datetimeStaticEnumerator);
        format.setDefaultValue(DATE_FORMAT_VALUE);
        modelItems.add(format);
        
        return modelItems;
    }
    
    @Override
    protected SimpleViewItemGroup _getMainTab()
    {
        SimpleViewItemGroup mainFieldset = super._getMainTab();
        
        ViewElement format = new ViewElement();
        format.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_DATE_FORMAT));
        mainFieldset.addViewItem(format);
        
        return mainFieldset;
    }
    
    @Override
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = super._getAdvancedTab();
        
        ViewElement minDate = new ViewElement();
        minDate.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_MIN_DATE));
        advancedFieldset.addViewItem(minDate);

        ViewElement maxDate = new ViewElement();
        maxDate.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_MAX_DATE));
        advancedFieldset.addViewItem(maxDate);

        ViewElement minDateTime = new ViewElement();
        minDateTime.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_MIN_DATETIME));
        advancedFieldset.addViewItem(minDateTime);

        ViewElement maxDateTime = new ViewElement();
        maxDateTime.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_MAX_DATETIME));
        advancedFieldset.addViewItem(maxDateTime);
        
        return advancedFieldset;
    }
    
    public String getStorageType(FormQuestion question)
    {
        String type = question.getValue(ATTRIBUTE_DATE_FORMAT);
        return DATE_TIME_FORMAT_VALUE.equals(type) ? ModelItemTypeConstants.DATETIME_TYPE_ID : ModelItemTypeConstants.DATE_TYPE_ID;
    }
    
    @Override
    protected ModelItem _getEntryModelItem(FormQuestion question)
    {
        ModelItem item = super._getEntryModelItem(question);
        if (DATE_TIME_FORMAT_VALUE.equals(question.getValue(ATTRIBUTE_DATE_FORMAT)))
        {
            ZonedDateTime minDate = question.getValue(ATTRIBUTE_MIN_DATETIME);
            ZonedDateTime maxDate = question.getValue(ATTRIBUTE_MAX_DATETIME);
            ((ElementDefinition) item).setValidator(new DateTimeIntervalFormValidator(null, isMandatory(question), minDate, maxDate));
        }
        else
        {
            LocalDate minDate = question.getValue(ATTRIBUTE_MIN_DATE);
            LocalDate maxDate = question.getValue(ATTRIBUTE_MAX_DATE);
            ((ElementDefinition) item).setValidator(new LocalDateIntervalFormValidator(null, isMandatory(question), minDate, maxDate));
        }
        return item;
    }
    
    @Override
    public void validateQuestionValues(Map<String, Object> values, Map<String, I18nizableText> errors)
    {
        super.validateQuestionValues(values, errors);

        String format = (String) values.get(ATTRIBUTE_DATE_FORMAT);
        switch (format)
        {
            case DATE_FORMAT_VALUE:
                String minDateAsString = (String) values.getOrDefault(ATTRIBUTE_MIN_DATE, null);
                String maxDateAsString = (String) values.getOrDefault(ATTRIBUTE_MAX_DATE, null);
                if (StringUtils.isNotBlank(minDateAsString) && StringUtils.isNotBlank(maxDateAsString))
                {
                    LocalDate minDate = LocalDate.parse(minDateAsString);
                    LocalDate maxDate = LocalDate.parse(maxDateAsString);
                    
                    if (minDate != null && maxDate != null && minDate.isAfter(maxDate))
                    {
                        errors.put(ATTRIBUTE_MIN_DATE, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATE_GAP_ERROR"));
                        errors.put(ATTRIBUTE_MAX_DATE, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATE_GAP_ERROR"));
                    }
                }
                break; 
            case DATE_TIME_FORMAT_VALUE:
                String minDateTimeAsString = (String) values.getOrDefault(ATTRIBUTE_MIN_DATETIME, null);
                String maxDateTimeAsString = (String) values.getOrDefault(ATTRIBUTE_MAX_DATETIME, null);
                if (StringUtils.isNotBlank(minDateTimeAsString) && StringUtils.isNotBlank(maxDateTimeAsString))
                {
                    ZonedDateTime minDateTime = DateUtils.parseZonedDateTime(minDateTimeAsString);
                    ZonedDateTime maxDateTime = DateUtils.parseZonedDateTime(maxDateTimeAsString);
                    if (minDateTime.isAfter(maxDateTime))
                    {
                        errors.put(ATTRIBUTE_MIN_DATETIME, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATE_GAP_ERROR"));
                        errors.put(ATTRIBUTE_MAX_DATETIME, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_DATE_GAP_ERROR"));
                    }
                }
                break;
            default:
                getLogger().error("Date format " + format + " is not supported");
                break;
        }
    }
    
    public I18nizableText getDefaultTitle()
    {
        return new I18nizableText("plugin.forms", DEFAULT_TITLE);
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(ATTRIBUTE_DATE_FORMAT);
        return fieldNames;
    }
}
