/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import java.util.List;
import java.util.Map;

import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.question.validators.NumberIntervalFormValidator;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Class for creating number question type
 */
public class NumberQuestionType extends AbstractFormQuestionType
{
    /** Constant for min attribute */
    public static final String ATTRIBUTE_MIN_DOUBLE = "min-double";
    
    /** Constant for max attribute */
    public static final String ATTRIBUTE_MAX_DOUBLE = "max-double";
    
    /** Constant for min attribute */
    public static final String ATTRIBUTE_MIN_INT = "min-int";
    
    /** Constant for max attribute */
    public static final String ATTRIBUTE_MAX_INT = "max-int";
    
    /** Constant for number type attribute */
    public static final String ATTRIBUTE_NUMBER_TYPE = "number-type";
    
    /** Name of double numberTypeStaticEnumerator entry  */
    public static final String DOUBLE_NUMBER_VALUE = "double";
    
    /** Name of integer numberTypeStaticEnumerator entry  */
    public static final String INTEGER_NUMBER_VALUE = "int";
    
    /** Constant for default title */
    public static final String DEFAULT_TITLE = "PLUGIN_FORMS_QUESTION_DEFAULT_TITLE_NUMBER";
    
    @Override
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = super._getAdvancedTab();
        
        ViewElement numberType = new ViewElement();
        numberType.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_NUMBER_TYPE));
        advancedFieldset.addViewItem(numberType);
        
        ViewElement min = new ViewElement();
        min.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_MIN_DOUBLE));
        advancedFieldset.addViewItem(min);
        
        ViewElement max = new ViewElement();
        max.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_MAX_DOUBLE));
        advancedFieldset.addViewItem(max);
        
        ViewElement minInt = new ViewElement();
        minInt.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_MIN_INT));
        advancedFieldset.addViewItem(minInt);
        
        ViewElement maxInt = new ViewElement();
        maxInt.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_MAX_INT));
        advancedFieldset.addViewItem(maxInt);
        
        return advancedFieldset;
    }

    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        // DOUBLE Conditions
        DisableConditions doubleDisableConditions = new DisableConditions();
        DisableCondition conditionDouble = new DisableCondition(ATTRIBUTE_NUMBER_TYPE, OPERATOR.NEQ, DOUBLE_NUMBER_VALUE);
        doubleDisableConditions.getConditions().add(conditionDouble);
        //MIN
        ElementDefinition minDouble = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MIN_DOUBLE, ModelItemTypeConstants.DOUBLE_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_MIN", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_MIN_DESC", null);
        minDouble.setDisableConditions(doubleDisableConditions);
        modelItems.add(minDouble);
        
        //MAX
        ElementDefinition maxDouble = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MAX_DOUBLE, ModelItemTypeConstants.DOUBLE_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_MAX", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_MAX_DESC", null);
        maxDouble.setDisableConditions(doubleDisableConditions);
        modelItems.add(maxDouble);
        
        // INTEGER Conditions
        DisableConditions integerDisableConditions = new DisableConditions();
        DisableCondition conditionInteger = new DisableCondition(ATTRIBUTE_NUMBER_TYPE, OPERATOR.NEQ, INTEGER_NUMBER_VALUE);
        integerDisableConditions.getConditions().add(conditionInteger);
        //MIN
        ElementDefinition minInt = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MIN_INT, ModelItemTypeConstants.LONG_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_MIN", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_MIN_DESC", null);
        minInt.setDisableConditions(integerDisableConditions);
        modelItems.add(minInt);
        
        //MAX
        ElementDefinition maxInt = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_MAX_INT, ModelItemTypeConstants.LONG_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_MAX", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_MAX_DESC", null);
        maxInt.setDisableConditions(integerDisableConditions);
        modelItems.add(maxInt);
        
        //NUMBER_TYPE
        ElementDefinition<String> numberType = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_NUMBER_TYPE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_NUMBER_TYPE", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_NUMBER_TYPE_DESC", null);

        StaticEnumerator<String> numberTypeStaticEnumerator = new StaticEnumerator<>();
        numberTypeStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_TYPE_DOUBLE"), DOUBLE_NUMBER_VALUE);
        numberTypeStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_TYPE_INT"), INTEGER_NUMBER_VALUE);
        numberType.setEnumerator(numberTypeStaticEnumerator);
        numberType.setDefaultValue(INTEGER_NUMBER_VALUE);
        modelItems.add(numberType);
        
        return modelItems;
    }
    
    public String getStorageType(FormQuestion question)
    {
        String type = question.getValue(ATTRIBUTE_NUMBER_TYPE);
        return type.equals(DOUBLE_NUMBER_VALUE) ? ModelItemTypeConstants.DOUBLE_TYPE_ID : ModelItemTypeConstants.LONG_TYPE_ID;
    }
    
    @Override
    protected ModelItem _getEntryModelItem(FormQuestion question)
    {
        ModelItem item = super._getEntryModelItem(question);
        if (DOUBLE_NUMBER_VALUE.equals(question.getValue(ATTRIBUTE_NUMBER_TYPE)))
        {
            ((ElementDefinition) item).setValidator(new NumberIntervalFormValidator(null, isMandatory(question), question.getValue(ATTRIBUTE_MIN_DOUBLE), question.getValue(ATTRIBUTE_MAX_DOUBLE)));
        }
        else
        {
            ((ElementDefinition) item).setValidator(new NumberIntervalFormValidator(null, isMandatory(question), question.getValue(ATTRIBUTE_MIN_INT), question.getValue(ATTRIBUTE_MAX_INT)));
        }
        return item;
    }
    
    @Override
    public void validateQuestionValues(Map<String, Object> values, Map<String, I18nizableText> errors)
    {
        super.validateQuestionValues(values, errors);
        
        String format = (String) values.get(ATTRIBUTE_NUMBER_TYPE);
        switch (format)
        {
            case DOUBLE_NUMBER_VALUE:
                Number minDouble = (Number) values.getOrDefault(ATTRIBUTE_MIN_DOUBLE, null);
                Number maxDouble = (Number) values.getOrDefault(ATTRIBUTE_MAX_DOUBLE, null);
                if (minDouble != null && maxDouble != null && minDouble.doubleValue() > maxDouble.doubleValue())
                {
                    errors.put(ATTRIBUTE_MIN_DOUBLE, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_GAP_ERROR"));
                    errors.put(ATTRIBUTE_MAX_DOUBLE, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_GAP_ERROR"));
                }
                break;
            
            case INTEGER_NUMBER_VALUE:
                Number min = (Number) values.getOrDefault(ATTRIBUTE_MIN_INT, null);
                Number max = (Number) values.getOrDefault(ATTRIBUTE_MAX_INT, null);
                if (min != null && max != null && min.longValue() > max.longValue())
                {
                    errors.put(ATTRIBUTE_MIN_INT, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_GAP_ERROR"));
                    errors.put(ATTRIBUTE_MAX_INT, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_NUMBER_GAP_ERROR"));
                }
                break;
            
            default:
                getLogger().error("Number format " + format + " is not supported");
                break;
        }
    }
    
    public I18nizableText getDefaultTitle()
    {
        return new I18nizableText("plugin.forms", DEFAULT_TITLE);
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(ATTRIBUTE_NUMBER_TYPE);
        return fieldNames;
    }
}
