/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;

/** 
 * Class for creating simple text questions 
 */
public class SimpleTextQuestionType extends AbstractFormQuestionType implements AutocompleteAwareQuestionType
{
    /** Constant for regexp attribute. */
    public static final String ATTRIBUTE_REGEXP = "regexp";
    
    /** Constant for custom regexp attribute. */
    public static final String ATTRIBUTE_CUSTOM_REGEX = "custom-regex";
    
    /** Constant for autofill attribute. */
    public static final String ATTRIBUTE_AUTOFILL = "autofill";
    
    /** Constant for default value attribute. */
    public static final String ATTRIBUTE_DEFAULT_VALUE = "default-value";
    
    /** Name of empty regexStaticEnumerator entry  */
    public static final String EMPTY_REGEX_VALUE = " ";
    
    /** Name of email regexStaticEnumerator entry  */
    public static final String EMAIL_REGEX_VALUE = "email";
    
    /** Name of phone regexStaticEnumerator entry  */
    public static final String PHONE_REGEX_VALUE = "phone";
    
    /** Name of regexStaticEnumerator entry that enable custom regex field */
    public static final String CUSTOM_REGEX_VALUE = "custom";
    
    /** Name of empty autofillStaticEnumerator entry */
    public static final String EMPTY_AUTOFILL_VALUE = " ";
    
    /** Name of email autofillStaticEnumerator entry */
    public static final String EMAIL_AUTOFILL_VALUE = "email";
    
    /** Name of id autofillStaticEnumerator entry */
    public static final String ID_AUTOFILL_VALUE = "id";
    
    /** Name of fullname autofillStaticEnumerator entry */
    public static final String FULLNAME_AUTOFILL_VALUE = "fullName";
    
    /** Name of firstName autofillStaticEnumerator entry */
    public static final String FIRSTNAME_AUTOFILL_VALUE = "firstName";
    
    /** Name of lastName autofillStaticEnumerator entry */
    public static final String LASTNAME_AUTOFILL_VALUE = "lastName";
    
    /** Name of autofillStaticEnumerator entry that enable default value field */
    public static final String CUSTOM_AUTOFILL_VALUE = "custom";
    
    /** Constant for placeholder attribute. */
    public static final String ATTRIBUTE_PLACEHOLDER = "placeholder";

    /** Constant for default title */
    public static final String DEFAULT_TITLE = "PLUGIN_FORMS_QUESTION_DEFAULT_TITLE_SIMPLE_TEXT";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The users manager */
    protected UserManager _userManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
    }
    
    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        //PLACEHOLDER
        ElementDefinition placeholder = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_PLACEHOLDER, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_PLACEHOLDER", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_PLACEHOLDER_DESC", null);
        modelItems.add(placeholder);
        
        //AUTOFILL
        ElementDefinition<String> autofill = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_AUTOFILL, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_DESC", null);

        StaticEnumerator<String> autofillStaticEnumerator = new StaticEnumerator<>();
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_EMPTY_TEXT"), EMPTY_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_EMAIL"), EMAIL_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_ID"), ID_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_FULLNAME"), FULLNAME_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_FIRSTNAME"), FIRSTNAME_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_LASTNAME"), LASTNAME_AUTOFILL_VALUE);
        autofillStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_AUTOFILL_CUSTOM"), CUSTOM_AUTOFILL_VALUE);
        autofill.setEnumerator(autofillStaticEnumerator);
        Map<String, I18nizableText> widgetParameters  = new HashMap<>();
        widgetParameters.put("naturalOrder", new I18nizableText("true"));
        autofill.setWidgetParameters(widgetParameters);
        autofill.setDefaultValue(EMPTY_AUTOFILL_VALUE);
        
        modelItems.add(autofill);
        
        //DEFAULT VALUE
        ElementDefinition defaultValue = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_DEFAULT_VALUE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_DEFAULT_VALUE", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_DEFAULT_VALUE_DESC", null);
        DisableConditions disableConditions = new DisableConditions();
        DisableCondition condition = new DisableCondition(ATTRIBUTE_AUTOFILL, OPERATOR.NEQ, CUSTOM_AUTOFILL_VALUE); 
        disableConditions.getConditions().add(condition);
        defaultValue.setDisableConditions(disableConditions);
        modelItems.add(defaultValue);
        
        //REGEX
        ElementDefinition<String> regexp = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_REGEXP, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_DESC", null);
        
        StaticEnumerator<String> regexStaticEnumerator = new StaticEnumerator<>();
        regexStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_EMPTY_TEXT"), EMPTY_REGEX_VALUE);
        regexStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_EMAIL"), EMAIL_REGEX_VALUE);
        regexStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_PHONE"), PHONE_REGEX_VALUE);
        regexStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_CUSTOM"), CUSTOM_REGEX_VALUE);
        regexp.setEnumerator(regexStaticEnumerator);
        Map<String, I18nizableText> regexpWidgetParameters  = new HashMap<>();
        regexpWidgetParameters.put("naturalOrder", new I18nizableText("true"));
        regexp.setWidgetParameters(regexpWidgetParameters);
        regexp.setDefaultValue(EMPTY_REGEX_VALUE);
        modelItems.add(regexp);
        
        //CUSTOM REGEX
        ElementDefinition customRegex = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_CUSTOM_REGEX, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_CUSTOM", "PLUGINS_FORMS_QUESTIONS_DIALOG_SIMPLE_TEXT_REGEXTYPE_CUSTOM_DESC", new DefaultValidator("^\\/.+\\/[gi]*$", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_REGEXP_ERROR_DESCRIPTION"), false));
        DisableConditions regexDisableConditions = new DisableConditions();
        DisableCondition regexCondition = new DisableCondition(ATTRIBUTE_REGEXP, OPERATOR.NEQ, CUSTOM_REGEX_VALUE); 
        regexDisableConditions.getConditions().add(regexCondition);
        customRegex.setDisableConditions(regexDisableConditions);
        modelItems.add(customRegex);
        
        //AUTOCOMPLETE
        modelItems.add(getAutocompleteModelItem(InputType.TEXT));
        
        return modelItems;
    }
    
    @Override
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = super._getAdvancedTab();
        
        ViewElement placeholder = new ViewElement();
        placeholder.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_PLACEHOLDER));
        advancedFieldset.addViewItem(placeholder);
        
        ViewElement autofill = new ViewElement();
        autofill.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_AUTOFILL));
        advancedFieldset.addViewItem(autofill);
        
        ViewElement defaultValue = new ViewElement();
        defaultValue.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_DEFAULT_VALUE));
        advancedFieldset.addViewItem(defaultValue);
        
        ViewElement regexp = new ViewElement();
        regexp.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_REGEXP));
        advancedFieldset.addViewItem(regexp);
        
        ViewElement customRegex = new ViewElement();
        customRegex.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_CUSTOM_REGEX));
        advancedFieldset.addViewItem(customRegex);
        
        advancedFieldset.addViewItem(getAutocompleteViewElement(getModel()));
        
        return advancedFieldset;
    }
    
    public String getStorageType(FormQuestion question)
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
    
    @Override
    protected ModelItem _getEntryModelItem(FormQuestion question)
    {
        ModelItem item = super._getEntryModelItem(question);
        ((ElementDefinition) item).setValidator(new DefaultValidator(_getEntryRegExpPattern(question), isMandatory(question)));
        return item;
    }

    @Override
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException
    {
        super.saxAdditionalInfos(contentHandler, question);
        
        if (StringUtils.isEmpty(question.getValue(ATTRIBUTE_DEFAULT_VALUE)))
        {
            UserIdentity userIdentity = _currentUserProvider.getUser();
            User user = userIdentity != null ? _userManager.getUser(userIdentity) : null;
            String autofill = user != null ? _getAutofillValue(question, user) : null;
            if (StringUtils.isNotBlank(autofill))
            {
                XMLUtils.createElement(contentHandler, "default-value", autofill);
            }
        }
        else
        {
            XMLUtils.createElement(contentHandler, "default-value", question.<String>getValue(ATTRIBUTE_DEFAULT_VALUE));
        }
        
        String pattern = _getRegExpPattern(question);
        if (StringUtils.isNotEmpty(pattern))
        {
            XMLUtils.createElement(contentHandler, "pattern", pattern);
        }
    }
    
    /**
     * Get autofill value
     * @param question the question
     * @param user the user
     * @return the autofill value.<code>null</code> if there is not autofill value
     */
    protected String _getAutofillValue(FormQuestion question, User user) 
    {
        String autofill = question.getValue(ATTRIBUTE_AUTOFILL);
        if (autofill != null)
        {
            switch (autofill)
            {
                case "email":
                    return user.getEmail();
                case "id":
                    return user.getIdentity().getLogin(); 
                case "fullName":
                    return user.getFullName();
                case "firstName":
                    return user.getFirstName();
                case "lastName":
                    return user.getLastName();
                default:
                    return null;
            }
        }
        return null;
    }
    
    /**
     * Get the validation pattern.
     * @param question the question
     * @return the validation pattern.
     */
    protected String _getRegExpPattern (FormQuestion question)
    {
        String regexpType = question.getValue(ATTRIBUTE_REGEXP);
        if (regexpType == null)
        {
            return null;
        }
        else if ("email".equals(regexpType))
        {
            return "/^([a-zA-Z0-9_\\.\\-\\+])+\\@(([a-zA-Z0-9\\-])+\\.)+([a-zA-Z0-9]{2,4})+$/";
        }
        else if ("phone".equals(regexpType))
        {
            return "/^(\\+?\\(?[0-9]{1,3}\\)?([\\s]?)(\\(0\\))?|0)([\\s]?)([0-9\\-\\+\\s]{4,})+$/";
        }
        else if ("custom".equals(regexpType) && !("".equals(question.getValue(ATTRIBUTE_CUSTOM_REGEX)))) 
        {
            return question.getValue(ATTRIBUTE_CUSTOM_REGEX);
        }
        
        return null;
    }
    
    /**
     * Get the validation pattern for entries.
     * @param question the question
     * @return the validation pattern.
     */
    protected String _getEntryRegExpPattern (FormQuestion question)
    {
        String regexpType = question.getValue(ATTRIBUTE_REGEXP);
        if (regexpType == null)
        {
            return null;
        }
        else if ("email".equals(regexpType))
        {
            return "^([a-zA-Z0-9_\\.\\-\\+])+\\@(([a-zA-Z0-9\\-])+\\.)+([a-zA-Z0-9]{2,4})+$";
        }
        else if ("phone".equals(regexpType))
        {
            return "^(\\+?\\(?[0-9]{1,3}\\)?([\\s]?)(\\(0\\))?|0)([\\s]?)([0-9\\-\\+\\s]{4,})+$";
        }
        else if ("custom".equals(regexpType) && !("".equals(question.getValue(ATTRIBUTE_CUSTOM_REGEX)))) 
        {
            String custom = question.getValue(ATTRIBUTE_CUSTOM_REGEX);
            return custom.substring(1, custom.length() - 1);
        }
        
        return null;
    }
    
    @Override
    public void validateQuestionValues(Map<String, Object> values, Map<String, I18nizableText> errors)
    {
        super.validateQuestionValues(values, errors);
        
        if (StringUtils.isNotBlank((String) values.get(ATTRIBUTE_CUSTOM_REGEX)) 
                && !Pattern.matches("^\\/.+\\/[gi]*$", (CharSequence) values.get(ATTRIBUTE_CUSTOM_REGEX)))
        {
            errors.put(ATTRIBUTE_CUSTOM_REGEX, new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_REGEXP_ERROR_DESCRIPTION"));
        }
    }
    
    public I18nizableText getDefaultTitle()
    {
        return new I18nizableText("plugin.forms", DEFAULT_TITLE);
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(ATTRIBUTE_REGEXP);
        fieldNames.add(ATTRIBUTE_CUSTOM_REGEX);
        return fieldNames;
    }

    @Override
    public boolean isCacheable(FormQuestion question)
    {
        return StringUtils.isBlank(question.getValue(ATTRIBUTE_AUTOFILL)) || question.getValue(ATTRIBUTE_AUTOFILL) == CUSTOM_AUTOFILL_VALUE;
    }

}
