/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.FormAndDirectoryCommonMethods;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.types.ChoicesListQuestionType;
import org.ametys.plugins.forms.repository.type.Rule;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CopiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.MovableAmetysObject;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.site.Site;

/**
 * Class representing a form, backed by a JCR node.<br>
 */
public class Form extends DefaultTraversableAmetysObject<FormFactory> implements ModifiableModelAwareDataAwareAmetysObject, MovableAmetysObject, SiteAwareAmetysObject, CopiableAmetysObject
{
    /** Property name for form displayed name */
    public static final String TITLE = "title";
    /** Property name for form description */
    public static final String DESCRIPTION = "description";
    /** Property name for form author */
    public static final String AUTHOR = "author";
    /** Property name for form last contributor */
    public static final String CONTRIBUTOR = "contributor";
    /** Property name for form creation date */
    public static final String CREATIONDATE = "creationDate";
    /** Property name for form last modification date */
    public static final String LASTMODIFICATIONDATE = "lastModificationDate";
    /** Property name for form workflow name */
    public static final String WORKFLOWNAME = "workflowName";
    /** Property name for form to limit one entry by user */
    public static final String LIMIT_TO_ONE_ENTRY_BY_USER = "limit-to-one-entry-by-user";
    /** Property name for form to limit the entries */
    public static final String LIMIT_ENTRIES_ENABLED = "limit-entries-enabled";
    /** Property name for form to enable queue */
    public static final String QUEUE_ENABLED = "queue-enabled";
    /** Property name for form's queue size */
    public static final String QUEUE_SIZE = "queue-size";
    /** Property name for form displayed message when closed because of limited number of entries and a queue is enabled */
    public static final String QUEUE_CLOSED_MSG = "queue-closed-message";
    /** Property name for form's authorized max number of entries */
    public static final String MAX_ENTRIES = "max-entries";
    /** Property name for form displayed message when entries are limited in number  */
    public static final String REMAINING_MSG = "remaining-message";
    /** Property name for form displayed message when closed because of limited number of entries */
    public static final String CLOSED_MSG = "closed-message";
    /** Property name for form entries admin emails */
    public static final String ADMIN_EMAILS = "admin-emails";
    /** Property name for form entries admin emails alternative source*/
    public static final String ADMIN_EMAILS_OTHER = "admin-emails-other";
    /** Property name for form entries admin email subject */
    public static final String ADMIN_EMAIL_SUBJECT = "admin-email-subject";
    /** Property name for form entries admin email body */
    public static final String ADMIN_EMAIL_BODY = "admin-emails-body";
    /** Property name for form acknowledgement of receipt's sender */
    public static final String RECEIPT_SENDER = "receipt-sender";
    /** Property name for form acknowledgement of receipt's receiver*/
    public static final String RECEIPT_RECEIVER = "receipt-receiver";
    /** Property name for form acknowledgement of receipt's subject */
    public static final String RECEIPT_SUBJECT = "receipt-subject";
    /** Property name for form acknowledgement of receipt's body */
    public static final String RECEIPT_BODY = "receipt-body";
    /** Property name for form acknowledgement of queue's sender */
    public static final String QUEUE_SENDER = "queue-sender";
    /** Property name for form acknowledgement of queue's receiver*/
    public static final String QUEUE_RECEIVER = "queue-receiver";
    /** Property name for form acknowledgement of queue's subject */
    public static final String QUEUE_SUBJECT = "queue-subject";
    /** Property name for form acknowledgement of queue's body */
    public static final String QUEUE_BODY = "queue-body";
    /** Property name for form start date */
    public static final String START_DATE = "startDate";
    /** Property name for form end date */
    public static final String END_DATE = "endDate";
    
    /**
     * Creates an {@link Form}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public Form(Node node, String parentPath, FormFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Set the title of this form.
     * @param title the short title
     */
    public void setTitle (String title)
    {
        this.setValue(TITLE, title);
    }
    
    /**
     * Set the description of this form.
     * @param description the description
     */
    public void setDescription (String description)
    {
        this.setValue(DESCRIPTION, description);
    }
    
    /**
     * Set the author of this form.
     * @param author the author
     */
    public void setAuthor(UserIdentity author)
    {
        this.setValue(AUTHOR, author);
    }

    /**
     * Set the last contributor of this form.
     * @param contributor the last contributor
     */
    public void setContributor(UserIdentity contributor)
    {
        this.setValue(CONTRIBUTOR, contributor);
    }
    
    /**
     * Set the date of the creation.
     * @param creationDate the last modification date to set.
     */
    public void setCreationDate(ZonedDateTime creationDate)
    {
        this.setValue(CREATIONDATE, creationDate);
    }
    
    /**
     * Set the date of the last modification.
     * @param lastModificationDate the last modification date to set.
     */
    public void setLastModificationDate(ZonedDateTime lastModificationDate)
    {
        this.setValue(LASTMODIFICATIONDATE, lastModificationDate);
    }
    
    /**
     * Get the title of the form
     * @return The title
     */
    public String getTitle()
    {        
        return this.getValue(TITLE);
    }
    
    /**
     * Get the description of the form
     * @return The description
     */
    public String getDescription()
    {
        return this.getValue(DESCRIPTION);
    }
    
    /**
     * Get the author of the form
     * @return The author
     */
    public UserIdentity getAuthor()
    {
        return this.getValue(AUTHOR);
    }
    
    /**
     * Get the last contributor of the form
     * @return The contributor
     */
    public UserIdentity getContributor()
    {
        return this.getValue(CONTRIBUTOR);
    }
    
    /**
     * Get the date of the last modification of the form
     * @return The date
     */
    public ZonedDateTime getCreationDate()
    {
        return this.getValue(CREATIONDATE);
    }
    
    /**
     * Get the date of the last modification of the form
     * @return The date
     */
    public ZonedDateTime getLastModificationDate()
    {
        return this.getValue(LASTMODIFICATIONDATE);
    }
    
    /**
     * Get the name of the workflow applied to the form
     * @return The workflow name
     */
    public String getWorkflowName()
    {
        return this.getValue(WORKFLOWNAME);
    }
    
    /**
     * Set the name of the workflow.
     * @param workflowName the name of the workflow to set.
     */
    public void setWorkflowName(String workflowName)
    {
        this.setValue(WORKFLOWNAME, workflowName);
    }
    
    /**
     * Get the receipt's sender of this form.
     * @return The receipt's sender
     */
    public Optional<String> getReceiptSender()
    {
        return Optional.ofNullable(getValue(RECEIPT_SENDER));
    }
    
    /**
     * Set the receipt's sender of this form.
     * @param sender The receipt's sender
     */
    public void setReceiptSender(String sender)
    {
        this.setValue(RECEIPT_SENDER, sender);
    }
    /**
     * Get the receipt's receiver of this form.
     * @return The receipt's receiver
     */
    public Optional<String> getReceiptReceiver()
    {
        return Optional.ofNullable(getValue(RECEIPT_RECEIVER));
    }
    
    /**
     * Set the receipt's receiver of this form.
     * @param receiver The receipt's receiver
     */
    public void setReceiptReceiver(String receiver)
    {
        this.setValue(RECEIPT_RECEIVER, receiver);
    }
    /**
     * Get the receipt's subject of this form.
     * @return The receipt's subject
     */
    public Optional<String> getReceiptSubject()
    {
        return Optional.ofNullable(getValue(RECEIPT_SUBJECT));
    }
    
    /**
     * Set the receipt's subject of this form.
     * @param subject The receipt's subject
     */
    public void setReceiptSubject(String subject)
    {
        this.setValue(RECEIPT_SUBJECT, subject);
    }
    /**
     * Get the receipt's body of this form.
     * @return The receipt's body
     */
    public Optional<String> getReceiptBody()
    {
        return Optional.ofNullable(getValue(RECEIPT_BODY));
    }
    
    /**
     * Set the receipt's body of this form.
     * @param body The receipt's body
     */
    public void setReceiptBody(String body)
    {
        this.setValue(RECEIPT_BODY, body);
    }
    
    /**
     * Indicate is this form can be considered as a mini survey
     * @return true if the form as only one question of type list and is limited to one answer by user
     */
    public boolean isMiniSurvey()
    {
        // do the cheap test before any expensive computation
        // Mini survey is restricted to only one entry by user
        if (!this.isLimitedToOneEntryByUser())
        {
            return false;
        }
        else
        {
            // There must be exactly one question that is not display only. And this question must be a choice list
            boolean hasAQuestion = false;
            for (FormQuestion question : this.getQuestions())
            {
                FormQuestionType type = question.getType();
                if (!type.onlyForDisplay(question))
                {
                    // first ChoicesListQuestion
                    if (type instanceof ChoicesListQuestionType && !hasAQuestion)
                    {
                        hasAQuestion = true;
                    }
                    // not a ChoicesListQuestion or second one
                    else
                    {
                        return false;
                    }
                }
            }
            return hasAQuestion;
        }
    }
    
    /**
     * Get if this form is limited to one entry by user
     * @return <code>true</code> if the form is limited to one entry by user
     */
    public boolean isLimitedToOneEntryByUser()
    {
        return this.getValue(LIMIT_TO_ONE_ENTRY_BY_USER, false, false);
    }
    
    /**
     * Limit or not to one entry by user
     * @param limit <code>true</code> to limit to one entry by user
     */
    public void limitToOneEntryByUser(boolean limit)
    {
        this.setValue(LIMIT_TO_ONE_ENTRY_BY_USER, limit);
    }
    
    /**
     * Get if the form entries are limited
     * @return <code>true</code> if the form entries are limited
     */
    public boolean isEntriesLimited()
    {
        return this.getValue(LIMIT_ENTRIES_ENABLED, false, false); 
    }
    
    /**
     * Limit or not the entries of the form
     * @param limitEntries <code>true</code> to limit the entries
     */
    public void limitEntries(boolean limitEntries)
    {
        this.setValue(LIMIT_ENTRIES_ENABLED, limitEntries); 
    }
    
    /**
     * Get if a queue is enabled
     * @return <code>true</code> if a queue is enabled
     */
    public boolean isQueueEnabled()
    {
        return isEntriesLimited() && this.getValue(QUEUE_ENABLED, false, false); 
    }
    
    /**
     * Enable of not the queue
     * @param enabled <code>true</code> to enable the queue
     */
    public void enableQueue(boolean enabled)
    {
        this.setValue(QUEUE_ENABLED, enabled); 
    }

    /**
     * Get the max number limit for entries in this form
     * @return the max number
     */
    public Optional<Long> getMaxEntries()
    {
        return Optional.ofNullable(getValue(MAX_ENTRIES));
    }
    
    /**
     * Set the max number limit for entries in this form
     * @param max The max number
     */
    public void setMaxEntries(Long max)
    {
        this.setValue(MAX_ENTRIES, max);
    }
    
    /**
     * Get the message that will be displayed to users when filling the form if a max entries limit is set
     * @return a message for users
     */
    public Optional<String> getRemainingMessage()
    {
        return Optional.ofNullable(getValue(REMAINING_MSG));
    }
    
    /**
     * Set the message that will be displayed to users when filling the form if a max entries limit is set
     * @param msg The message for users
     */
    public void setRemainingMessage(String msg)
    {
        this.setValue(REMAINING_MSG, msg);
    }
    
    /**
     * Get the message that will be displayed to users when the entries limit is reached
     * @return a message for users
     */
    public Optional<String> getClosedMessage()
    {
        return Optional.ofNullable(getValue(CLOSED_MSG));
    }
    
    /**
     * Set the message that will be displayed to users when the entries limit is reached
     * @param msg The message for users
     */
    public void setClosedMessage(String msg)
    {
        this.setValue(CLOSED_MSG, msg);
    }
    
    /**
     * Get the queue max size in this form
     * @return the max number
     */
    public Optional<Long> getQueueSize()
    {
        return Optional.ofNullable(getValue(QUEUE_SIZE));
    }
    
    /**
     * Set the queue max size in this form
     * @param max The max number
     */
    public void setQueueSize(Long max)
    {
        this.setValue(QUEUE_SIZE, max);
    }
    
    /**
     * Get the message that will be displayed to users when the entries limit is reached
     * and a queue is enabled
     * @return a message for users
     */
    public Optional<String> getClosedQueueMessage()
    {
        return Optional.ofNullable(getValue(QUEUE_CLOSED_MSG));
    }
    
    /**
     * Set the message that will be displayed to users when the entries limit is reached
     * and a queue is enabled
     * @param msg The message for users
     */
    public void setClosedQueueMessage(String msg)
    {
        this.setValue(QUEUE_CLOSED_MSG, msg);
    }
    
    /**
     * Get the email's sender for notifying exit of queue.
     * @return The email's sender
     */
    public Optional<String> getQueueMailSender()
    {
        return Optional.ofNullable(getValue(QUEUE_SENDER));
    }
    
    /**
     * Set the email's sender for notifying exit of queue.
     * @param sender The email's sender
     */
    public void setQueueMailSender(String sender)
    {
        this.setValue(QUEUE_SENDER, sender);
    }
    /**
     * Get the email's receiver for notifying exit of queue.
     * @return The email's receiver
     */
    public Optional<String> getQueueMailReceiver()
    {
        return Optional.ofNullable(getValue(QUEUE_RECEIVER));
    }
    
    /**
     * Set the email's receiver for notifying exit of queue.
     * @param receiver The email's receiver
     */
    public void setQueueMailtReceiver(String receiver)
    {
        this.setValue(QUEUE_RECEIVER, receiver);
    }
    /**
     * Get the email's subject for notifying exit of queue.
     * @return The email's subject
     */
    public Optional<String> getQueueMailSubject()
    {
        return Optional.ofNullable(getValue(QUEUE_SUBJECT));
    }
    
    /**
     * Set the email's subject for notifying exit of queue.
     * @param subject The email's subject
     */
    public void setQueueMailSubject(String subject)
    {
        this.setValue(QUEUE_SUBJECT, subject);
    }
    /**
     * Get the email's body for notifying exit of queue.
     * @return The email's body
     */
    public Optional<String> getQueueMailBody()
    {
        return Optional.ofNullable(getValue(QUEUE_BODY));
    }
    
    /**
     * Set the email's body for notifying exit of queue.
     * @param body The email's body
     */
    public void setQueueMailBody(String body)
    {
        this.setValue(QUEUE_BODY, body);
    }

    /**
     * Get the admin emails
     * @return the admin emails
     */
    public Optional<String[]> getAdminEmails()
    {
        return Optional.ofNullable(getValue(ADMIN_EMAILS));
    }
    
    /**
     * Set the admin emails
     * @param emails the admin emails
     */
    public void setAdminEmails(String[] emails)
    {
        this.setValue(ADMIN_EMAILS, emails);
    }
    
    /**
     * Get the alternative source for admin emails
     * @return the other admin emails
     */
    public Optional<String> getOtherAdminEmails()
    {
        return Optional.ofNullable(getValue(ADMIN_EMAILS_OTHER));
    }
    
    /**
     * Set the alternative source for admin emails
     * @param emails the other admin emails source
     */
    public void setOtherAdminEmails(String emails)
    {
        this.setValue(ADMIN_EMAILS_OTHER, emails);
    }
    
    /**
     * Get the admin email body
     * @return the admin email body
     */
    public Optional<String> getAdminEmailBody()
    {
        return Optional.ofNullable(getValue(ADMIN_EMAIL_BODY));
    }
    
    /**
     * Set the admin email body
     * @param body the admin email body
     */
    public void setAdminEmailBody(String body)
    {
        this.setValue(ADMIN_EMAIL_BODY, body);
    }
    /**
     * Get the admin email subject
     * @return the admin email subject
     */
    public Optional<String> getAdminEmailSubject()
    {
        return Optional.ofNullable(getValue(ADMIN_EMAIL_SUBJECT));
    }
    
    /**
     * Set the admin email subject
     * @param subject the admin email subject
     */
    public void setAdminEmailSubject(String subject)
    {
        this.setValue(ADMIN_EMAIL_SUBJECT, subject);
    }
    
    /**
     * Set the start date
     * @param date The start date
     */
    public void setStartDate(LocalDate date) 
    {
        setValue(START_DATE, date);
    }
    
    /**
     * Get the start date
     * @return The the start date
     */
    public LocalDate getStartDate()
    {
        return this.getValue(START_DATE);
    }
    
    /**
     * Set the end date
     * @param date The end date
     */
    public void setEndDate(LocalDate date) 
    {
        setValue(END_DATE, date);
    }
    
    /**
     * Get the end date
     * @return The the end date
     */
    public LocalDate getEndDate()
    {
        return this.getValue(END_DATE);
    }
    
    public ModifiableIndexableDataHolder getDataHolder()
    {
        JCRRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, this._getFactory().getModel());
    }

    public String getSiteName() throws AmetysRepositoryException
    {
        return getSite().getName();
    }

    public Site getSite() throws AmetysRepositoryException
    {
        AmetysObject parent = getParent();
        while (parent != null && !(parent instanceof Site)) 
        {
            parent = parent.getParent();
        }
        if (parent == null)
        {
            throw new AmetysRepositoryException("An error occurred with form with id '" + getId() + "'. Forms must always be linked to a site");
        }
        return (Site) parent;
    }
    
    @Override
    public Form copyTo(ModifiableTraversableAmetysObject parent, String name) throws AmetysRepositoryException
    {
        Form form = parent.createChild(name, FormFactory.FORM_NODETYPE);
        form.setTitle(getTitle());
        
        for (FormPage page : getPages())
        {
            page.copyTo(form, page.getName());
        }
        
        return form;
    }

    @Override
    public AmetysObject copyTo(ModifiableTraversableAmetysObject parent, String name, List<String> restrictTo) throws AmetysRepositoryException
    {
        return copyTo(parent, name);
    }
    
    /**
     * Get the form pages.
     * @return the form pages.
     * @throws AmetysRepositoryException if an error occurs when retrieving the pages of the form
     */
    public List<FormPage> getPages() throws AmetysRepositoryException
    {
        return getChildren().stream()
                .filter(child -> child instanceof FormPage)
                .map(child -> (FormPage) child)
                .collect(Collectors.toList());
    }
    
    /**
     * <code>true</code> if the form has entries
     * @return <code>true</code> if the form has entries
     */
    public boolean hasEntries()
    {
        return !getEntries().isEmpty();
    }
    /**
     * Get the form entries.
     * @return the form entries.
     * @throws AmetysRepositoryException if an error occurs when retrieving the entries of the form
     */
    public List<FormEntry> getEntries() throws AmetysRepositoryException
    {
        FormEntryDAO formEntryDAO = _getFactory().getFormEntryDAO();
        return formEntryDAO.getFormEntries(this, false, List.of());
    }
    
    /**
     * Get the active form entries.
     * @return a list of the active form entries.
     * @throws AmetysRepositoryException if an error occurs when retrieving the entries of the form
     */
    public List<FormEntry> getActiveEntries() throws AmetysRepositoryException
    {
        FormEntryDAO formEntryDAO = _getFactory().getFormEntryDAO();
        return formEntryDAO.getFormEntries(this, true, List.of());
    }
    
    /**
    *  Get a question by its name.
    * @param name the question name.
    * @return the question.
    * @throws AmetysRepositoryException if an error occurs when retrieving a question of a form
    */
    public FormQuestion getQuestion(String name) throws AmetysRepositoryException
    {
        return getQuestions().stream()
            .filter(q -> q.getNameForForm().equals(name))
            .findFirst()
            .orElse(null);
    }
    
    /**
     * Get the form questions.
     * @return the form questions.
     * @throws AmetysRepositoryException if an error occurs when retrieving all the questions of a form
     */
    public List<FormQuestion> getQuestions() throws AmetysRepositoryException
    {
        return getPages().stream()
                .map(FormPage::getQuestions)
                .flatMap(List::stream)
                .collect(Collectors.toList());
    }
    
    /**
     * Verify that no question has the same name for form as the current question in this form
     * @param uniqueName name for form of the current question
     * @return false if id already exist, true if not
     */
    public boolean isQuestionNameUnique(String uniqueName)
    {
        return !getQuestionsNames().contains(uniqueName);
    }
    
    /**
     * Returns a unique question name in the form
     * @param originalName The original name
     * @return a unique question name
     */
    public String findUniqueQuestionName (String originalName)
    {
        String filterName = NameHelper.filterName(originalName);
        String name = filterName;
        List<String> questionNames = getQuestionsNames();
        while (questionNames.contains(name))
        {
            name = filterName + "-" + org.ametys.core.util.StringUtils.generateKey().toLowerCase();
        }
        return name;
    }
    
    /**
     * Get all the nameForForms of the questions in this form
     * @return a list of the names for form
     */
    public List<String> getQuestionsNames()
    {
        return getQuestions().stream()
                .map(FormQuestion::getNameForForm)
                .toList();
    }
    
    /**
     * Returns a unique question title in the form
     * @param originalTitle The original title
     * @return a unique question title
     */
    public String findUniqueQuestionTitle (String originalTitle)
    {
        String title = originalTitle;
        int index = 1;
        List<String> questionTitles = getQuestions().stream()
            .map(FormQuestion::getTitle)
            .toList();
        while (questionTitles.contains(title))
        {
            title = originalTitle + " " + (index++);
        }
        return title;
    }
    
    /**
     * Get the rules having sourceQuestionId as sourceId
     * @param sourceQuestionId The source question  
     * @return a map of question target's name for form and associated rule
     */
    public Map<FormQuestion, Rule> getQuestionsRule(String sourceQuestionId)
    {
        Map<FormQuestion, Rule> questionsRule = new HashMap<>();
        for (FormQuestion question : getQuestions())
        {
            Optional<Rule> questionRule = question.getFirstQuestionRule();
            if (questionRule.map(q -> q.getSourceId().equals(sourceQuestionId)).orElse(false))
            {
                questionsRule.put(question, questionRule.get());
            }
        }
        return questionsRule;
    }
    
    /**
     * Delete the rules having sourceQuestionId as sourceId
     * @param sourceQuestionId The source question  
     */
    public void deleteQuestionsRule(String sourceQuestionId)
    {
        for (FormQuestion question : getQuestions())
        {
            Optional<Rule> questionRule = question.getFirstQuestionRule();
            
            if (questionRule.map(q -> q.getSourceId().equals(sourceQuestionId)).orElse(false))
            {
                question.getRepeater(FormQuestion.ATTRIBUTE_RULES).removeEntry(0);
            }
        }
        saveChanges();
    }
    
    /**
     * <code>true</code> if the form has a workflow
     * @return <code>true</code> if the form has a workflow
     */
    public boolean hasWorkflow()
    {
        return StringUtils.isNotBlank(getWorkflowName());
    }
    
    @Override
    public boolean canMoveTo(AmetysObject newParent) throws AmetysRepositoryException
    {
        return FormAndDirectoryCommonMethods.canMoveTo(getSiteName(), newParent, this, _getFactory().getFormDirectoriesDAO());
    }
    
    @Override
    public void moveTo(AmetysObject newParent, boolean renameIfExist) throws AmetysRepositoryException, RepositoryIntegrityViolationException
    {
        FormAndDirectoryCommonMethods.moveTo(newParent, renameIfExist, this);
    }
    
    @Override
    public void orderBefore(AmetysObject siblingNode) throws AmetysRepositoryException
    {
        FormAndDirectoryCommonMethods.orderBefore(siblingNode, this);
    }
    
    /**
     * Rights profiles
     */
    public enum FormProfile
    {
        /** Read access */
        READ_ACCESS,
        /** Write access */
        WRITE_ACCESS,
        /** Right access */
        RIGHT_ACCESS;
        
        @Override
        public String toString()
        {
            return name().toLowerCase();
        }
    }
    
    /**
     * Returns true if the form is cacheable.
     * @return true if the form is cacheable.
     */
    public boolean isCacheable()
    {
        // As an empty form or form with unconfigured questions will not be displayed, it is cacheable
        if (getQuestions().isEmpty() || getQuestions().stream().anyMatch(q -> !q.getType().isQuestionConfigured(q)))
        {
            return true;
        }
        
        // If a form can have different states due to the number of entries, or is displayed only at certains dates, the form is not cacheable anymore
        if (isEntriesLimited() || this.isLimitedToOneEntryByUser() || getStartDate() != null || getEndDate() != null)
        {
            return false;
        }
        
        // A form is cacheable only if all questions are cacheable
        return getQuestions().stream().allMatch(FormQuestion::isCacheable);

    }
    

}
