/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository;

import java.time.ZonedDateTime;
import java.util.Optional;

import javax.jcr.Node;

import org.ametys.cms.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.cms.repository.WorkflowAwareContentHelper;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.SimpleAmetysObject;
import org.ametys.plugins.workflow.repository.WorkflowAwareAmetysObject;

/**
 * Class representing a form entry, backed by a JCR node.<br>
 */
public class FormEntry extends SimpleAmetysObject<FormEntryFactory> implements ModifiableModelAwareDataAwareAmetysObject, WorkflowAwareAmetysObject
{
    /** Constant for entry attribute prefix.*/
    public static final String SYSTEM_ATTRIBUTE_PREFIX = "ametys-";
    
    /** Constant for id of an entry.*/
    public static final String ATTRIBUTE_ID = SYSTEM_ATTRIBUTE_PREFIX + "id";
    
    /** Constant for the date and time of submission attribute. */
    public static final String ATTRIBUTE_SUBMIT_DATE = SYSTEM_ATTRIBUTE_PREFIX + "submit-date";
    
    /** The IP address of the person who answered the form. */
    public static final String ATTRIBUTE_IP = SYSTEM_ATTRIBUTE_PREFIX + "ipAddress";
    
    /** The id of the user who answered the form. */
    public static final String ATTRIBUTE_USER = SYSTEM_ATTRIBUTE_PREFIX + "user";
    
    /** Constant for active status of entry. Default is true */
    public static final String ATTRIBUTE_ACTIVE = SYSTEM_ATTRIBUTE_PREFIX + "active";
    
    /**
     * Creates an {@link FormEntry}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public FormEntry(Node node, String parentPath, FormEntryFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Get form
     * @return the form
     */
    public Form getForm()
    {
        return this.getParent().getParent();
    }
    
    /**
     * Retrieves the id.
     * @return the id.
     */
    public Long getEntryId()
    {
        return getValue(ATTRIBUTE_ID);
    }
    
    /**
     * Set the id.
     * @param id the id.
     */
    public void setEntryId(Long id)
    {
        setValue(ATTRIBUTE_ID, id);
    }
    
    /**
     * Retrieves the ip.
     * @return the ip.
     */
    public String getIP()
    {
        return getValue(ATTRIBUTE_IP);
    }
    
    /**
     * Set the ip.
     * @param ip the ip.
     */
    public void setIP(String ip)
    {
        setValue(ATTRIBUTE_IP, ip);
    }
    
    /**
     * Retrieves the submitDate.
     * @return the submitDate.
     */
    public ZonedDateTime getSubmitDate()
    {
        return getValue(ATTRIBUTE_SUBMIT_DATE);
    }
    
    /**
     * Set the submitDate.
     * @param submitDate the submitDate.
     */
    public void setSubmitDate(ZonedDateTime submitDate)
    {
        setValue(ATTRIBUTE_SUBMIT_DATE, submitDate);
    }
    
    /**
     * Set the identity of the user answering the form
     * @param user the user identity
     */
    public void setUser(UserIdentity user)
    {
        setValue(ATTRIBUTE_USER, user);
    }
    
    /**
     * Get active status of the entry
     * @return the active status
     */
    public Boolean isActive()
    {
        return getValue(ATTRIBUTE_ACTIVE, false, true);
    }
    
    /**
     * Set active status of the entry
     * @param isActive the active status
     */
    public void setActive(Boolean isActive)
    {
        setValue(ATTRIBUTE_ACTIVE, isActive);
    }
    
    /**
     * Get the identity of the user answering the form
     * @return the user identity
     */
    public UserIdentity getUser()
    {
        return getValue(ATTRIBUTE_USER);
    }
    
    public long getWorkflowId() throws AmetysRepositoryException
    {
        return WorkflowAwareContentHelper.getWorkflowId(this);
    }

    public void setWorkflowId(long workflowId) throws AmetysRepositoryException
    {
        WorkflowAwareContentHelper.setWorkflowId(this, workflowId);
    }

    public long getCurrentStepId() throws AmetysRepositoryException
    {
        return WorkflowAwareContentHelper.getCurrentStepId(this);
    }

    public void setCurrentStepId(long stepId) throws AmetysRepositoryException
    {
        WorkflowAwareContentHelper.setCurrentStepId(this, stepId);
    }
    
    @Override
    public ModifiableIndexableDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, _getFactory().getFormEntryModel(getForm()), Optional.empty(), Optional.empty(), Optional.of(this));
    }
}
