/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.io.IOException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.ui.RibbonControlsManager;
import org.ametys.plugins.workflow.support.WorkflowHelper;

/**
 * This class generates the ids of the extensions used for each workflow tab
 */
public abstract class AbstractFormEntriesWorkflowsGenerator extends ServiceableGenerator
{
    /** The beginning of the id every workflow tab extension */
    private static final String __FORM_WORKFLOW_TAB_EXTENSION_ID = "form-workflow-tab";
    
    /** The Workflow component for form entries */
    protected WorkflowHelper _workflowHelper;
    
    /** The ribbon control manager */
    protected RibbonControlsManager _ribbonControlsManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _workflowHelper = (WorkflowHelper) serviceManager.lookup(WorkflowHelper.ROLE);
        _ribbonControlsManager = (RibbonControlsManager) serviceManager.lookup(RibbonControlsManager.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        String[] formWorkflowNames = _workflowHelper.getWorkflowNames();
        
        contentHandler.startDocument();
        
        AttributesImpl rootAttrs = new AttributesImpl();
        rootAttrs.addCDATAAttribute("tabLabelKey", _getRibbonTabKeyToOverride());
        XMLUtils.startElement(contentHandler, "root", rootAttrs);
        
        AttributesImpl workflowTabAtts = new AttributesImpl();
        workflowTabAtts.addCDATAAttribute("tabId", __FORM_WORKFLOW_TAB_EXTENSION_ID);

        AttributesImpl groupAtts = new AttributesImpl();
        groupAtts.addCDATAAttribute("label", "plugin.forms:RIBBON_TABS_TAB_FORM_ENTRY_WORKFLOW_LABEL");
        groupAtts.addCDATAAttribute("order", "2");
        XMLUtils.startElement(contentHandler, "workflow-tab", workflowTabAtts);
        XMLUtils.startElement(contentHandler, "group", groupAtts);
        for (String workflowName : formWorkflowNames)
        {
            if (workflowName.startsWith(_getPrefixWorkflowName()))
            {
                String defaultWorkflowButtonsId = _getDefaultWorkflowButtonsId();
                String id = _getPrefixWorkflowButton() + workflowName;
                if (_ribbonControlsManager.hasExtension(id)) // Test if the workflow buttons are defined in the plugin
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("id", id);
                    XMLUtils.createElement(contentHandler, "control", attrs);
                }
                else if (StringUtils.isNotBlank(defaultWorkflowButtonsId))
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("ref-id", defaultWorkflowButtonsId);
                    XMLUtils.startElement(contentHandler, "control", attrs);
                    AttributesImpl workflowAttrs = new AttributesImpl();
                    workflowAttrs.addCDATAAttribute("name", workflowName);
                    XMLUtils.createElement(contentHandler, "workflow", workflowAttrs);
                    XMLUtils.endElement(contentHandler, "control");
                }
                
            }
        }
        XMLUtils.endElement(contentHandler, "group");
        XMLUtils.endElement(contentHandler, "workflow-tab");
        XMLUtils.endElement(contentHandler, "root");
        
        contentHandler.endDocument();
    }
    
    /**
     * Get prefix of workflows
     * @return the prefix of workflows
     */
    protected abstract String _getPrefixWorkflowName();
    
    /**
     * Get prefix of workflow buttons
     * @return the prefix of workflow buttons
     */
    protected abstract String _getPrefixWorkflowButton();
    
    /**
     * Get the id of default implementation of workflow buttons
     * @return the id of default implementation of workflow buttons
     */
    protected abstract String _getDefaultWorkflowButtonsId();
    
    /**
     * Get label key of ribbon tab to override
     * @return the label key of ribbon tab to override
     */
    protected abstract String _getRibbonTabKeyToOverride();
}
