/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.io.IOException;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder.UserInput;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.helper.FormMailHelper;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.plugins.workflow.component.WorkflowArgument;
import org.ametys.plugins.workflow.support.WorkflowElementDefinitionHelper;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.web.repository.site.SiteManager;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

import jakarta.mail.MessagingException;

/**
 * OS workflow function to send mail after an action is triggered.
 * The author of a form entry is also notified if the receipt is set
 */
public class FormSendMailFunction extends AbstractLogEnabled implements Component, EnhancedFunction, Serviceable
{
    /** Sender argument of the email */
    public static final String ARG_SENDER = "sender";
    /** Sender role argument of the email */
    public static final String ARG_SENDER_ROLE = "sender-role";
    /** Recipient argument of the email */
    public static final String ARG_RECIPIENT = "recipient";
    /** Recipient role argument of the email */
    public static final String ARG_RECIPIENT_ROLE = "recipient-role";
    /** Subject key argument of the email */
    public static final String ARG_SUBJECT_KEY = "subject-key";
    /** Body key argument of the email */
    public static final String ARG_BODY_KEY = "body-key";
    /** Body xsl argument of the email */
    public static final String ARG_BODY_XSL = "body-xsl";
    /** Comment argument of the email */
    public static final String ARG_COMMENT = "comment";
    /** Send if comment argument of the email */
    public static final String ARG_SEND_IF_COMMENT = "send-if-comment";

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The user manager */
    protected UserManager _userManager;

    /** The source resolver. */
    protected SourceResolver _sourceResolver;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** I18nUtils */
    protected I18nUtils _i18nUtils; 
    
    /** The workflow form mail EP */
    protected WorkflowFormMailExtensionPoint _workflowFormMailEP;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The form mail helper */
    protected FormMailHelper _formMailHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _workflowFormMailEP = (WorkflowFormMailExtensionPoint) manager.lookup(WorkflowFormMailExtensionPoint.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _formMailHelper = (FormMailHelper) manager.lookup(FormMailHelper.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        String sendIfComment = _getStringParam(args, ARG_SEND_IF_COMMENT);
        String comment = _getStringParam(transientVars, ARG_COMMENT);
        // If the parameter send-if-comment is true, send the mail only if the comment is not blank
        if (!StringUtils.equals(sendIfComment, "true") || StringUtils.isNotBlank(comment))
        {
            FormEntry formEntry = (FormEntry) transientVars.get(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY);
            
            String sender = _getSender(formEntry, transientVars, args);
            List<UserAndMail> recipients = _getRecipients(formEntry, transientVars, args);
            try
            {
                String lang = _formDAO.getFormLocale(formEntry.getForm());
                I18nizableText subject = _getSubject(formEntry, transientVars, args);
                String subjectAsString = _i18nUtils.translate(subject, lang);
                for (UserAndMail recipient : recipients)
                {
                    String body = _getBody(formEntry, transientVars, args, recipient, subject);

                    try
                    {
                        SendMailHelper.newMail()
                            .withAsync(true)
                            .withSubject(subjectAsString)
                            .withHTMLBody(body)
                            .withSender(sender)
                            .withRecipient(recipient.mail())
                            .sendMail();
                    }
                    catch (MessagingException | IOException e)
                    {
                        getLogger().warn("Could not send a workflow notification mail to " + recipient, e);
                    }
                }
            }
            catch (IOException e)
            {
                getLogger().warn("Could not send a workflow notification mail", e);
            }
        }
    }
    
    /**
     * Get the sender of the mail
     * First look at the configured sender role in sender-role
     * Then look at the configured sender in sender
     * Then take the current user
     * Then take the sender in admin configuration
     * @param formEntry the form entry
     * @param transientVars the workflow transient vars
     * @param args the workflow args 
     * @return the mail subject as string
     */
    protected String _getSender(FormEntry formEntry, Map transientVars, Map args)
    {
        String role = _getStringParam(args, ARG_SENDER_ROLE);
        if (StringUtils.isNotBlank(role))
        {
            WorkflowFormMail workflowMail = _workflowFormMailEP.getExtension(role);
            List<String> emails = workflowMail.getEmails(formEntry, transientVars, args);
            if (emails != null && !emails.isEmpty())
            {
                return emails.get(0);
            }
        }

        String sender = _getStringParam(args, ARG_SENDER);
        if (StringUtils.isNotBlank(sender))
        {
            return sender;
        }
        else
        {
            User user = _userManager.getUser(_currentUserProvider.getUser());
            String mail = user != null ? user.getEmail() : null;
            if (StringUtils.isNotBlank(mail))
            {
                return mail;
            }
        }
        
        return Config.getInstance().getValue("smtp.mail.from");
    }
    
    /**
     * Get the list of recipients to send the mail
     * First look at the configured recipient role in recipient-role
     * Then look at the configured recipient in recipient
     * Then take the entry user
     * @param formEntry the form entry
     * @param transientVars the workflow transient vars
     * @param args the workflow args 
     * @return the list of user or mail 
     */
    protected List<UserAndMail> _getRecipients(FormEntry formEntry, Map transientVars, Map args)
    {
        String role = _getStringParam(args, ARG_RECIPIENT_ROLE);
        if (StringUtils.isNotBlank(role))
        {
            WorkflowFormMail workflowMail = _workflowFormMailEP.getExtension(role);
            List<String> emails = workflowMail.getEmails(formEntry, transientVars, args);
            if (emails != null && !emails.isEmpty())
            {
                return emails.stream()
                        .map(mail -> new UserAndMail(null, mail))
                        .toList();
            }
        }
        
        String recipient = _getStringParam(args, ARG_RECIPIENT);
        if (StringUtils.isNotBlank(recipient))
        {
            return List.of(new UserAndMail(null, recipient));
        }
        else
        {
            User user = _userManager.getUser(formEntry.getUser());
            return user != null ? List.of(new UserAndMail(user, user.getEmail())) : List.of();
        }
    }
    
    /**
     * Get the mail subject
     * First look at the configured subject i18n key in subject-key
     * Then take the default i18n key
     * @param formEntry the form entry
     * @param transientVars the workflow transient vars
     * @param args the workflow args 
     * @return the mail subject as string
     */
    protected I18nizableText _getSubject(FormEntry formEntry, Map transientVars, Map args)
    {
        String subjectI18nKey = _getStringParam(args, ARG_SUBJECT_KEY);
        if (StringUtils.isNotBlank(subjectI18nKey))
        {
            return new I18nizableText(null, subjectI18nKey, _getSubjectI18nParams(formEntry));
        }
        
        return new I18nizableText(null, "", _getSubjectI18nParams(formEntry));
    }
    
    /**
     * Get the mail body
     * First look at the configured xsl in mail-xsl
     * Then look at the configured body i18n key in body-key
     * Then take the default i18n key
     * @param formEntry the form entry
     * @param transientVars the workflow transient vars
     * @param args the workflow args 
     * @param receiver the receiver
     * @param subject the subject of the mail
     * @return the mail body as string
     * @throws IOException if an error occurred
     */
    protected String _getBody(FormEntry formEntry, Map transientVars, Map args, UserAndMail receiver, I18nizableText subject) throws IOException
    {
        String xslPath = _getStringParam(args, ARG_BODY_XSL);
        if (StringUtils.isNotBlank(xslPath))
        {
            String lang = _formDAO.getFormLocale(formEntry.getForm());
            
            String comment = _getStringParam(transientVars, ARG_COMMENT);
            Map<String, Object> additionalParams = new HashMap<>();
            additionalParams.put("xsl", xslPath);
            if (StringUtils.isNotBlank(comment))
            {
                additionalParams.put("comment", comment);
            }
            
            User user = receiver.user();
            boolean withReadingRestriction = user != null ? Objects.equals(user.getIdentity(), formEntry.getUser()) : false;
            
            String body = _formMailHelper.getMail("send-mail-function.html", formEntry, additionalParams, withReadingRestriction);
            
            return StandardMailBodyHelper.newHTMLBody()
                .withLanguage(lang)
                .withTitle(subject)
                .withMessage(body)
                .build();
        }
        
        return _getBodyFromI18nKey(formEntry, transientVars, args, subject);
    }
    
    /**
     * Get the mail body from i18n key
     * First look at the configured body i18n key in body-key
     * Then take the default i18n key
     * @param formEntry the form entry
     * @param transientVars the workflow transient vars
     * @param args the workflow args 
     * @param subject the mail subject
     * @return the mail body as string
     * @throws IOException if an error occurred
     */
    protected String _getBodyFromI18nKey(FormEntry formEntry, Map transientVars, Map args, I18nizableText subject) throws IOException
    {
        String lang = _formDAO.getFormLocale(formEntry.getForm());
        
        String comment = _getStringParam(transientVars, ARG_COMMENT);
        String bodyI18nKey = _getStringParam(args, ARG_BODY_KEY);
        I18nizableText bodyKey = StringUtils.isNotBlank(bodyI18nKey)
                ? new I18nizableText(null, bodyI18nKey, _getBodyI18nParams(formEntry))
                : new I18nizableText("plugin.forms", "PLUGINS_FORMS_SEND_MAIL_FUNCTION_DEFAULT_BODY", _getBodyI18nParams(formEntry));
        
        MailBodyBuilder mailBody = StandardMailBodyHelper.newHTMLBody()
                .withLanguage(lang)
                .withTitle(subject)
                .withMessage(bodyKey);
        
        // Get the workflow comment
        if (StringUtils.isNotEmpty(comment))
        {
            List<String> params = new ArrayList<>();
            params.add(comment);
            
            UserIdentity userIdentity = _currentUserProvider.getUser();
            mailBody.withUserInput(new UserInput(_userManager.getUser(userIdentity), ZonedDateTime.now(), comment.replaceAll("\r?\n", "<br/>")));
        }
        
        return mailBody.build();
    }
    
    /**
     * Get the i18n parameters of mail subject
     * @param formEntry the form entry
     * @return the i18n parameters
     */
    protected List<String> _getSubjectI18nParams (FormEntry formEntry)
    {
        List<String> params = new ArrayList<>();
        params.add(formEntry.getForm().getTitle());  // {0}
        
        User user = _userManager.getUser(formEntry.getUser());
        if (user != null)
        {
            params.add(user.getFullName()); // {1}
        }
        
        return params;
    }
    
    /**
     * Get the i18n parameters of mail body text
     * @param formEntry the form entry
     * @return the i18n parameters
     */
    protected List<String> _getBodyI18nParams (FormEntry formEntry)
    {
        Form form = formEntry.getForm();
        String siteName = form.getSiteName();
        
        UserIdentity userIdentity = formEntry.getUser();
        User user = userIdentity != null ? _userManager.getUser(userIdentity) : null;
        
        List<String> params = new ArrayList<>();
        
        params.add(user != null ? user.getFullName() : StringUtils.EMPTY); // {0}
        params.add(form.getTitle()); // {1}
        params.add(_formDAO.getDashboardUri(siteName)); // {2}
        params.add(_formDAO.getAdminDashboardUri(siteName)); // {3}
        
        return params;
    }

    private String _getStringParam(Map args, String key)
    {
        Object object = args.get(key);
        return object != null ? (String) object : null;
    }
    
    private record UserAndMail(User user, String mail) { /* empty */ }

    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.POST;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public List<WorkflowArgument> getArguments()
    {
        WorkflowArgument senderRole = WorkflowElementDefinitionHelper.getElementDefinition(
            ARG_SENDER_ROLE,
            new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_SENDER_ROLE_LABEL"),
            new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_SENDER_ROLE_DESCRIPTION"),
            false,
            false
        );
        StaticEnumerator<String> roleStaticEnumerator = new StaticEnumerator<>();
        for (String epId : _workflowFormMailEP.getExtensionsIds())
        {
            roleStaticEnumerator.add(new I18nizableText(epId), epId);
        }
        senderRole.setEnumerator(roleStaticEnumerator);
        
        WorkflowArgument recipientRole = WorkflowElementDefinitionHelper.getElementDefinition(
            ARG_RECIPIENT_ROLE,
            new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_ROLE_LABEL"),
            new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_ROLE_DESCRIPTION"),
            false,
            false
        );
        recipientRole.setEnumerator(roleStaticEnumerator);
        
        StaticEnumerator<String> booleanStaticEnumerator = new StaticEnumerator<>();
        booleanStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_TRUE_LABEL"), "true");
        booleanStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_FALSE_LABEL"), "false");
        WorkflowArgument sendIfComment = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_SEND_IF_COMMENT,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_SEND_IF_COMMENT_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_SEND_IF_COMMENT_DESCRIPTION"),
                false,
                false
                );
        sendIfComment.setEnumerator(booleanStaticEnumerator);
        
        return List.of(
            WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_SENDER,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_SENDER_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_SENDER_DESCRIPTION"),
                false,
                false
            ), 
            senderRole, 
            WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_RECIPIENT,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_DESCRIPTION"),
                false,
                true
            ), 
            recipientRole, 
            WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_SUBJECT_KEY,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_SUBJECT_KEY_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_SUBJECT_KEY_DESCRIPTION"),
                false,
                false
            ), 
            WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_BODY_KEY,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_BODY_KEY_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_BODY_KEY_DESCRIPTION"),
                false,
                false
            ), 
            WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_BODY_XSL,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_BODY_XSL_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_BODY_XSL_DESCRIPTION"),
                false,
                false
            ), 
            WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_COMMENT,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_COMMENT_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_ARGUMENT_COMMENT_DESCRIPTION"),
                false,
                false
            ), 
            sendIfComment
        );
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORM_SEND_MAIL_FUNCTION_LABEL");
    }
}
