/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This factory creates Ametys.message.MessageTarget for a form directory.
 * See #createTargets for more details.
 * @private
 */
Ext.define('Ametys.plugins.forms.FormDirectoryMessageTargetFactory', {
    extend: 'Ametys.message.factory.DefaultMessageTargetFactory',
        
    /**
     * Create the targets for a message
     * @param {Object} parameters The parameters needed by the factory to create the message. Can not be null. Handled elements are
     * @param {String[]} parameters.id The form's identifiers.
     * @param {Function} callback The callback function called when the targets are created. Parameters are
     * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
     */
    createTargets: function(parameters, callback)
    {
        if (parameters.formDirectory)
        {
            this._createTargets(parameters.formDirectory, callback);
        }
        else
        {
            var id = parameters.id;
            Ametys.plugins.forms.dao.FormDirectoryDAO.getFormDirectoryProperties([Ametys.getAppParameter("siteName"), id], Ext.bind(this._createTargets, this, [callback], 1));
        }
    },

    /**
     * Create the form directories targets
     * @param {Ametys.plugins.forms.FormDirectory} formDirectory The form directory
     * @param {Function} callback The callback function called when the targets are created.
     * @private
     */
    _createTargets: function (formDirectory, callback)
    {
        var targets = [];
        
        targets.push(Ext.create("Ametys.message.MessageTarget", {
            id: Ametys.message.MessageTarget.FORM_DIRECTORY,
            parameters: Ext.merge(formDirectory.getProperties(), {formDirectory: formDirectory})
        }));
            
        callback(targets);
    }
});

/**
 * This factory creates Ametys.message.MessageTarget for a form.
 * See #createTargets for more details.
 * @private
 */
Ext.define('Ametys.plugins.forms.FormMessageTargetFactory', {
	extend: 'Ametys.message.factory.DefaultMessageTargetFactory',
		
	/**
	 * Create the targets for a message
	 * @param {Object} parameters The parameters needed by the factory to create the message. Can not be null. Handled elements are
	 * @param {String[]} parameters.id The form's identifier.
	 * @param {Function} callback The callback function called when the targets are created. Parameters are
	 * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
	 */
    createTargets: function(parameters, callback)
    {
        if (parameters.form)
        {
            this._createTargets(parameters.form, callback);
        }
        else
        {
            var id = parameters.id;
            Ametys.plugins.forms.dao.FormDAO.getFormProperties([id, true, true], Ext.bind(this._createTargets, this, [callback], 1));
        }
    },

	/**
	 * Create the forms targets
     * @param {Ametys.plugins.forms.Form} form The form
	 * @param {Function} callback The callback function called when the targets are created.
	 * @private
	 */
	_createTargets: function (form, callback)
	{
		var targets = [];
        
        targets.push(Ext.create("Ametys.message.MessageTarget", {
            id: Ametys.message.MessageTarget.FORM_TARGET,
            parameters: Ext.merge(form.getProperties(), {form: form})
        }));
            
        callback(targets);
	}
});

/**
 * This factory creates Ametys.message.MessageTarget for a form page.
 * See #createTargets for more details.
 * @private
 */
Ext.define('Ametys.plugins.forms.PageMessageTargetFactory', {
	extend: 'Ametys.message.factory.DefaultMessageTargetFactory',
	
	/**
	 * Create the targets for a message
	 * @param {Object} parameters The parameters needed by the factory to create the message. Can not be null. Handled elements are
	 * @param {String[]} parameters.id The form page's identifier.
	 * @param {Function} callback The callback function called when the targets are created.
	 * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
	 */
    createTargets: function(parameters, callback)
    {
        if (parameters.page)
        {
            this._createTargets(parameters.page, callback);
        }
        else
        {
            var id = parameters.id;
            Ametys.plugins.forms.dao.FormPageDAO.getPageProperties([id], Ext.bind(this._createTargets, this, [callback], 1));
        }
    },

	/**
	 * Create the form pages targets
     * @param {Ametys.plugins.forms.Page} page The form page
	 * @param {Function} callback The callback function called when the targets are created. Parameters are
	 * @private
	 */
	_createTargets: function (page, callback)
	{
		var targets = [];
        
        targets.push(Ext.create("Ametys.message.MessageTarget", {
            id: Ametys.message.MessageTarget.FORM_PAGE,
            parameters: Ext.merge(page.getProperties(), {page: page})
        }));
            
        callback(targets);
	}
});


/**
 * This factory creates Ametys.message.MessageTarget for a form page.
 * See #createTargets for more details.
 * @private
 */
Ext.define('Ametys.plugins.forms.QuestionMessageTargetFactory', {
	extend: 'Ametys.message.factory.DefaultMessageTargetFactory',
	
	/**
	 * Create the targets for a message
	 * @param {Object} parameters The parameters needed by the factory to create the message. Can not be null. Handled elements are
	 * @param {String[]} parameters.id The form question's identifier.
	 * @param {Function} callback The callback function called when the targets are created.
	 * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
	 */
    createTargets: function(parameters, callback)
    {
        if (parameters.question)
        {
            this._createTargets(parameters.question, callback);
        }
        else
        {
            var id = parameters.id;
            Ametys.plugins.forms.dao.FormQuestionDAO.getQuestionProperties([id], Ext.bind(this._createTargets, this, [callback], 1));
        }
    },

	/**
	 * Create the form question targets
     * @param {Ametys.plugins.forms.Question} question The question
	 * @param {Function} callback The callback function called when the targets are created. Parameters are
	 * @private
	 */
	_createTargets: function (question, callback)
	{
		var targets = [];
        
        targets.push(Ext.create("Ametys.message.MessageTarget", {
            id: Ametys.message.MessageTarget.FORM_QUESTION,
            parameters: Ext.merge(question.getProperties(), {question: question})
        }));
            
        callback(targets);
	}
});

/**
 * This factory creates Ametys.message.MessageTarget for a form entry.
 * See #createTargets for more details.
 * @private
 */
Ext.define('Ametys.plugins.forms.EntryMessageTargetFactory', {
	extend: 'Ametys.message.factory.DefaultMessageTargetFactory',
	
	/**
	 * Create the targets for a message
	 * @param {Object} parameters The parameters needed by the factory to create the message. Can not be null. Handled elements are
	 * @param {String[]} parameters.ids The form entry's identifiers.
	 * @param {Function} callback The callback function called when the targets are created. Parameters are
	 * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
	 */
    createTargets: function(parameters, callback)
    {
        if (parameters.entry)
        {
            this._createTargets(parameters.entry, callback);
        }
        else if (parameters.id)
        {
            Ametys.plugins.forms.dao.FormEntryDAO.getFormEntryProperties([parameters.id], Ext.bind(this._createTargets, this, [callback], 1));
        }
    },

	/**
	 * Create the form entries targets
     * @param {Ametys.plugins.forms.FormEntry} entry The form entry
	 * @param {Function} callback The callback function called when the targets are created.
	 * @private
	 */
	_createTargets: function (entry, callback)
	{
		var targets = [];
        
        targets.push(Ext.create("Ametys.message.MessageTarget", {
            id: Ametys.message.MessageTarget.FORM_ENTRY,
            parameters: Ext.merge(entry.getProperties(), {entry: entry})
        }));
            
        callback(targets);
	}
});