/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;

import org.ametys.core.util.DateUtils;

/** Forms date utils */
public final class FormDateUtils
{
    private FormDateUtils()
    {
        // empty
    }
    
    /** 
     * Converts this {@link LocalDate} object to a {@link ZonedDateTime}.
     * @param date the LocalDate
     * @return an instant representing the same point on the time-line as this {@link LocalDate} object
     */
    public static ZonedDateTime asZonedDateTime(LocalDate date)
    {
        return date.atStartOfDay(ZoneId.of("UTC"));
    }
    
    /** 
     * Converts a {@link ZonedDateTime} object to {@link String} using the ISO date formatter
     * @param zonedDateTime the zoned date time
     * @return the zoned date time as a {@link String}
     */
    public static String zonedDateTimeToString(ZonedDateTime zonedDateTime)
    {
        return zonedDateTime.format(DateUtils.getISODateTimeFormatter());
    }
    
    /** 
     * Parses a String into a {@link ZonedDateTime}, using ISO date formatter.
     * @param zonedDateTimeAsString the zoned date time as string
     * @return the {@link ZonedDateTime} object
     */
    public static ZonedDateTime parseZonedDateTime(String zonedDateTimeAsString)
    {
        return ZonedDateTime.parse(zonedDateTimeAsString, DateUtils.getISODateTimeFormatter());
    }
}
