/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms;

import java.io.IOException;
import java.util.Map.Entry;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.AbstractGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.forms.actions.ProcessFormAction;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;

import com.google.common.collect.Multimap;

/**
 * Action that processes the user submitted data on a form.
 */
public class FormErrorsGenerator extends AbstractGenerator
{
    @SuppressWarnings("unchecked")
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        Form form = (Form) request.getAttribute("form");
        FormErrors errors = new FormErrors((Multimap<String, I18nizableText>) request.getAttribute("form-errors"));
        
        Multimap<String, I18nizableText> errorMap = errors.getErrors();
        
        contentHandler.startDocument();
        
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("id", form.getId());
        atts.addCDATAAttribute("title", form.getTitle());
        
        XMLUtils.startElement(contentHandler, "form-errors", atts);
        
        for (Entry<String, I18nizableText> fieldErrors : errorMap.entries())
        {
            String errorName = fieldErrors.getKey();
            I18nizableText message = fieldErrors.getValue();
            
            if (StringUtils.isEmpty(errorName) 
                    || "form-access".equals(errorName) 
                    || "entries-limit-reached".equals(errorName)
                    || "scheduled-not-open".equals(errorName)
                )
            {
                XMLUtils.startElement(contentHandler, "form-access");
                message.toSAX(contentHandler, "error");
                XMLUtils.endElement(contentHandler, "form-access");
            }
            else if (ProcessFormAction.CAPTCHA_KEY.equals(errorName))
            {
                XMLUtils.startElement(contentHandler, "captcha");
                message.toSAX(contentHandler, "error");
                XMLUtils.endElement(contentHandler, "captcha");
            }
            else
            {
                FormQuestion question = form.getQuestion(errorName);
                
                atts.clear();
                atts.addCDATAAttribute("id", question.getId());
                atts.addCDATAAttribute("name", errorName);
                atts.addCDATAAttribute("title", question.getTitle());
                atts.addCDATAAttribute("type", question.getType().getId());
                
                XMLUtils.startElement(contentHandler, "question", atts);
                
                message.toSAX(contentHandler, "error");
                
                XMLUtils.endElement(contentHandler, "question");
            }
        }
        XMLUtils.endElement(contentHandler, "form-errors");
        
        contentHandler.endDocument();
    }
}
