/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Stream;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.dao.FormDirectoryDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.Form.FormProfile;
import org.ametys.plugins.forms.repository.FormDirectory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;

import com.google.common.base.Predicates;


/**
 * Get forms action
 */
public class GetFormsAction extends ServiceableAction
{
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** DAO for manipulating form directory */
    protected FormDirectoryDAO _formDirectoryDAO;
    
    /** DAO for manipulating form */
    protected FormDAO _formDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _userProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _formDirectoryDAO = (FormDirectoryDAO) serviceManager.lookup(FormDirectoryDAO.ROLE);
        _formDAO = (FormDAO) serviceManager.lookup(FormDAO.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        String profileId = StringUtils.defaultIfEmpty((String) jsParameters.get("profile"), "read_access");
        
        if (!"read_access".equals(profileId) && !"write_access".equals(profileId) && !"right_access".equals(profileId))
        {
            throw new IllegalArgumentException("Unexpected profile identifier : " + profileId);
        }
        
        FormProfile profile = FormProfile.valueOf(profileId.toUpperCase());
        FormDirectory parentDirectory = _getParentDirectory(jsParameters);
        Boolean onlyDirectories = (Boolean) jsParameters.getOrDefault("onlyDirectories", Boolean.FALSE);
        Boolean onlyConfiguredForm = (Boolean) jsParameters.getOrDefault("onlyConfiguredForm", Boolean.FALSE);
        Boolean onlyForms = onlyDirectories ? false : (Boolean) jsParameters.getOrDefault("onlyForms", Boolean.FALSE);
        Boolean testIfEmptyDirectory = (Boolean) jsParameters.getOrDefault("testIfEmptyDirectory", Boolean.FALSE);
        Boolean allDescendants = (Boolean) jsParameters.getOrDefault("allDescendants", Boolean.FALSE);
        
        UserIdentity user = _userProvider.getUser();
        
        List<Map<String, Object>> nodes = new ArrayList<>();
        Stream<Form> childFormsStream = onlyDirectories
                ? Stream.empty()
                : _getChildForms(parentDirectory, !allDescendants, profile, user)
                    .filter(f -> !onlyConfiguredForm || _formDAO.isFormConfigured(f));
        childFormsStream
                .map(f -> _formDAO.getFormProperties(f, false, false))
                .forEach(nodes::add);
        
        Stream<Pair<FormDirectory, AdditionalInfoOnDirectory>> childDirectoriesStream = onlyForms
                ? Stream.empty()
                : _getChildFormDirectories(parentDirectory, testIfEmptyDirectory, profile, user);
        childDirectoriesStream
                .forEach(p ->
                {
                    Map<String, Object> props = _formDirectoryDAO.getFormDirectoryProperties(p.getLeft(), false);
                    AdditionalInfoOnDirectory additionalInfo = p.getRight();
                    additionalInfo.fillAdditionalInfo(props);
                    switch (profile)
                    {
                        case WRITE_ACCESS:
                            if ((boolean) props.get("displayForWrite"))
                            {
                                nodes.add(props);
                            }
                            break;
                        case RIGHT_ACCESS:
                            if ((boolean) props.get("displayForRights"))
                            {
                                nodes.add(props);
                            }
                            break;
                        case READ_ACCESS:
                        default:
                            if ((boolean) props.get("displayForRead"))
                            {
                                nodes.add(props);
                            }
                            break;
                    }
                });
        
        Map<String, Object> result = new HashMap<>();
        result.put("forms", nodes);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
    
    private FormDirectory _getParentDirectory(Map jsParameters)
    {
        Predicate<String> isNotRoot = Predicates.not(Predicates.equalTo(FormDirectoryDAO.ROOT_FORM_DIRECTORY_ID));
        FormDirectory parent = Optional.of("node")
            .map(jsParameters::get)
            .filter(String.class::isInstance)
            .map(String.class::cast)
            .filter(isNotRoot)
            .map(_resolver::<FormDirectory>resolveById)
            .orElse(_formDirectoryDAO.getFormDirectoriesRootNode((String) jsParameters.get("siteName"))/* root node must exist, so it must be created if it does not exist yet */);
        return parent;
    }
    
    private Stream<Pair<FormDirectory, AdditionalInfoOnDirectory>> _getChildFormDirectories(FormDirectory parentDirectory, boolean testIfEmptyDirectory, FormProfile profile, UserIdentity user)
    {
        AmetysObjectIterable<FormDirectory> childFormDirectories = _formDirectoryDAO.getChildFormDirectories(parentDirectory);
        
        Stream<FormDirectory> childFormDirectoriesStream = childFormDirectories.stream();
        if (testIfEmptyDirectory)
        {
            // indicates if directory do not have (not necessarily direct) form children (i.e. is empty)
            return childFormDirectoriesStream
                    .map(ct -> Pair.of(ct, new AdditionalInfoOnDirectory(_hasNoDescendant(ct, profile, user))));
        }
        else
        {
            return childFormDirectoriesStream
                    .map(ct -> Pair.of(ct, new AdditionalInfoOnDirectory()));
        }
    }
    
    private Boolean _hasNoDescendant(FormDirectory formDirectory, FormProfile profile, UserIdentity user)
    {
        boolean onlyDirectChildren = false; // in order to test no only direct children, but all descendants
        Stream<Form> childForms = _getChildForms(formDirectory, onlyDirectChildren, profile, user);
        return !childForms.findAny().isPresent();
    }
    
    private Stream<Form> _getChildForms(FormDirectory parentDirectory, boolean onlyDirectChildren, FormProfile profile, UserIdentity user)
    {
        Stream<Form> childForms;
        switch (profile)
        {
            case WRITE_ACCESS:
                childForms = _formDirectoryDAO.getChildFormsInWriteAccess(parentDirectory, onlyDirectChildren, user);
                break;
            case RIGHT_ACCESS:
                childForms = _formDirectoryDAO.getChildFormsInRightAccess(parentDirectory, onlyDirectChildren, user);
                break;
            case READ_ACCESS:
            default:
                childForms = _formDirectoryDAO.getChildFormsInReadAccess(parentDirectory, onlyDirectChildren, user);
                break;
        }
        
        return childForms;
    }
    
    private static class AdditionalInfoOnDirectory
    {
        private Optional<Boolean> _hasNoDescendant = Optional.empty();
        
        AdditionalInfoOnDirectory()
        {
        }
        
        AdditionalInfoOnDirectory(boolean hasNoDescendant)
        {
            _hasNoDescendant = Optional.of(hasNoDescendant);
        }
        
        void fillAdditionalInfo(Map<String, Object> props)
        {
            _hasNoDescendant.ifPresent(b -> props.put("hasNoDescendantForm", b));
        }
    }
}
