/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.jcr;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.JcrConstants;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.Field;
import org.ametys.plugins.forms.content.Field.FieldType;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.repository.site.SiteManager;

/**
 * Form properties manager : stores and retrieves form properties.
 */
public class FormPropertiesManager extends AbstractLogEnabled implements Serviceable, Component, PluginAware
{
    /** Pattern for options value */
    public static final Pattern OPTION_VALUE_PATTERN = Pattern.compile("^option-([0-9]+)-value$");
    
    /** The avalon component ROLE. */
    public static final String ROLE = FormPropertiesManager.class.getName();
    
    /** JCR relative path to root node. */
    public static final String ROOT_REPO = AmetysObjectResolver.ROOT_REPO;
    
    /** Plugins root node name. */
    public static final String PLUGINS_NODE = "ametys-internal:plugins";
    
    /** Forms node name. */
    public static final String FORMS_NODE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":forms";
    
    /** Language property */
    public static final String LANGUAGE_PROPERTY = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":language";

    /** Site property */
    public static final String SITE_PROPERTY = RepositoryConstants.NAMESPACE_PREFIX + ":site";
    
    /** "ID" property name. */
    public static final String FORM_PROPERTY_ID = RepositoryConstants.NAMESPACE_PREFIX + ":id";
    
    /** "Label" property name. */
    public static final String FORM_PROPERTY_LABEL = RepositoryConstants.NAMESPACE_PREFIX + ":label";
    
    /** "Receipt field ID" property name. */
    public static final String FORM_PROPERTY_RECEIPT_FIELD_ID = RepositoryConstants.NAMESPACE_PREFIX + ":receipt-field-id";
    
    /** "Receipt field ID" property name. */
    public static final String FORM_PROPERTY_RECEIPT_FROM_ADDRESS = RepositoryConstants.NAMESPACE_PREFIX + ":receipt-from-address";
    
    /** "Receipt field ID" property name. */
    public static final String FORM_PROPERTY_RECEIPT_SUBJECT = RepositoryConstants.NAMESPACE_PREFIX + ":receipt-subject";
    
    /** "Receipt field ID" property name. */
    public static final String FORM_PROPERTY_RECEIPT_BODY = RepositoryConstants.NAMESPACE_PREFIX + ":receipt-body";
    
    /** The uuid of the page where to redirect to */
    public static final String FORM_PROPERTY_REDIRECT_TO = RepositoryConstants.NAMESPACE_PREFIX + ":redirect-to";
    
    /** "Emails" property name. */
    public static final String FORM_PROPERTY_EMAILS = RepositoryConstants.NAMESPACE_PREFIX + ":notification-emails";
    
    /** "Workflow name" property name. */
    public static final String FORM_PROPERTY_WORKFLOW_NAME = RepositoryConstants.NAMESPACE_PREFIX + ":workflow-name";
    
    /** "ID" field property name. */
    public static final String FIELD_PROPERTY_ID = RepositoryConstants.NAMESPACE_PREFIX + ":id";
    
    /** "Type" field property name. */
    public static final String FIELD_PROPERTY_TYPE = RepositoryConstants.NAMESPACE_PREFIX + ":type";
    
    /** "Name" field property name. */
    public static final String FIELD_PROPERTY_NAME = RepositoryConstants.NAMESPACE_PREFIX + ":name";
    
    /** "Label" field property name. */
    public static final String FIELD_PROPERTY_LABEL = RepositoryConstants.NAMESPACE_PREFIX + ":label";
    
    /** Field properties prefix. */
    public static final String FIELD_PROPERTY_PREFIX = RepositoryConstants.NAMESPACE_PREFIX + ":property-";

    /** "Limit" property name. */
    public static final String FORM_PROPERTY_LIMIT = RepositoryConstants.NAMESPACE_PREFIX + ":limit";
    
    /** "Remaining places" property name. */
    public static final String FORM_PROPERTY_REMAINING_PLACES = RepositoryConstants.NAMESPACE_PREFIX + ":remaining-places";
    
    /** "No remaining places" property name. */
    public static final String FORM_PROPERTY_NO_REMAINING_PLACES = RepositoryConstants.NAMESPACE_PREFIX + ":no-remaining-places";
    
    /** The JCR repository. */
    protected Repository _repository;
    
    /** The Site manager. */
    protected SiteManager _siteManager;
    
    /** The resolver for ametys objects */
    protected AmetysObjectResolver _resolver;
    
    /** The plugin name. */
    protected String _pluginName;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _repository = (Repository) serviceManager.lookup(AbstractRepository.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    /**
     * Get a form from the repository.
     * @param id the form ID.
     * @return the Form or null if no form with this ID exists.
     * @throws FormsException if an error occurs.
     */
    public Form getForm(String id) throws FormsException
    {
        return getForm(null, id);
    }
    
    /**
     * Get a form from the repository.
     * @param siteName the site name.
     * @param id the form ID.
     * @return the Form or null if no form with this ID exists.
     * @throws FormsException if an error occurs.
     */
    public Form getForm(String siteName, String id) throws FormsException
    {
        Session session = null;
        try
        {
            session = _repository.login();
            
            Form form = null;
            
            // Build the query
            String xpathQuery = "//element(*, ametys:content)";
            if (siteName != null)
            {
                xpathQuery += "[@" + SITE_PROPERTY + "='" + siteName + "']";
            }
            xpathQuery += "/" + FORMS_NODE + "/*[@" + FORM_PROPERTY_ID + " = '" + id + "']";
            
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            @SuppressWarnings("deprecation")
            Query query = queryManager.createQuery(xpathQuery, Query.XPATH);
            NodeIterator nodeIterator = query.execute().getNodes();
            
            if (nodeIterator.hasNext())
            {
                Node node = nodeIterator.nextNode();
                
                form = _extractForm(node);
            }
            
            return form;
        }
        catch (RepositoryException e)
        {
            throw new FormsException("Error executing the query to find the form of id " + id, e);
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
    }
    
    /**
     * Get the most recent frozen node that contain the form of the given id 
     * @param formId the id of the form
     * @return the list of frozen content nodes containing the form
     * @throws FormsException if an error occurs while retrieving the forms' frozen content nodes
     */
    public Node getMostRecentFormFrozenContent(String formId) throws FormsException
    {
        Session session = null;
        try
        {
            session = _repository.login();
            
            String xpathQuery = "//element(" + formId + ", nt:frozenNode)/../.. order by @" + RepositoryConstants.NAMESPACE_PREFIX + ":" + DefaultContent.METADATA_MODIFIED + " descending";
            
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            @SuppressWarnings("deprecation")
            Query query = queryManager.createQuery(xpathQuery, Query.XPATH);
            NodeIterator nodeIterator = query.execute().getNodes();
            
            if (nodeIterator.hasNext())
            {
                return (Node) nodeIterator.next();
            }
            
            return null;
        }
        catch (RepositoryException e)
        {
            getLogger().error("Error executing the query to find the frozen content nodes for the form '" + formId + "'.", e);
            return null;
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
    }
    
    /**
     * Get the forms in a specified content.
     * @param content the content.
     * @return the forms as a list.
     * @throws FormsException if an error occurs.
     */
    public List<Form> getForms(Content content) throws FormsException
    {
        Session session = null;
        try
        {
            List<Form> forms = new ArrayList<>();
            
            session = _repository.login();
            
            // FIXME API getNode should be VersionableAmetysObject
            if (content instanceof JCRAmetysObject)
            {
                Node contentNode = ((JCRAmetysObject) content).getNode();
                
                if (contentNode.hasNode(FORMS_NODE))
                {
                    Node formsNode = contentNode.getNode(FORMS_NODE);
                    
                    NodeIterator nodes = formsNode.getNodes();
                    
                    while (nodes.hasNext())
                    {
                        Node node = nodes.nextNode();
                        
                        Form form = _extractForm(node);
                        
                        if (form != null)
                        {
                            forms.add(form);
                        }
                    }
                }
            }
            return forms;
        }
        catch (RepositoryException e)
        {
            getLogger().error("Error getting forms for a content.", e);
            throw new FormsException("Error getting forms for a content.", e);
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
    }
    
    /**
     * Get all the contents containing at least one form of the given site with the given language
     * @param siteName the site name.
     * @param language the language
     * @return the forms' list or null if none was found
     * @throws FormsException if an error occurs.
     */
    public List<Node> getFormContentNodes(String siteName, String language) throws FormsException
    {
        List<Node> contentNodes = new ArrayList<> ();
        Session session = null;
        try
        {
            session = _repository.login();
            
            String xpathQuery = "//element(*, ametys:content)[@" + SITE_PROPERTY + " = '" + siteName + "' and @" + LANGUAGE_PROPERTY + " = '" + language + "']/" + FORMS_NODE;
            
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            @SuppressWarnings("deprecation")
            Query query = queryManager.createQuery(xpathQuery, Query.XPATH);
            NodeIterator nodeIterator = query.execute().getNodes();
            
            while (nodeIterator.hasNext())
            {
                Node formNode = nodeIterator.nextNode();
                contentNodes.add(formNode.getParent());
            }
            
            return contentNodes;
        }
        catch (RepositoryException e)
        {
            throw new FormsException("Error executing the query to find the forms of the site '" + siteName + "' and of language '" + language + "'.", e);
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
    }   
    
    /**
     * Get the content containing the form with the given id
     * @param formId the id of the form
     * @return the {@link Content} containing the form or <code>null</code> if not found
     * @throws FormsException if something goes wrong when either querying the form JCR node or finding its parent {@link Content} 
     */
    public Content getFormContent(String formId) throws FormsException
    {
        Session session = null;
        try
        {
            session = _repository.login();
            
            // Build the query
            String xpathQuery = "//element(*, ametys:content)/" + FORMS_NODE + "/*[@" + FORM_PROPERTY_ID + " = '" + formId + "']";
            
            // Execute
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            @SuppressWarnings("deprecation")
            Query query = queryManager.createQuery(xpathQuery, Query.XPATH);
            NodeIterator nodeIterator = query.execute().getNodes();
            
            if (nodeIterator.hasNext())
            {
                Node node = nodeIterator.nextNode().getParent().getParent();
                Content content = (Content) _resolver.resolve(node, false);
                return content;
            }
            
            return null;
        }
        catch (RepositoryException e)
        {
            throw new FormsException("Error executing the query to find the content containing the form of id " + formId, e);
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
        
    }
    
    
    /**
     * Extract all the form objects from a node
     * @param node the node
     * @return the forms list of this node
     * @throws FormsException if an error occurs
     * @throws RepositoryException if an error occurs when getting the properties of a node
     */
    public List<Form> getForms(Node node) throws FormsException, RepositoryException
    {
        List<Form> forms = new ArrayList<> ();
        try
        {
            if (node.hasNode(FORMS_NODE))
            {
                Node formsNode = node.getNode(FORMS_NODE);
                if (formsNode != null)
                {
                    NodeIterator formsNodeIterator = formsNode.getNodes();
                    while (formsNodeIterator.hasNext())
                    {
                        Node formNode = formsNodeIterator.nextNode();
                        Form form = _extractForm(formNode);
                        if (form != null)
                        {
                            forms.add(form);
                        }
                    }
                }
            }
            
            return forms;
        }
        catch (RepositoryException e)
        {
            throw new FormsException("Error executing the query to find the forms of the node '" + node.getName() + "' (" + node.getIdentifier() + ").", e);
        }
    }
    
    /**
     * Store the properties of a form in the repository.
     * @param siteName the site name.
     * @param form the form object.
     * @param content the form content.
     * @throws FormsException if an error occurs storing the form.
     */
    public void createForm(String siteName, Form form, Content content) throws FormsException
    {
        try
        {
            // FIXME API getNode should be VersionableAmetysObject
            if (content instanceof JCRAmetysObject)
            {
                Node contentNode = ((JCRAmetysObject) content).getNode();
                
                Node formsNode = _createOrGetFormsNode(contentNode);
                
                Node formNode = _storeForm(formsNode, form);
                
                _fillFormNode(formNode, form);
                
                for (Field field : form.getFields())
                {
                    Node fieldNode = _storeField(formNode, field);
                    _fillFieldNode(fieldNode, field);
                }
                
                contentNode.getSession().save();
            }
        }
        catch (RepositoryException e)
        {
            throw new FormsException("Repository exception while storing the form properties.", e);
        }
    }
    
    /**
     * Update the properties of a form in the repository.
     * @param siteName the site name.
     * @param form the form object.
     * @param content the form content.
     * @throws FormsException if an error occurs storing the form.
     */
    public void updateForm(String siteName, Form form, Content content) throws FormsException
    {
        Session session = null;
        try
        {
            session = _repository.login();
            
            String id = form.getId();
            
            // FIXME API getNode should be VersionableAmetysObject
            if (content instanceof JCRAmetysObject)
            {
                String xpathQuery = "//element(*, ametys:content)/" + FORMS_NODE + "/*[@" + FORM_PROPERTY_ID + " = '" + id + "']";
                
                QueryManager queryManager = session.getWorkspace().getQueryManager();
                @SuppressWarnings("deprecation")
                Query query = queryManager.createQuery(xpathQuery, Query.XPATH);
                NodeIterator nodeIterator = query.execute().getNodes();
                
                if (nodeIterator.hasNext())
                {
                    Node formNode = nodeIterator.nextNode();
                    
                    _fillFormNode(formNode, form);
                    
                    _updateFields(form, formNode);
                    
                    if (session.hasPendingChanges())
                    {
                        session.save();
                    }
                }
            }
        }
        catch (RepositoryException e)
        {
            throw new FormsException("Repository exception while storing the form properties.", e);
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
    }
    
    /**
     * Get the value for display
     * @param field The field
     * @param value The value
     * @return The value to display
     */
    public String getDisplayValue (Field field, String value)
    {
        Map<String, String> properties = field.getProperties();
        for (String key : properties.keySet())
        {
            Matcher matcher = OPTION_VALUE_PATTERN.matcher(key);
            if (matcher.matches())
            {
                // Trim value since some rendering (including default rendering) add many leading spaces
                if (StringUtils.trim(value).equals(StringUtils.trim(properties.get(key))))
                {
                    String index = matcher.group(1);
                    if (properties.containsKey("option-" + index + "-label"))
                    {
                        return properties.get("option-" + index + "-label");
                    }
                }
            }
        }
        return value;
    }
    
    /**
     * Extracts a form from a JCR Node.
     * @param formNode the form node.
     * @return the Form object.
     * @throws RepositoryException if a repository error occurs.
     */
    protected Form _extractForm(Node formNode) throws RepositoryException
    {
        Form form = null;
        
        String id = _getSingleProperty(formNode, FORM_PROPERTY_ID, "");
        if (!StringUtils.isEmpty(id))
        {
            String label = _getSingleProperty(formNode, FORM_PROPERTY_LABEL, "");
            String receiptFieldId = _getSingleProperty(formNode, FORM_PROPERTY_RECEIPT_FIELD_ID, "");
            String receiptFieldBody = _getSingleProperty(formNode, FORM_PROPERTY_RECEIPT_BODY, "");
            String receiptFieldSubject = _getSingleProperty(formNode, FORM_PROPERTY_RECEIPT_SUBJECT, "");
            String receiptFieldFromAddress = _getSingleProperty(formNode, FORM_PROPERTY_RECEIPT_FROM_ADDRESS, "");
            String redirectTo = _getSingleProperty(formNode, FORM_PROPERTY_REDIRECT_TO, "");
            Collection<String> emails = _getMultipleProperty(formNode, FORM_PROPERTY_EMAILS);
            String workflowName = _getSingleProperty(formNode, FORM_PROPERTY_WORKFLOW_NAME, "");
            String limit = _getSingleProperty(formNode, FORM_PROPERTY_LIMIT, "");
            String remainingPlaces = _getSingleProperty(formNode, FORM_PROPERTY_REMAINING_PLACES, "");
            String noRemainingPlaces = _getSingleProperty(formNode, FORM_PROPERTY_NO_REMAINING_PLACES, "");
            
            form = new Form();
            
            Content content = _resolver.resolve(formNode.getParent().getParent(), false);
            form.setContentId(content.getId());
            
            form.setId(id);
            form.setLabel(label);
            form.setReceiptFieldId(receiptFieldId);
            form.setReceiptFieldBody(receiptFieldBody);
            form.setReceiptFieldSubject(receiptFieldSubject);
            form.setReceiptFieldFromAddress(receiptFieldFromAddress);
            form.setNotificationEmails(new HashSet<>(emails));
            form.setRedirectTo(redirectTo);
            form.setWorkflowName(workflowName);
            form.setLimit(limit);
            form.setRemainingPlaces(remainingPlaces);
            form.setNoRemainingPlaces(noRemainingPlaces);
            
            _extractFields(formNode, form);
        }
        
        return form;
    }
    
    /**
     * Extracts a form from a JCR Node.
     * @param formNode the form node.
     * @param form the form object.
     * @throws RepositoryException if a repository error occurs.
     */
    protected void _extractFields(Node formNode, Form form) throws RepositoryException
    {
        NodeIterator nodes = formNode.getNodes();
        while (nodes.hasNext())
        {
            Node node = nodes.nextNode();
            
            Field field = _extractField(node);
            
            form.getFields().add(field);
        }
    }
    
    /**
     * Extracts a field from a JCR Node.
     * @param fieldNode the field node.
     * @return the Field object.
     * @throws RepositoryException if a repository error occurs.
     */
    protected Field _extractField(Node fieldNode) throws RepositoryException
    {
        Field field = null;
        
        String id = _getSingleProperty(fieldNode, FIELD_PROPERTY_ID, "");
        String type = _getSingleProperty(fieldNode, FIELD_PROPERTY_TYPE, "");
        
        // TODO Try/catch in case of enum name not found.
        FieldType fieldType = FieldType.valueOf(type);
        
        if (!StringUtils.isEmpty(id) && !StringUtils.isEmpty(type))
        {
            String name = _getSingleProperty(fieldNode, FIELD_PROPERTY_NAME, "");
            String label = _getSingleProperty(fieldNode, FIELD_PROPERTY_LABEL, "");
            Map<String, String> properties = _getFieldProperties(fieldNode);
            
            field = new Field(fieldType);
            
            field.setId(id);
            field.setName(name);
            field.setLabel(label);
            field.setProperties(properties);
        }
        
        return field;
    }
    
    /**
     * Persist the form in a repository node.
     * @param contentNode the content node in which the form is to be stored.
     * @param form the form object to persist.
     * @return the newly created form node.
     * @throws RepositoryException if a repository error occurs while filling the node.
     */
    protected Node _storeForm(Node contentNode, Form form) throws RepositoryException
    {
        String name = form.getId();
        if (StringUtils.isBlank(name))
        {
            name = "form";
        }
        
        String nodeName = NameHelper.filterName(name);
        String notExistingNodeName = _getNotExistingNodeName(contentNode, nodeName);
        
        Node formNode = contentNode.addNode(notExistingNodeName);
        formNode.addMixin(JcrConstants.MIX_REFERENCEABLE);
        
        return formNode;
    }
    
    /**
     * Fill a form node.
     * @param formNode the form node.
     * @param form the form object.
     * @throws RepositoryException if a repository error occurs while filling the node.
     * @throws FormsException if a forms error occurs while filling the node.
     */
    protected void _fillFormNode(Node formNode, Form form) throws RepositoryException, FormsException
    {
        Set<String> emails = form.getNotificationEmails();
        String[] emailArray = emails.toArray(new String[emails.size()]);
        
        formNode.setProperty(FORM_PROPERTY_ID, form.getId());
        formNode.setProperty(FORM_PROPERTY_LABEL, form.getLabel());
        formNode.setProperty(FORM_PROPERTY_RECEIPT_FIELD_ID, form.getReceiptFieldId());
        formNode.setProperty(FORM_PROPERTY_RECEIPT_BODY, form.getReceiptFieldBody());
        formNode.setProperty(FORM_PROPERTY_RECEIPT_SUBJECT, form.getReceiptFieldSubject());
        formNode.setProperty(FORM_PROPERTY_RECEIPT_FROM_ADDRESS, form.getReceiptFieldFromAddress());
        formNode.setProperty(FORM_PROPERTY_EMAILS, emailArray);
        formNode.setProperty(FORM_PROPERTY_REDIRECT_TO, form.getRedirectTo());
        formNode.setProperty(FORM_PROPERTY_WORKFLOW_NAME, form.getWorkflowName());
        formNode.setProperty(FORM_PROPERTY_REMAINING_PLACES, form.getRemainingPlaces());
        formNode.setProperty(FORM_PROPERTY_LIMIT, form.getLimit());
        formNode.setProperty(FORM_PROPERTY_NO_REMAINING_PLACES, form.getNoRemainingPlaces());
    }
    
    /**
     * Store a field node.
     * @param formNode the form node.
     * @param field the field.
     * @return the newly created field node.
     * @throws RepositoryException if a repository error occurs while filling the node.
     * @throws FormsException if a forms error occurs while filling the node.
     */
    protected Node _storeField(Node formNode, Field field) throws RepositoryException, FormsException
    {
        String name = field.getId();
        if (StringUtils.isBlank(name))
        {
            name = "field";
        }
        
        String nodeName = NameHelper.filterName(name);
        String notExistingNodeName = _getNotExistingNodeName(formNode, nodeName);
        
        Node fieldNode = formNode.addNode(notExistingNodeName);
        
        fieldNode.addMixin(JcrConstants.MIX_REFERENCEABLE);
        
        return fieldNode;
    }
    
    /**
     * Fill a field node.
     * @param fieldNode the field node.
     * @param field the field object.
     * @throws RepositoryException if a repository error occurs while filling the node.
     * @throws FormsException if a forms error occurs while filling the node.
     */
    protected void _fillFieldNode(Node fieldNode, Field field) throws RepositoryException, FormsException
    {
        fieldNode.setProperty(FIELD_PROPERTY_ID, field.getId());
        fieldNode.setProperty(FIELD_PROPERTY_TYPE, field.getType().toString());
        fieldNode.setProperty(FIELD_PROPERTY_NAME, field.getName());
        fieldNode.setProperty(FIELD_PROPERTY_LABEL, field.getLabel());
        
        Map<String, String> fieldProperties = field.getProperties();
        for (String propertyName : fieldProperties.keySet())
        {
            String value = fieldProperties.get(propertyName);
            if (value != null)
            {
                String name = FIELD_PROPERTY_PREFIX + propertyName;
                fieldNode.setProperty(name, value);
            }
        }
    }    
    /**
     * Update the field nodes of a form.
     * @param form the new form object.
     * @param formNode the node of the form to update.
     * @throws RepositoryException if a repository error occurs while updating the fields.
     * @throws FormsException if a forms error occurs while updating the fields.
     */
    protected void _updateFields(Form form, Node formNode) throws RepositoryException, FormsException
    {
        Map<String, Field> fieldMap = form.getFieldMap();
        
        NodeIterator fieldNodes = formNode.getNodes();
        
        while (fieldNodes.hasNext())
        {
            Node fieldNode = fieldNodes.nextNode();
            if (fieldNode.hasProperty(FIELD_PROPERTY_ID))
            {
                String fieldId = fieldNode.getProperty(FIELD_PROPERTY_ID).getString();
                
                // The field still exist in the new form : update its properties.
                if (fieldMap.containsKey(fieldId))
                {
                    Field field = fieldMap.get(fieldId);
                    
                    _fillFieldNode(fieldNode, field);
                    
                    fieldMap.remove(fieldId);
                }
                else
                {
                    // The field doesn't exist anymore : delete it.
                    fieldNode.remove();
                }
            }
        }
        
        // Now the map contains the field to add.
        for (Map.Entry<String, Field> entry : fieldMap.entrySet())
        {
            Field newField = entry.getValue();
            Node fieldNode = _storeField(formNode, newField);
            _fillFieldNode(fieldNode, newField);
        }
    }

    /**
     * Get a name for a node which doesn't already exist in this node.
     * @param container the container node.
     * @param baseName the base wanted node name.
     * @return the name, free to be taken.
     * @throws RepositoryException if a repository error occurs.
     */
    protected String _getNotExistingNodeName(Node container, String baseName) throws RepositoryException
    {
        String name = baseName;
        
        int index = 2;
        while (container.hasNode(name))
        {
            name = baseName + index;
            index++;
        }
        
        return name;
    }
    
    /**
     * Get a single property value.
     * @param node the JCR node.
     * @param propertyName the name of the property to get.
     * @param defaultValue the default value if the property does not exist.
     * @return the single property value.
     * @throws RepositoryException if a repository error occurs.
     */
    protected String _getSingleProperty(Node node, String propertyName, String defaultValue) throws RepositoryException
    {
        String value = defaultValue;
        
        if (node.hasProperty(propertyName))
        {
            value = node.getProperty(propertyName).getString();
        }
        
        return value;
    }
    
    /**
     * Get the values of a string array property.
     * @param node the node.
     * @param propertyName the name of the property to get.
     * @return the values.
     * @throws RepositoryException if a repository error occurs.
     */
    protected Collection<String> _getMultipleProperty(Node node, String propertyName) throws RepositoryException
    {
        List<String> values = new ArrayList<>();
        
        if (node.hasProperty(propertyName))
        {
            Value[] propertyValues = node.getProperty(propertyName).getValues();
            for (Value value : propertyValues)
            {
                values.add(value.getString());
            }
        }
        
        return values;
    }
    
    /**
     * Get additional configuration from properties.
     * @param node the JCR node.
     * @return the additional configuration as a Map.
     * @throws RepositoryException if a repository error occurs.
     */
    protected Map<String, String> _getFieldProperties(Node node) throws RepositoryException
    {
        Map<String, String> values = new HashMap<>();
        
        PropertyIterator propertyIt = node.getProperties(FIELD_PROPERTY_PREFIX + "*");
        while (propertyIt.hasNext())
        {
            Property property = propertyIt.nextProperty();
            String propName = property.getName();
            String name = propName.substring(FIELD_PROPERTY_PREFIX.length(), propName.length());
            String value = property.getString();
            
            values.put(name, value);
        }
        
        return values;
    }
    
    /**
     * Remove a form
     * @param form The form to remove
     * @param content The content holding the form
     * @throws FormsException of an exception occurs when manipulating the forms' repository nodes
     */
    public void remove(Form form, Content content) throws FormsException
    {
        Session session = null;
        try
        {
            session = _repository.login();
            
            String id = form.getId();
            
            String xpathQuery = "//element(*, ametys:content)[@jcr:uuid = '" + ((JCRAmetysObject) content).getNode().getIdentifier() + "']/" + FORMS_NODE + "/*[@" + FORM_PROPERTY_ID + " = '" + id + "']";
            
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            @SuppressWarnings("deprecation")
            Query query = queryManager.createQuery(xpathQuery, Query.XPATH);
            NodeIterator nodeIterator = query.execute().getNodes();
            
            if (nodeIterator.hasNext())
            {
                Node formNode = nodeIterator.nextNode();
                formNode.remove();
                
                if (session.hasPendingChanges())
                {
                    session.save();
                }
            }
        }
        catch (RepositoryException e)
        {
            throw new FormsException("Repository exception while storing the form properties.", e);
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
    }
    
    /**
     * Get or create the forms node in a content node.
     * @param baseNode the content base node.
     * @return the forms node.
     * @throws RepositoryException if an error occurs.
     */
    protected Node _createOrGetFormsNode(Node baseNode) throws RepositoryException
    {
        Node node = null;
        if (baseNode.hasNode(FORMS_NODE))
        {
            node = baseNode.getNode(FORMS_NODE);
        }
        else
        {
            node = baseNode.addNode(FORMS_NODE, "nt:unstructured");
        }
        return node;
    }
}
