/*
 * Copyright 2021 Anyware Services
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.processing;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.cocoon.servlet.multipart.PartOnDisk;
import org.apache.cocoon.servlet.multipart.RejectedPart;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.core.captcha.CaptchaHelper;
import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.datasource.dbtype.SQLDatabaseTypeExtensionPoint;
import org.ametys.core.group.InvalidModificationException;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.forms.content.Field;
import org.ametys.plugins.forms.content.Field.FieldType;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.data.FieldValue;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.DbTypeHelper;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.workflow.store.JdbcWorkflowStore;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.URIPrefixHandler;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.WorkflowException;

import jakarta.mail.MessagingException;

/**
 * Helper that processes the user submitted data on a form.
 */
public class ProcessFormHelper extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** The Avalon role */
    public static final String ROLE = ProcessFormHelper.class.getName();
    
    /** The form ID parameter. */
    public static final String PARAM_FORM_ID = "ametys-form-id";

    /** The content ID parameter. */
    protected static final String PARAM_CONTENT_ID = "ametys-content-id";

    /** The integer validation pattern. */
    protected static final Pattern _INT_PATTERN = Pattern.compile(FormValidators.getIntegerPattern());

    /** The integer validation pattern. */
    protected static final Pattern _FLOAT_PATTERN = Pattern.compile(FormValidators.getFloatPattern());

    /** The integer validation pattern. */
    protected static final Pattern _DATE_PATTERN = Pattern.compile(FormValidators.getDatePattern());

    /** The integer validation pattern. */
    protected static final Pattern _TIME_PATTERN = Pattern.compile(FormValidators.getTimePattern());

    /** The integer validation pattern. */
    protected static final Pattern _DATETIME_PATTERN = Pattern.compile(FormValidators.getDateTimePattern());

    /** The email validation pattern. */
    protected static final Pattern _EMAIL_PATTERN = SendMailHelper.EMAIL_VALIDATION;

    /** The phone validation pattern. */
    protected static final Pattern _PHONE_PATTERN = Pattern.compile(FormValidators.getPhonePattern());

    /** The date format pattern. */
    protected static final DateFormat _DATE_FORMAT = new SimpleDateFormat(FormValidators.getDateFormat());

    /** The time format pattern. */
    protected static final DateFormat _TIME_FORMAT = new SimpleDateFormat(FormValidators.getTimeFormat());

    /** The date and time format pattern. */
    protected static final DateFormat _DATETIME_FORMAT = new SimpleDateFormat(FormValidators.getDateTimeFormat());
    
    private static final String __FORM_ENTRY_PATTERN = "${form}";
    
    private static Pattern __OPTION_INDEX = Pattern.compile("^option-([0-9]+)-value$");
    
    /** Form properties manager. */
    protected FormPropertiesManager _formPropertiesManager;

    /** Form table manager. */
    protected FormTableManager _formTableManager;

    /** The source resolver. */
    protected SourceResolver _sourceResolver;

    /** The ametys object resolver */
    protected AmetysObjectResolver _ametysObjectResolver;

    /** The site manager. */
    protected SiteManager _siteManager;

    /** The plugin name. */
    protected String _pluginName;
    
    /** The URI prefix handler */
    protected URIPrefixHandler _prefixHandler;
    
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    /** The workflow helper component */
    protected WorkflowHelper _workflowHelper;
    
    /** The SQLDatabaseTypeExtensionPoint instance */
    protected SQLDatabaseTypeExtensionPoint _sqlDatabaseTypeExtensionPoint;
    
    /** The context */
    protected Context _context;

    /** Rights manager */
    protected RightManager _rightManager;
    
    /** current user provider */
    protected CurrentUserProvider _currentUserProvied;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _formPropertiesManager = (FormPropertiesManager) serviceManager.lookup(FormPropertiesManager.ROLE);
        _formTableManager = (FormTableManager) serviceManager.lookup(FormTableManager.ROLE);
        _sourceResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _ametysObjectResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _sqlDatabaseTypeExtensionPoint = (SQLDatabaseTypeExtensionPoint) serviceManager.lookup(SQLDatabaseTypeExtensionPoint.ROLE);
        _prefixHandler = (URIPrefixHandler) serviceManager.lookup(URIPrefixHandler.ROLE);
        _workflowProvider = (WorkflowProvider) serviceManager.lookup(WorkflowProvider.ROLE);
        _workflowHelper = (WorkflowHelper) serviceManager.lookup(WorkflowHelper.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _currentUserProvied = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
    }

    /**
     * Process form
     * @param form the form
     * @param site the site
     * @param pluginName the plugin name
     * @return the form informations
     * @throws Exception if an error occurred
     */
    public FormInformations processForm(Form form, Site site, String pluginName) throws Exception
    {
        FormInformations formInformations = new FormInformations();
        
        Map objectModel = ContextHelper.getObjectModel(_context);
        Request request = ObjectModelHelper.getRequest(objectModel);
        _pluginName = pluginName;
        
        FormErrors errors = new FormErrors(form, new LinkedHashMap<>());
        formInformations.setFormErrors(errors);
        
        Map<String, FieldValue> input = _getInput(form, request, errors);

        _validateInput(form, input, errors, request);
        
        int totalSubmissions = 0;

        if (errors.hasErrors())
        {
            return formInformations;
        }
         
        String entryId = null;
        if (!StringUtils.isEmpty(form.getLimit()))
        {
            synchronized (this)
            { 
                totalSubmissions = _formTableManager.getTotalSubmissions(form.getId());

                if (Integer.parseInt(form.getLimit()) <= totalSubmissions)
                {
                    errors.setLimitReached(true);
                    return formInformations;
                }
                entryId = _insertInput(form, input, objectModel);
                if (StringUtils.isBlank(entryId))
                {
                    errors.setInsertionFailed(true);
                    return formInformations;
                }
            }
            
        }
        else
        {
            entryId = _insertInput(form, input, objectModel);
            if (StringUtils.isBlank(entryId))
            {
                errors.setInsertionFailed(true);
                return formInformations;
            }
        }
        
        _sendEmails(form, input, site, totalSubmissions);

        formInformations.setRedirection(_getFormRedirection(form, entryId));
        
        return formInformations;
    }
    
    /**
     * Get the form redirection
     * @param form the form 
     * @param entryId the entryId
     * @return the form redirection url
     */
    protected String _getFormRedirection(Form form, String entryId)
    {
        if (StringUtils.isNotEmpty(form.getRedirectTo()))
        {
            String pageId = form.getRedirectTo();
            try
            {
                Page page = _ametysObjectResolver.resolveById(pageId);
                return _prefixHandler.getAbsoluteUriPrefix(page.getSiteName()) + "/" + page.getSitemapName() + "/" + page.getPathInSitemap() + ".html";
                
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn("The form '" + form.getId() + "' wants to redirect to the unexisting page '" + pageId + "'. Redirecting to default page.", e);
            }
        }
        
        return null;
    }
    
    /**
     * Get the user input.
     * 
     * @param form the Form object.
     * @param request the user request.
     * @param errors the input errors.
     * @return the user data as a Map of column name -&gt; column entry.
     */
    protected Map<String, FieldValue> _getInput(Form form, Request request, FormErrors errors)
    {
        Map<String, FieldValue> entries = new LinkedHashMap<>();

        // For each field declared in the form,
        for (Field field : form.getFields())
        {
            final String id = field.getId();
            final String name = field.getName();

            FieldValue entry = null;

            switch (field.getType())
            {
                case TEXT:
                case COST:
                case HIDDEN:
                case PASSWORD:
                    String sValue = (String) request.get(name);
                    entry = new FieldValue(id, Types.VARCHAR, sValue, field);
                    break;
                case SELECT:
                    String[] values = request.getParameterValues(name);
                    sValue = values == null ? "" : StringUtils.join(values, "\n");
                    entry = new FieldValue(id, Types.VARCHAR, sValue, field);
                    break;
                case TEXTAREA:
                    sValue = (String) request.get(name);
                    entry = new FieldValue(id, Types.LONGVARCHAR, sValue, field);
                    break;
                case RADIO:
                    if (!entries.containsKey(name))
                    {
                        sValue = (String) request.get(name);
                        entry = new FieldValue(name, Types.VARCHAR, sValue, field);
                    }
                    else
                    {
                        // The value exists, clone it, concatenating the label.
                        if (StringUtils.isNotEmpty(field.getLabel()))
                        {
                            Field radioField = entries.get(name).getField();
                            Field dummyField = new Field(radioField.getId(), radioField.getType(), radioField.getName(), radioField.getLabel() + "/" + field.getLabel(),
                                    radioField.getProperties());
                            entries.get(name).setField(dummyField);
                        }
                    }
                    break;
                case CHECKBOX:
                    boolean bValue = request.get(name) != null;
                    entry = new FieldValue(id, Types.BOOLEAN, bValue, field);
                    break;
                case FILE:
                    entry = _getFileEntry(request, field, id, name, errors);
                    break;
                case CAPTCHA:
                    final String formId = request.getParameter(PARAM_FORM_ID);
                    final String contentId = request.getParameter(PARAM_CONTENT_ID);

                    final String encodedName = contentId + "%20" + formId + "%20" + field.getId();

                    final String captchaValue = request.getParameter(encodedName);
                    final String captchaKey = request.getParameter(encodedName + "-key");

                    entry = new FieldValue(id, Types.OTHER, new String[] {captchaValue, captchaKey}, field);
                    break;
                default:
                    break;
            }

            if (entry != null)
            {
                entries.put(entry.getColumnName(), entry);
            }
        }

        return entries;
    }

    /**
     * Get a file entry from the request.
     * 
     * @param request the user request.
     * @param field the field.
     * @param id the entry ID.
     * @param name the field name.
     * @param errors the form errors.
     * @return the file entry.
     */
    protected FieldValue _getFileEntry(Request request, Field field, String id, String name, FormErrors errors)
    {
        FieldValue entry = null;

        final String keyPrefix = "PLUGINS_FORMS_FORMS_RENDER_ERROR_FILE_";

        Part part = (Part) request.get(name);
        if (part instanceof RejectedPart rejectedPart)
        {
            if (rejectedPart.getMaxContentLength() == 0)
            {
                errors.addError(field.getId(), new I18nizableText("plugin." + _pluginName, keyPrefix + "INFECTED"));
            }
            else
            {
                errors.addError(field.getId(), new I18nizableText("plugin." + _pluginName, keyPrefix + "REJECTED"));
            }
        }
        else
        {
            PartOnDisk uploadedFilePart = (PartOnDisk) part;
            if (uploadedFilePart != null)
            {
                entry = new FieldValue(id, Types.BLOB, uploadedFilePart.getFile(), field);
            }
            else
            {
                entry = new FieldValue(id, Types.BLOB, null, field);
            }
        }

        return entry;
    }

    /**
     * Insert the user submission in the database.
     * 
     * @param form the Form object.
     * @param input the user input.
     * @param objectModel The object model
     * @return the entry id if the insertion has succeed, null in case of error
     * @throws WorkflowException if an exception occurs while initializing a workflow instance for a form entry
     * @throws InvalidModificationException If an error occurs
     */
    protected String _insertInput(Form form, Map<String, FieldValue> input, Map objectModel) throws WorkflowException, InvalidModificationException
    {
        boolean success = true;
        String entryId = null;
        
        final String tableName = FormTableManager.TABLE_PREFIX + form.getId();

        Connection connection = null;
        PreparedStatement stmt = null;
        
        try
        {
            String dataSourceId = Config.getInstance().getValue(FormTableManager.FORMS_POOL_CONFIG_PARAM);
            connection = ConnectionHelper.getConnection(dataSourceId);
            
            String dbType = ConnectionHelper.getDatabaseType(connection);
            
            List<String> columns = new ArrayList<>();
            List<String> values = new ArrayList<>();

            if (DbTypeHelper.insertIdentity(dbType))
            {
                columns.add("id");
                
                if (ConnectionHelper.DATABASE_ORACLE.equals(dbType))
                {
                    values.add("seq_" + form.getId() + ".nextval");
                }
                else
                {
                    values.add("?");
                }
            }
            
            // creation date
            columns.add(FormTableManager.CREATION_DATE_FIELD);
            values.add("?");

            // login
            columns.add(FormTableManager.LOGIN_FIELD);
            values.add("?");

            // populationId
            columns.add(FormTableManager.POPULATION_ID_FIELD);
            values.add("?");
            
            Iterator<FieldValue> entries = _getEntriesToInsert(input.values()).iterator();
            while (entries.hasNext())
            {
                FieldValue entry = entries.next();
                String colName = entry.getColumnName();
                columns.add(_sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, colName));
                values.add("?");

                if (entry.getType() == Types.BLOB)
                {
                    String fileNameColumn = colName + FormTableManager.FILE_NAME_COLUMN_SUFFIX;
                    String normalizedName = DbTypeHelper.normalizeName(dbType, fileNameColumn);
                    
                    columns.add(_sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, normalizedName));
                    values.add("?");
                }
            }
            
            if (_formTableManager.hasWorkflowIdColumn(form.getId()))
            {
                // Ensure compatibility with form entries that remained without workflow
                columns.add(_sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, FormTableManager.WORKFLOW_ID_FIELD));
                values.add("?");
            }

            String sql = new StringBuilder()
                .append("INSERT INTO ")
                .append(_sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, tableName))
                .append(" (")
                .append(StringUtils.join(columns, ", "))
                .append(") VALUES (")
                .append(StringUtils.join(values, ", "))
                .append(")")
                .toString();

            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Inserting a user submission in the database :\n" + sql);
            }

            stmt = connection.prepareStatement(sql);

            _setParameters(form, input, stmt, dbType);
            
            stmt.executeUpdate();
            
            ConnectionHelper.cleanup(stmt);
            
            entryId = _getEntryId(connection, dbType, tableName);
            
            if (_formTableManager.hasWorkflowIdColumn(form.getId()))
            {
                success = _createWorkflow(form, objectModel, entryId);
            }
        }
        catch (SQLException e)
        {
            getLogger().error("Error inserting submission data.", e);
            success = false;
        }
        finally
        {
            ConnectionHelper.cleanup(stmt);
            ConnectionHelper.cleanup(connection);
        }
        
        return success ? entryId : null;
    }

    private String _getEntryId(Connection connection, String dbType, String tableName) throws SQLException, InvalidModificationException
    {
        String id = null;
        if (ConnectionHelper.DATABASE_MYSQL.equals(dbType))
        {
            try (PreparedStatement stmt = connection.prepareStatement("SELECT id FROM " + _sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, tableName) + " WHERE id = last_insert_id()");
                 ResultSet rs = stmt.executeQuery())
            {
                if (rs.next())
                {
                    id = rs.getString("id");
                }
                else
                {
                    if (connection.getAutoCommit())
                    {
                        throw new InvalidModificationException("Cannot retrieve inserted group. Group was created but listeners not called : base may be inconsistant");
                    }
                    else
                    {
                        connection.rollback();
                        throw new InvalidModificationException("Cannot retrieve inserted group. Rolling back");
                    }
                }
            }
        }
        else if (ConnectionHelper.DATABASE_DERBY.equals(dbType))
        {
            try (PreparedStatement stmt = connection.prepareStatement("VALUES IDENTITY_VAL_LOCAL ()"); 
                 ResultSet rs = stmt.executeQuery())
            {
                if (rs.next())
                {
                    id = rs.getString(1);
                }
            }
        }
        else if (ConnectionHelper.DATABASE_HSQLDB.equals(dbType))
        {
            
            try (PreparedStatement stmt = connection.prepareStatement("CALL IDENTITY ()");
                 ResultSet rs = stmt.executeQuery())
            {
                if (rs.next())
                {
                    id = rs.getString(1);
                }
            }
        }
        else if (ConnectionHelper.DATABASE_POSTGRES.equals(dbType))
        {
            try (PreparedStatement stmt = connection.prepareStatement("SELECT currval('groups_id_seq')");
                 ResultSet rs = stmt.executeQuery())
            {
                if (rs.next())
                {
                    id = rs.getString(1);
                }
            }
        }
        
        return id;
    }
    
    private boolean _createWorkflow(Form form, Map objectModel, String entryId) throws InvalidActionException
    {
        boolean success = true;
        if (entryId != null)
        {
            // create workflow with entry id
            Workflow workflow = _workflowProvider.getExternalWorkflow(JdbcWorkflowStore.ROLE);
            
            String workflowName = form.getWorkflowName();
            int initialActionId = _workflowHelper.getInitialAction(workflowName); 
            
            Map<String, Object> inputs = new HashMap<>();
            inputs.put("formId", form.getId());
            inputs.put("entryId", entryId);
            inputs.put(ObjectModelHelper.PARENT_CONTEXT, ObjectModelHelper.getContext(objectModel));
            inputs.put(ObjectModelHelper.REQUEST_OBJECT, ObjectModelHelper.getRequest(objectModel));
            
            try
            {
                long workflowInstanceId = workflow.initialize(form.getWorkflowName(), initialActionId, inputs);
                // insert workflow id in db
                success = _updateWorkflowId(form.getId(), entryId, workflowInstanceId);
            }
            catch (Exception e) 
            {
                getLogger().error("Error inserting submission data.", e);
                _removeFormEntry(form.getId(), entryId);
                success = false;
            }
        }
        
        return success;
    }

    private boolean _removeFormEntry(String formId, String entryId)
    {
        boolean success = true;
        
        final String tableName = FormTableManager.TABLE_PREFIX + formId;

        Connection connection = null;
        PreparedStatement stmt = null;
        
        try
        {
            String dataSourceId = Config.getInstance().getValue(FormTableManager.FORMS_POOL_CONFIG_PARAM);
            connection = ConnectionHelper.getConnection(dataSourceId);
            
            String dbType = ConnectionHelper.getDatabaseType(connection);

            StringBuilder sql = new StringBuilder();

            sql.append("DELETE FROM ").append(_sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, tableName))
            .append(" WHERE id = ?");

            stmt = connection.prepareStatement(sql.toString());
            stmt.setString(1, entryId);
            
            stmt.executeUpdate();
        }
        catch (SQLException e)
        {
            getLogger().error("Error inserting submission data.", e);
            success = false;
        }
        finally
        {
            ConnectionHelper.cleanup(stmt);
            ConnectionHelper.cleanup(connection);
        }
        
        return success;
    }

    private boolean _updateWorkflowId(String formId, String entryId, long workflowId)
    {
        boolean success = true;
        
        final String tableName = FormTableManager.TABLE_PREFIX + formId;

        Connection connection = null;
        PreparedStatement stmt = null;
        
        try
        {
            String dataSourceId = Config.getInstance().getValue(FormTableManager.FORMS_POOL_CONFIG_PARAM);
            connection = ConnectionHelper.getConnection(dataSourceId);
            
            String dbType = ConnectionHelper.getDatabaseType(connection);

            StringBuilder sql = new StringBuilder();

            sql.append("UPDATE ").append(_sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, tableName))
            .append(" SET ").append(_sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, FormTableManager.WORKFLOW_ID_FIELD)).append(" = ?")
            .append(" WHERE id = ?");

            stmt = connection.prepareStatement(sql.toString());
            stmt.setLong(1, workflowId);
            stmt.setString(2, entryId);
            
            stmt.executeUpdate();
        }
        catch (SQLException e)
        {
            getLogger().error("Error inserting submission data.", e);
            success = false;
        }
        finally
        {
            ConnectionHelper.cleanup(stmt);
            ConnectionHelper.cleanup(connection);
        }
        
        return success;
    }

    /**
     * Set the parameters into the prepared statement.
     * 
     * @param form the form.
     * @param input the user input.
     * @param stmt the prepared statement.
     * @param dbType the database type.
     * @throws SQLException if a SQL error occurs.
     * @throws WorkflowException if an exception occurs while initializing a workflow instance for a form entry
     */
    protected void _setParameters(Form form, Map<String, FieldValue> input, PreparedStatement stmt, String dbType) throws SQLException, WorkflowException
    {
        // First two are id (auto-increment) and creation date
        int index = 1;
        if (DbTypeHelper.insertIdentity(dbType) && !ConnectionHelper.DATABASE_ORACLE.equals(dbType))
        {
            DbTypeHelper.setIdentity(stmt, index, dbType);
            index++;
        }

        stmt.setTimestamp(index, new Timestamp(System.currentTimeMillis()));
        index++;

        UserIdentity user = _currentUserProvied.getUser();
        if (user != null)
        {
            stmt.setString(index, user.getLogin());
            index++;
            
            stmt.setString(index, user.getPopulationId());
            index++;
        }
        else
        {
            stmt.setNull(index, Types.VARCHAR);
            index++;
            stmt.setNull(index, Types.VARCHAR);
            index++;
        }

        // Then set all the entries to be stored into the prepared statement.
        Collection<FieldValue> entries = _getEntriesToInsert(input.values());
        for (FieldValue entry : entries)
        {
            if (entry.getField().getType() == FieldType.COST)
            {
                stmt.setString(index, _computeCost(input));
            }
            else if (entry.getValue() == null)
            {
                stmt.setNull(index, entry.getType());
                if (entry.getType() == Types.BLOB)
                {
                    index++;
                    stmt.setNull(index, Types.VARCHAR);
                }
            }
            else if (entry.getType() == Types.BLOB && entry.getValue() instanceof File)
            {
                File file = (File) entry.getValue();

                try
                {
                    _sqlDatabaseTypeExtensionPoint.setBlob(dbType, stmt, index, new FileInputStream(file), file.length());
                    index++;

                    stmt.setString(index, file.getName());
                }
                catch (IOException e)
                {
                    // Should never happen, as it was checked before.
                    getLogger().error("Can't read uploaded file.", e);
                }
            }
            else if (entry.getType() == Types.BOOLEAN && entry.getValue() instanceof Boolean)
            {
                stmt.setInt(index, Boolean.TRUE.equals(entry.getValue()) ? 1 : 0);
            }
            else
            {
                stmt.setObject(index, entry.getValue(), entry.getType());
            }
            index++;
        }
        
        // Ensure compatibility with form entries that remained without workflow
        if (_formTableManager.hasWorkflowIdColumn(form.getId()))
        {
            stmt.setLong(index, -1);
        }
    }

    private String _computeCost(Map<String, FieldValue> input)
    {
        double v = input.values().stream()
            .filter(fv -> fv.getField().getType() == FieldType.SELECT 
                        && "true".equals(fv.getField().getProperties().getOrDefault("partofcost", "false")))
            .mapToDouble(this::_getCost)
            .sum();
        
        return Double.toString(v);
    }
    
    private double _getCost(FieldValue fv)
    {
        return fv.getField().getProperties().entrySet().stream()
            .mapToDouble(e -> {
                Matcher m = __OPTION_INDEX.matcher(e.getKey());
                if (m.matches() && _equals(e.getValue(), fv.getValue()))
                {
                    String v = fv.getField().getProperties().getOrDefault("option-" + m.group(1) + "-cost", "0");
                    return Double.parseDouble(v);
                }
                else
                {
                    return 0.0;
                }
            })
            .sum();
    }
    
    private boolean _equals(String optionVaue, Object rawSelectedValues)
    {
        String[] selectedValues = StringUtils.split((String) rawSelectedValues, '\n');
        for (String selectedValue : selectedValues)
        {
            if (StringUtils.equals(optionVaue.trim(), selectedValue.trim())) // Comparing trim value since some rendering (including default rendering) add many leading spaces
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Validate the user input.
     * 
     * @param form the Form object.
     * @param input the user input.
     * @param errors the FormErrors object to fill.
     * @param request the user request.
     */
    protected void _validateInput(Form form, Map<String, FieldValue> input, FormErrors errors, Request request)
    {
        for (FieldValue entry : input.values())
        {
            Field field = entry.getField();
            switch (field.getType())
            {
                case TEXT:
                    errors.addErrors(field.getId(), _validateTextField(entry, request));
                    break;
                case PASSWORD:
                    errors.addErrors(field.getId(), _validatePassword(entry, request));
                    break;
                case SELECT:
                    errors.addErrors(field.getId(), _validateSelect(entry, request));
                    break;
                case TEXTAREA:
                    errors.addErrors(field.getId(), _validateTextarea(entry, request));
                    break;
                case RADIO:
                    errors.addErrors(field.getId(), _validateRadio(entry, request));
                    break;
                case CHECKBOX:
                    errors.addErrors(field.getId(), _validateCheckbox(entry, request));
                    break;
                case FILE:
                    errors.addErrors(field.getId(), _validateFile(entry, request));
                    break;
                case CAPTCHA:
                    errors.addErrors(field.getId(), _validateCaptcha(entry, request));
                    break;
                case HIDDEN:
                default:
                    break;
            }
        }
    }
    
    /**
     * Validate a text field.
     * 
     * @param entry the text field entry.
     * @param request the user request.
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validateTextField(FieldValue entry, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        final String textPrefix = "PLUGINS_FORMS_FORMS_RENDER_ERROR_TEXT_";

        errors.addAll(_validateMandatory(entry, textPrefix));

        errors.addAll(_validateConfirmation(entry, request, textPrefix));

        String regexpType = properties.get("regexptype");
        if (StringUtils.isEmpty(regexpType) || "text".equals(regexpType))
        {
            errors.addAll(_validateTextLength(entry, textPrefix));
        }
        else if (!StringUtils.isBlank(value))
        {
            _validateNonblankRegexp(entry, errors, value, textPrefix, regexpType);
        }

        return errors;
    }

    private void _validateNonblankRegexp(FieldValue entry, List<I18nizableText> errors, String value, final String textPrefix, String regexpType)
    {
        if ("int".equals(regexpType) && StringUtils.isBlank(value))
        {
            errors.addAll(_validateInteger(entry, textPrefix));
        }
        else if ("float".equals(regexpType))
        {
            errors.addAll(_validateFloat(entry, textPrefix));
        }
        else if ("email".equals(regexpType))
        {
            if (StringUtils.isNotEmpty(value) && !_EMAIL_PATTERN.matcher(value).matches())
            {
                errors.add(new I18nizableText("plugin." + _pluginName, "PLUGINS_FORMS_FORMS_RENDER_ERROR_TEXT_EMAIL"));
            }
        }
        else if ("phone".equals(regexpType))
        {
            if (StringUtils.isNotEmpty(value) && !_PHONE_PATTERN.matcher(value).matches())
            {
                errors.add(new I18nizableText("plugin." + _pluginName, "PLUGINS_FORMS_FORMS_RENDER_ERROR_TEXT_PHONE"));
            }
        }
        else if ("date".equals(regexpType))
        {
            errors.addAll(_validateDate(entry, textPrefix));
        }
        else if ("time".equals(regexpType))
        {
            errors.addAll(_validateTime(entry, textPrefix));
        }
        else if ("datetime".equals(regexpType))
        {
            errors.addAll(_validateDateTime(entry, textPrefix));
        }
        else if ("custom".equals(regexpType))
        {
            errors.addAll(_validateCustomRegexp(entry, textPrefix));
        }
    }

    /**
     * Validate a password field.
     * 
     * @param entry the password field entry.
     * @param request the user request.
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validatePassword(FieldValue entry, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();

        final String keyPrefix = "PLUGINS_FORMS_FORMS_RENDER_ERROR_PASSWORD_";

        errors.addAll(_validateMandatory(entry, keyPrefix));

        errors.addAll(_validateConfirmation(entry, request, keyPrefix));

        errors.addAll(_validateCustomRegexp(entry, keyPrefix));

        return errors;
    }

    /**
     * Validate a select input.
     * 
     * @param entry the select input entry.
     * @param request the user request.
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validateSelect(FieldValue entry, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();

        final String keyPrefix = "PLUGINS_FORMS_FORMS_RENDER_ERROR_SELECT_";

        errors.addAll(_validateMandatory(entry, keyPrefix));

        return errors;
    }

    /**
     * Validate a textarea.
     * 
     * @param entry the textarea entry.
     * @param request the user request.
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validateTextarea(FieldValue entry, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();

        final String keyPrefix = "PLUGINS_FORMS_FORMS_RENDER_ERROR_TEXTAREA_";

        errors.addAll(_validateMandatory(entry, keyPrefix));

        return errors;
    }

    /**
     * Validate a radio input.
     * 
     * @param entry the radio input entry.
     * @param request the user request.
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validateRadio(FieldValue entry, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();

        final String keyPrefix = "PLUGINS_FORMS_FORMS_RENDER_ERROR_RADIO_";

        errors.addAll(_validateMandatory(entry, keyPrefix));

        return errors;

    }

    /**
     * Validate a checkbox input.
     * 
     * @param entry the checkbox entry.
     * @param request the user request.
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validateCheckbox(FieldValue entry, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();
        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        Boolean value = (Boolean) entry.getValue();

        final String keyPrefix = "PLUGINS_FORMS_FORMS_RENDER_ERROR_CHECKBOX_";

        if (Boolean.parseBoolean(properties.get("mandatory")) && !value)
        {
            errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "MANDATORY"));
        }

        return errors;
    }

    /**
     * Validate a file input.
     * 
     * @param entry the file input entry.
     * @param request the user request.
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validateFile(FieldValue entry, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();
        
        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        File file = (File) entry.getValue();
        
        final String keyPrefix = "PLUGINS_FORMS_FORMS_RENDER_ERROR_FILE_";
        
        if (Boolean.parseBoolean(properties.get("mandatory")) && file == null)
        {
            errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "MANDATORY"));
        }
        
        if (file != null)
        {
            // Validate file readability.
            if (!file.isFile() || !file.canRead())
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "INVALID"));
            }
            
            // Validate file extensions.
            String fileExtensions = StringUtils.defaultString(properties.get("fileextension"));
            String[] fileExtArray = fileExtensions.split(",");
            
            boolean extensionOk = false;
            for (int i = 0; i < fileExtArray.length && !extensionOk; i++)
            {
                String ext = fileExtArray[i].trim().toLowerCase();
                extensionOk = file.getName().toLowerCase().endsWith(ext);
            }
            
            if (!extensionOk)
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "EXTENSION"));
            }
            
            float maxLength = _getFloat(properties.get("maxsize"), Float.MAX_VALUE);
            if (maxLength < Float.MAX_VALUE)
            {
                maxLength = maxLength * 1024 * 1024;
            }
            
            if (file.length() > maxLength)
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "TOOLARGE", Collections.singletonList(properties.get("maxsize"))));
            }
        }
        
        return errors;
    }

    /**
     * Validate a captcha.
     * 
     * @param entry the captcha entry.
     * @param request the user request.
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validateCaptcha(FieldValue entry, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();

        final String keyPrefix = "PLUGINS_FORMS_FORMS_RENDER_ERROR_CAPTCHA_";

        String[] values = (String[]) entry.getValue();
        String value = values[0];
        String key = values[1];

        if (!CaptchaHelper.checkAndInvalidate(key, value))
        {
            errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "INVALID"));
        }
        return errors;
    }

    /**
     * Validate a mandatory field.
     * @param entry the field value
     * @param keyPrefix the key profix
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validateMandatory(FieldValue entry, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        if (Boolean.parseBoolean(properties.get("mandatory")) && StringUtils.isBlank(value))
        {
            errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "MANDATORY"));
        }

        return errors;
    }

    /**
     * Validate a confirmation.
     * @param entry the field value
     * @param request the request
     * @param keyPrefix the key prefix
     * @return the list of error messages.
     */
    protected List<I18nizableText> _validateConfirmation(FieldValue entry, Request request, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        if (Boolean.parseBoolean(properties.get("confirmation")))
        {
            String confName = field.getName() + "_confirmation";
            String confValue = request.getParameter(confName);

            if (!value.equals(confValue))
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "CONFIRMATION"));
            }
        }

        return errors;
    }

    private List<I18nizableText> _validateTextLength(FieldValue entry, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        Integer minValue = _getInteger(properties.get("minvalue"), null);

        if (minValue != null)
        {
            if (StringUtils.isNotBlank(value) && value.length() < minValue)
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "MINLENGTH", Collections.singletonList(minValue.toString())));
            }
        }

        return errors;
    }

    private List<I18nizableText> _validateInteger(FieldValue entry, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        if (!_INT_PATTERN.matcher(value).matches())
        {
            errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "INTEGER"));
        }
        else
        {
            Integer intValue = _getInteger(value, null);
            Integer minValue = _getInteger(properties.get("minvalue"), null);
            Integer maxValue = _getInteger(properties.get("maxvalue"), null);

            if (minValue != null && intValue != null && intValue < minValue)
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "INTEGER_MINVALUE", Collections.singletonList(minValue.toString())));
            }
            if (maxValue != null && intValue != null && intValue > maxValue)
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "INTEGER_MAXVALUE", Collections.singletonList(maxValue.toString())));
            }
        }

        return errors;
    }

    private List<I18nizableText> _validateFloat(FieldValue entry, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        if (!_FLOAT_PATTERN.matcher(value).matches())
        {
            errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "FLOAT"));
        }
        else
        {
            Float floatValue = _getFloat(value, null);
            if (floatValue != null)
            {
                Integer minValue = _getInteger(properties.get("minvalue"), null);
                Integer maxValue = _getInteger(properties.get("maxvalue"), null);

                if (minValue != null && floatValue < minValue)
                {
                    errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "FLOAT_MINVALUE", Collections.singletonList(minValue.toString())));
                }
                if (maxValue != null && floatValue > maxValue)
                {
                    errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "FLOAT_MAXVALUE", Collections.singletonList(maxValue.toString())));
                }
            }
        }

        return errors;
    }

    private List<I18nizableText> _validateDate(FieldValue entry, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        if (!_DATE_PATTERN.matcher(value).matches())
        {
            errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "DATE"));
        }

        Date date = _getDate(value, null, _DATE_FORMAT);
        if (date != null)
        {
            Date minDate = _getDate(properties.get("minvalue"), null, _DATE_FORMAT);
            Date maxDate = _getDate(properties.get("maxvalue"), null, _DATE_FORMAT);

            if (minDate != null && date.before(minDate))
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "DATE_MINVALUE", Collections.singletonList(properties.get("minvalue"))));
            }
            if (maxDate != null && date.after(maxDate))
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "DATE_MAXVALUE", Collections.singletonList(properties.get("maxvalue"))));
            }
        }

        return errors;
    }

    private List<I18nizableText> _validateTime(FieldValue entry, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        if (!_TIME_PATTERN.matcher(value).matches())
        {
            errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "TIME"));
        }

        Date time = _getDate(value, null, _TIME_FORMAT);
        if (time != null)
        {
            Date minTime = _getDate(properties.get("minvalue"), null, _TIME_FORMAT);
            Date maxTime = _getDate(properties.get("maxvalue"), null, _TIME_FORMAT);

            if (minTime != null && time.before(minTime))
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "TIME_MINVALUE", Collections.singletonList(properties.get("minvalue"))));
            }
            if (maxTime != null && time.after(maxTime))
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "TIME_MAXVALUE", Collections.singletonList(properties.get("maxvalue"))));
            }
        }

        return errors;
    }

    private List<I18nizableText> _validateDateTime(FieldValue entry, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        if (!_DATETIME_PATTERN.matcher(value).matches())
        {
            errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "DATETIME"));
        }

        Date date = _getDate(value, null, _DATETIME_FORMAT);
        if (date != null)
        {
            Date minDate = _getDate(properties.get("minvalue"), null, _DATETIME_FORMAT);
            Date maxDate = _getDate(properties.get("maxvalue"), null, _DATETIME_FORMAT);

            if (minDate != null && date.before(minDate))
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "DATETIME_MINVALUE", Collections.singletonList(properties.get("minvalue"))));
            }
            if (maxDate != null && date.after(maxDate))
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "DATETIME_MAXVALUE", Collections.singletonList(properties.get("maxvalue"))));
            }
        }

        return errors;
    }

    private List<I18nizableText> _validateCustomRegexp(FieldValue entry, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();

        Field field = entry.getField();
        Map<String, String> properties = field.getProperties();
        String value = StringUtils.defaultString((String) entry.getValue());

        if (StringUtils.isNotBlank(value))
        {
            Integer minValue = _getInteger(properties.get("minvalue"), null);

            if (minValue != null && value.length() < minValue)
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "MINLENGTH", Collections.singletonList(minValue.toString())));
            }

            String jsPattern = properties.get("regexp");
            if (StringUtils.isNotBlank(jsPattern))
            {
                try
                {
                    String decodedJsPattern = URIUtils.decode(StringUtils.defaultString(jsPattern));
                    Pattern pattern = _getPattern(decodedJsPattern);

                    if (!pattern.matcher(value).matches())
                    {
                        errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "REGEXP"));
                    }
                }
                catch (PatternSyntaxException e)
                {
                    // Ignore, just don't validate.
                }
            }
        }

        return errors;
    }
    
    /**
     * Parses a JS pattern (i.e. "/^[a-z]+$/i")
     * @param jsRegexp the JS regexp string
     * @return the compiled Pattern object.
     */
    private Pattern _getPattern(String jsRegexp)
    {
        Pattern pattern = null;
        String regex = "";
        int flags = 0;

        int firstSlash = jsRegexp.indexOf('/');
        int lastSlash = jsRegexp.lastIndexOf('/');

        if (firstSlash > -1 && lastSlash > firstSlash)
        {
            regex = jsRegexp.substring(firstSlash + 1, lastSlash);
            if ("i".equals(jsRegexp.substring(lastSlash + 1)))
            {
                flags = Pattern.CASE_INSENSITIVE;
            }
            pattern = Pattern.compile(regex, flags);
        }

        return pattern;
    }

    private Integer _getInteger(String stringValue, Integer defaultValue)
    {
        Integer value = defaultValue;
        if (StringUtils.isNotEmpty(stringValue))
        {
            try
            {
                value = Integer.parseInt(stringValue);
            }
            catch (NumberFormatException e)
            {
                // Ignore.
            }
        }
        return value;
    }

    private Float _getFloat(String stringValue, Float defaultValue)
    {
        Float value = defaultValue;
        if (StringUtils.isNotEmpty(stringValue))
        {
            try
            {
                value = Float.parseFloat(stringValue);
            }
            catch (NumberFormatException e)
            {
                // Ignore.
            }
        }
        return value;
    }

    private Date _getDate(String stringValue, Date defaultValue, DateFormat format)
    {
        Date value = defaultValue;
        if (StringUtils.isNotEmpty(stringValue))
        {
            try
            {
                value = format.parse(stringValue);
            }
            catch (ParseException e)
            {
                // Ignore.
            }
        }
        return value;
    }

    /**
     * Retain only the entries that are to be inserted in the database.
     * 
     * @param entries the entries to filter.
     * @return the entries to insert in the database.
     */
    protected Collection<FieldValue> _getEntriesToInsert(Collection<FieldValue> entries)
    {
        List<FieldValue> filteredEntries = new ArrayList<>(entries.size());

        for (FieldValue entry : entries)
        {
            if (!FieldType.CAPTCHA.equals(entry.getField().getType()))
            {
                filteredEntries.add(entry);
            }
        }

        return filteredEntries;
    }

    /**
     * Send the receipt and notification emails.
     * 
     * @param form the Form object.
     * @param input the user input.
     * @param site the site.
     * @param totalSubmissions total number of submissions. Can be null if the form don't have limits.
     */
    protected void _sendEmails(Form form, Map<String, FieldValue> input, Site site, Integer totalSubmissions)
    {
        Config config = Config.getInstance();
        String sender = config.getValue("smtp.mail.from");

        if (site != null)
        {
            sender = site.getValue("site-mail-from");
        }

        _sendNotificationEmails(form, input, sender, site);

        _sendReceiptEmail(form, input, sender);

        if (!StringUtils.isEmpty(form.getLimit()) && Integer.parseInt(form.getLimit()) == totalSubmissions + 1)
        {
            _sendLimitEmail(form, input, sender, site);
        }
    }

    /**
     * Send the notification emails.
     * 
     * @param form the form.
     * @param input the user input.
     * @param sender the sender e-mail.
     * @param site the site
     */
    protected void _sendNotificationEmails(Form form, Map<String, FieldValue> input, String sender, Site site)
    {
        Set<String> emails = form.getNotificationEmails();
        try
        {
            String label = URIUtils.decode(form.getLabel());
            String siteTitle = site != null ? site.getTitle() : StringUtils.EMPTY;
            
            // Get subject
            I18nizableText subject = new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_RESULTS_SUBJECT", List.of(label, siteTitle));
            
            // Get body
            I18nizableText html = new I18nizableText("plugin.forms", "PLUGINS_FORMS_CONTENT_MAIL_RESULTS_TEXT", List.of(label));
            String prettyHtmlBody = StandardMailBodyHelper.newHTMLBody()
                    .withTitle(subject)
                    .withMessage(html)
                    .withDetails(new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_RESULTS_DETAILS_TITLE"), _getMail("entry.html", form, input), false)
                    .build();
            
            // Get text
            String params = "?type=results&form-name=" + form.getLabel();
            String text = _getMail("results.txt" + params, form, input);
            
            Collection<File> files = _getFiles(input);

            for (String email : emails)
            {
                if (StringUtils.isNotEmpty(email))
                {
                    try
                    {
                        SendMailHelper.newMail()
                                      .withSubject(_i18nUtils.translate(subject))
                                      .withHTMLBody(prettyHtmlBody)
                                      .withTextBody(text)
                                      .withAttachments(files)
                                      .withSender(sender)
                                      .withRecipient(email)
                                      .sendMail();
                    }
                    catch (MessagingException | IOException e)
                    {
                        getLogger().error("Error sending the notification mail to " + email, e);
                    }
                }
            }
        }
        catch (IOException e)
        {
            getLogger().error("Error creating the notification message.", e);
        }
    }

    /**
     * Send the receipt email.
     * 
     * @param form the form.
     * @param input the user input.
     * @param sender the sender e-mail.
     */
    protected void _sendReceiptEmail(Form form, Map<String, FieldValue> input, String sender)
    {
        String email = "";
        try
        {
            String receiptFieldId = form.getReceiptFieldId();
            if (StringUtils.isNotEmpty(receiptFieldId))
            {
                FieldValue receiptEntry = input.get(receiptFieldId);

                if (receiptEntry.getValue() != null)
                {
                    email = receiptEntry.getValue().toString();

                    if (_EMAIL_PATTERN.matcher(email).matches())
                    {
                        String subject = URIUtils.decode(form.getReceiptFieldSubject());
                        String bodyTxt = URIUtils.decode(form.getReceiptFieldBody());
                        String bodyHTML = bodyTxt.replaceAll("\r?\n", "<br/>");
                        
                        if (bodyTxt.contains(__FORM_ENTRY_PATTERN))
                        {
                            String entry2html = _getMail("entry.html", form, input);
                            String entry2text = _getMail("entry.txt", form, input);
                            
                            bodyTxt = StringUtils.replace(bodyTxt, __FORM_ENTRY_PATTERN, entry2text);
                            bodyHTML = StringUtils.replace(bodyHTML, __FORM_ENTRY_PATTERN, entry2html);
                        }
                        
                        String prettyHtmlBody = StandardMailBodyHelper.newHTMLBody()
                                .withTitle(subject)
                                .withMessage(bodyHTML)
                                .build();
                        
                        String overrideSender = form.getReceiptFieldFromAddress();

                        SendMailHelper.newMail()
                                      .withSubject(subject)
                                      .withHTMLBody(prettyHtmlBody)
                                      .withTextBody(bodyTxt)
                                      .withSender(StringUtils.isEmpty(overrideSender) ? sender : overrideSender)
                                      .withRecipient(email)
                                      .sendMail();
                    }
                }
            }
        }
        catch (MessagingException | IOException e)
        {
            getLogger().error("Error sending the receipt mail to " + email, e);
        }
    }

    /**
     * Send the limit email.
     * 
     * @param form the form.
     * @param input the user input.
     * @param sender the sender e-mail.
     * @param site the site
     */
    protected void _sendLimitEmail(Form form, Map<String, FieldValue> input, String sender, Site site)
    {
        Set<String> emails = form.getNotificationEmails();
        try
        {
            String label = URIUtils.decode(form.getLabel());
            String siteTitle = site != null ? site.getTitle() : StringUtils.EMPTY;
            
            // Get subject
            I18nizableText subject = new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_LIMIT_SUBJECT", List.of(label, siteTitle));
            
            // Get body
            I18nizableText html = new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_LIMIT_TEXT", List.of(label));
            String prettyHtmlBody = StandardMailBodyHelper.newHTMLBody()
                    .withTitle(subject)
                    .withMessage(html)
                    .build();
            
            // Get text
            I18nizableText text = new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_LIMIT_TEXT_NO_HTML", List.of(label));
            
            Collection<File> files = _getFiles(input);

            for (String email : emails)
            {
                if (StringUtils.isNotEmpty(email))
                {
                    try
                    {
                        SendMailHelper.newMail()
                                      .withSubject(_i18nUtils.translate(subject))
                                      .withHTMLBody(prettyHtmlBody)
                                      .withTextBody(_i18nUtils.translate(text))
                                      .withAttachments(files)
                                      .withSender(sender)
                                      .withRecipient(email)
                                      .sendMail();
                    }
                    catch (MessagingException e)
                    {
                        getLogger().error("Error sending the limit mail to " + email, e);
                    }
                }
            }
        }
        catch (IOException e)
        {
            getLogger().error("Error creating the limit message.", e);
        }
    }
    /**
     * Get a mail pipeline's content.
     * 
     * @param resource the mail resource pipeline (i.e. "results.html" or
     *            "receipt.txt").
     * @param form the Form.
     * @param input the user input.
     * @return the mail content.
     * @throws IOException if an error occurs.
     */
    protected String _getMail(String resource, Form form, Map<String, FieldValue> input) throws IOException
    {
        Source src = null;

        try
        {
            String uri = "cocoon:/mail/" + resource;
            Map<String, Object> parameters = new HashMap<>();
            parameters.put("form", form);
            parameters.put("input", input);

            src = _sourceResolver.resolveURI(uri, null, parameters);
            Reader reader = new InputStreamReader(src.getInputStream(), "UTF-8");
            return IOUtils.toString(reader);
        }
        finally
        {
            _sourceResolver.release(src);
        }
    }

    /**
     * Get the files of a user input.
     * 
     * @param input the user input.
     * @return the files submitted by the user.
     */
    protected Collection<File> _getFiles(Map<String, FieldValue> input)
    {
        List<File> files = new ArrayList<>();

        for (FieldValue entry : input.values())
        {
            if (FieldType.FILE.equals(entry.getField().getType()))
            {
                File file = (File) entry.getValue();
                if (file != null)
                {
                    files.add(file);
                }
            }
        }

        return files;
    }
    
    /**
     * Class representing form informations
     */
    public static class FormInformations
    {
        private FormErrors _formErrors;
        private String _redirection;
        
        /**
         * Construction for a form information
         */
        public FormInformations()
        {
            _formErrors = null;
            _redirection = null;
        }
        
        /**
         * The form errors
         * @return the form errors
         */
        public FormErrors getFormErrors()
        {
            return _formErrors;
        }
        
        /**
         * Set the form errors
         * @param formErrors the form errors
         */
        public void setFormErrors(FormErrors formErrors)
        {
            _formErrors = formErrors;
        }
        
        /**
         * Get the form redirection
         * @return the redirection
         */
        public String getRedirection()
        {
            return _redirection;
        }
        
        /**
         * Set the form redirection
         * @param redirection the redirection
         */
        public void setRedirection(String redirection)
        {
            _redirection = redirection;
        }
    }

}
