/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.table;

import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;

import org.apache.commons.lang.StringUtils;

import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.datasource.dbtype.SQLDatabaseTypeExtensionPoint;
import org.ametys.plugins.forms.content.table.FormTableManager.DbColumn;

/**
 * Database abstraction helper.
 */
public final class DbTypeHelper
{
    private DbTypeHelper()
    {
        // Hides the default constructor.
    }
    
    /**
     * Normalize a name (table name, column name, and so on) according to the database type.
     * @param dbType the database type.
     * @param str the string to normalize.
     * @return the normalized string.
     */
    public static String normalizeName(String dbType, String str)
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_ORACLE:
                return StringUtils.replace(str, "-", "_");
            default:
                return str;
        }
    }
    
    /**
     * Filter a name (table name, column name, and so on) according to the database type
     * so that it can be passed to {@link DatabaseMetaData} methods, such as getTables and getColumns.
     * @param dbType the database type.
     * @param str the string to filter.
     * @return the filtered string.
     */
    public static String filterName(String dbType, String str)
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_ORACLE:
                return str.toUpperCase();
            default:
                return str;
        }
    }
    
    /**
     * Get the varchar type identifier corresponding to the database type.
     * @param dbType the database type.
     * @return the real varchar type identifier (to be included in CREATE statement).
     */
    public static String getVarcharType(String dbType)
    {
        return "VARCHAR(255)";
    }
    
    /**
     * Get the text type identifier corresponding to the database type.
     * @param dbType the database type.
     * @return the real text type identifier (to be included in CREATE statement).
     */
    public static String getTextType(String dbType)
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_DERBY:
                return "LONG VARCHAR";
            case ConnectionHelper.DATABASE_ORACLE:
                return "CLOB";
            default:
                return "TEXT";
        }
    }
    
    /**
     * Get the boolean type identifier corresponding to the database type.
     * @param dbType the database type.
     * @return the real boolean type identifier (to be included in CREATE statement).
     */
    public static String getBooleanType(String dbType)
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_DERBY:
            case ConnectionHelper.DATABASE_POSTGRES:
                return "SMALLINT";
            case ConnectionHelper.DATABASE_ORACLE:
                return "NUMBER(1)";
            default:
                return "INT(1)";
        }
    }
    
    /**
     * Get the binary type identifier corresponding to the database type.
     * @param dbType the database type.
     * @return the real binary type identifier (to be included in CREATE statement).
     */
    public static String getBinaryType(String dbType)
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_MYSQL:
                return "LONGBLOB";
            case ConnectionHelper.DATABASE_POSTGRES:
                return "BYTEA";
            default:
                return "BLOB";
        }
    }
    
    /**
     * Get the integer type identifier corresponding to the database type.
     * @param dbType the database type.
     * @return the integer type identifier (to be included in CREATE statement).
     */
    public static String getIntegerType(String dbType)
    {
        return "INTEGER";
    }
    
    /**
     * Get the date and time type identifier corresponding to the database type.
     * @param dbType the database type.
     * @return the real date and time type identifier (to be included in CREATE statement).
     */
    public static String getDateTimeType(String dbType)
    {
        return "TIMESTAMP";
    }
    
    /**
     * Get the "identity" type (int, integer, number, ...)
     * @param dbType the database type.
     * @return the "identity" type
     */
    public static String getIdentityType (String dbType)
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_ORACLE:
                return "number NOT NULL";
            case ConnectionHelper.DATABASE_MYSQL:
            case ConnectionHelper.DATABASE_DERBY:
                return "int NOT NULL";
            default:
                return "";
        }
    }
    /**
     * Get the "identity" string marker (AUTO_INCREMENT, SERIAL...)
     * @param dbType the database type.
     * @return the "identity" string marker.
     */
    public static String getIdentityMarker(String dbType)
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_MYSQL:
                return "AUTO_INCREMENT";
            case ConnectionHelper.DATABASE_DERBY:
                return "GENERATED BY DEFAULT AS IDENTITY (START WITH 1, INCREMENT BY 1)";
            case ConnectionHelper.DATABASE_POSTGRES:
                return "SERIAL";
            default:
                return "";
        }
    }
    
    /**
     * Get the "identity" value (NULL, sequence next val call, ...)
     * @param dbType the database type.
     * @return the "identity" value.
     */
    public static String getIdentityValue(String dbType)
    {
        return "NULL";
    }
    
    /**
     * Get the "now" function (NOW(), CURDATE, ...)
     * @param dbType the database type.
     * @return the "now" function.
     */
    public static String getCurrentDateFunction(String dbType)
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_MYSQL:
                return "NOW()";
            case ConnectionHelper.DATABASE_ORACLE:
                return "CURDATE";
            case ConnectionHelper.DATABASE_POSTGRES:
                return "CURRENT_TIMESTAMP";
            case ConnectionHelper.DATABASE_DERBY:
            default:
                return "";
        }
    }
    
    /**
     * Test if we have to include the identity column in INSERTs.
     * @param dbType the database type.
     * @return true if the identity is to be included in INSERT statements, false otherwise. 
     */
    public static boolean insertIdentity(String dbType)
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_DERBY:
            case ConnectionHelper.DATABASE_POSTGRES:
                return false;
            default:
                return true;
        }
    }
    
    /**
     * Set the identity value into the statement.
     * @param stmt the prepared statement.
     * @param index the column index.
     * @param dbType the database type.
     * @throws SQLException if a SQL exception occurs.
     */
    public static void setIdentity(PreparedStatement stmt, int index, String dbType) throws SQLException
    {
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_MYSQL:
                stmt.setNull(index, Types.INTEGER);
                break;
            case ConnectionHelper.DATABASE_DERBY:
                break;
            case ConnectionHelper.DATABASE_POSTGRES:
                stmt.setNull(index, Types.INTEGER);
                break;
            case ConnectionHelper.DATABASE_ORACLE:
                return;
            default:
                stmt.setNull(index, Types.INTEGER);
                break;
                
        }
    }
    
    /**
     * Get a SQL statement to rename a column.
     * @param tableName the table name.
     * @param column the column to rename.
     * @param newName the new column name.
     * @param dbType the database type.
     * @param sqlDatabaseTypeExtensionPoint SQLDatabaseTypeExtensionPoint
     * @return the SQL rename query.
     */
    public static String getRenameColumnStatement(String tableName, DbColumn column, String newName, String dbType, SQLDatabaseTypeExtensionPoint sqlDatabaseTypeExtensionPoint)
    {
        return getRenameColumnStatement(tableName, column.getName(), newName, column.getColumnTypeIdentifier(), dbType, sqlDatabaseTypeExtensionPoint);
    }
    
    /**
     * Get a SQL statement to rename a column.
     * @param tableName the table name.
     * @param columnName the current column name.
     * @param newName the new column name.
     * @param columnType the full column type string (i.e TEXT, VARCHAR(255), INT(1), and so on).
     * @param dbType the database type.
     * @param sqlDatabaseTypeExtensionPoint SQLDatabaseTypeExtensionPoint
     * @return the SQL rename query.
     */
    public static String getRenameColumnStatement(String tableName, String columnName, String newName, String columnType, String dbType, SQLDatabaseTypeExtensionPoint sqlDatabaseTypeExtensionPoint)
    {
        StringBuilder buff = new StringBuilder();
        
        String escapedTableName = sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, tableName);
        String escapedColumnName = sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, columnName);
        String escapedNewName = sqlDatabaseTypeExtensionPoint.languageEscapeTableName(dbType, newName);
        
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_MYSQL:
                buff.append("ALTER TABLE ").append(escapedTableName);
                buff.append(" CHANGE ").append(escapedColumnName);
                buff.append(" ").append(escapedNewName).append(" ");
                buff.append(columnType);
                buff.append(" DEFAULT NULL");
                break;
            case ConnectionHelper.DATABASE_DERBY:
                buff.append("RENAME COLUMN ").append(escapedTableName).append(".").append(escapedColumnName);
                buff.append(" TO ").append(escapedNewName);
                break;
            case ConnectionHelper.DATABASE_ORACLE:
            case ConnectionHelper.DATABASE_POSTGRES:
            default:
                buff.append("ALTER TABLE ").append(escapedTableName);
                buff.append(" RENAME COLUMN ").append(escapedColumnName);
                buff.append(" TO ").append(escapedNewName);
                break;
        }
        
        return buff.toString();
    }
    
}
