/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.forms.dao;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.collections.ListUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.upload.UploadManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.forms.FormEvents;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.FormQuestionTypeExtensionPoint;
import org.ametys.plugins.forms.question.sources.AbstractSourceType;
import org.ametys.plugins.forms.question.sources.ChoiceOption;
import org.ametys.plugins.forms.question.sources.ChoiceSourceType;
import org.ametys.plugins.forms.question.sources.ChoiceSourceTypeExtensionPoint;
import org.ametys.plugins.forms.question.types.ChoicesListQuestionType;
import org.ametys.plugins.forms.repository.CopyFormUpdater;
import org.ametys.plugins.forms.repository.CopyFormUpdaterExtensionPoint;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormPage;
import org.ametys.plugins.forms.repository.FormPageRule;
import org.ametys.plugins.forms.repository.FormPageRule.PageRuleType;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.repository.type.Rule;
import org.ametys.plugins.forms.repository.type.Rule.QuestionRuleType;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.parameters.ParametersManager;

/** DAO for manipulating form questions */
public class FormQuestionDAO extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** The Avalon role */
    public static final String ROLE = FormQuestionDAO.class.getName();

    /** Name for rules root jcr node */
    public static final String RULES_ROOT = "ametys-internal:form-page-rules";
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    /** Manager for retrieving uploaded files */
    protected UploadManager _uploadManager;
    /** JSON helper */
    protected JSONUtils _jsonUtils;
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    /** The form question type extension point */
    protected FormQuestionTypeExtensionPoint _formQuestionTypeExtensionPoint;
    /** The parameters manager */
    protected ParametersManager _parametersManager;
    /** The Avalon context */
    protected Context _context;
    /** The cocoon context */
    protected org.apache.cocoon.environment.Context _cocoonContext;
    /** The choice source type extension point */
    protected ChoiceSourceTypeExtensionPoint _choiceSourceTypeExtensionPoint;
    /**The form DAO */
    protected FormDAO _formDAO;
    /** The right manager */
    protected RightManager _rightManager;
    /** The copy form updater extension point */
    protected CopyFormUpdaterExtensionPoint _copyFormEP;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) serviceManager.lookup(ObservationManager.ROLE);
        _parametersManager = (ParametersManager) serviceManager.lookup(ParametersManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _uploadManager = (UploadManager) serviceManager.lookup(UploadManager.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _formQuestionTypeExtensionPoint = (FormQuestionTypeExtensionPoint) serviceManager.lookup(FormQuestionTypeExtensionPoint.ROLE);
        _parametersManager = (ParametersManager) serviceManager.lookup(ParametersManager.ROLE);
        _formDAO = (FormDAO) serviceManager.lookup(FormDAO.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _copyFormEP = (CopyFormUpdaterExtensionPoint) serviceManager.lookup(CopyFormUpdaterExtensionPoint.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (org.apache.cocoon.environment.Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    /**
     * Provides the current user.
     * @return the user which cannot be <code>null</code>.
     */
    protected UserIdentity _getCurrentUser()
    {      
        return _currentUserProvider.getUser();
    }
    
    /**
     * Gets properties of a form question
     * @param id The id of the form question
     * @return The properties
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getQuestionProperties (String id)
    {
        try
        {
            FormQuestion question = _resolver.resolveById(id);
            return getQuestionProperties(question, true);
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Can't find question with id: {}. It probably has just been deleted", id, e);
            Map<String, Object> infos = new HashMap<>();
            infos.put("id", id);
            return infos;
        }
    }
    
    /**
     * Gets properties of a form question
     * @param question The form question
     * @param withRight <code>true</code> to have the rights in the properties
     * @return The properties
     */
    public Map<String, Object> getQuestionProperties (FormQuestion question, boolean withRight)
    {
        Map<String, Object> properties = new HashMap<>();
        
        boolean hasTerminalRule = _hasTerminalRule(question);
        List<String> questionTitlesWithRule = _getQuestionTitlesWithRule(question);
        List<String> pageTitlesWithRule = _getPageTitlesWithRule(question);

        properties.put("type", "question"); 
        properties.put("hasTerminalRule", hasTerminalRule);
        properties.put("pageTitlesWithRule", pageTitlesWithRule);
        properties.put("questionTitlesWithRule", questionTitlesWithRule);
        properties.put("isReadRestricted", question.isReadRestricted());
        properties.put("isModifiable", question.isModifiable());
        properties.put("hasChildren", false);

        /** Use in the bus message */
        properties.put("id", question.getId());
        properties.put("title", question.getTitle());
        properties.put("questionType", question.getType().getId());
        properties.put("pageId", question.getFormPage().getId());
        properties.put("formId", question.getForm().getId());
        properties.put("iconGlyph", question.getType().getIconGlyph());
        properties.put("typeLabel", question.getType().getLabel());
        properties.put("hasEntries", !question.getForm().getEntries().isEmpty());
        properties.put("hasRule", hasTerminalRule || !pageTitlesWithRule.isEmpty() || !questionTitlesWithRule.isEmpty());
        properties.put("isConfigured", question.getType().isQuestionConfigured(question));
        
        if (withRight)
        {
            properties.put("rights", _getUserRights(question));
        }
        else
        {
            properties.put("canWrite", _formDAO.hasWriteRightOnForm(_currentUserProvider.getUser(), question));
        }
        
        return properties;
    }
    
    /**
     * Get options from the choice list question
     * @param questionId the choice list question id
     * @return the map of option
     */
    @Callable (rights = FormDAO.HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, I18nizableText> getChoiceListQuestionOptions (String questionId)
    {
        FormQuestion question = _resolver.resolveById(questionId);
        if (question.getType() instanceof ChoicesListQuestionType type)
        {
            return type.getOptions(question);
        }
        
        return new HashMap<>();
    }

    /**
     * Get user rights for the given form question
     * @param question the form question
     * @return the set of rights
     */
    protected Set<String> _getUserRights (FormQuestion question)
    {
        UserIdentity user = _currentUserProvider.getUser();
        return _rightManager.getUserRights(user, question);
    }
    
    private boolean _hasTerminalRule(FormQuestion question)
    {
        return question.getPageRules()
                .stream()
                .map(FormPageRule::getType)
                .filter(t -> t == PageRuleType.FINISH)
                .findAny()
                .isPresent();
    }

    /**
     * Get the question titles having rule concerning the given question
     * @param question the question
     * @return the list of question titles
     */
    protected List<String> _getQuestionTitlesWithRule(FormQuestion question)
    {
        return question.getForm()
            .getQuestionsRule(question.getId())
            .keySet()
            .stream()
            .map(FormQuestion::getTitle)
            .toList();
    }
    
    /**
     * Get the page titles having rule concerning the given question
     * @param question the question
     * @return the list of page titles
     */
    protected List<String> _getPageTitlesWithRule(FormQuestion question)
    {
        return question.getPageRules()
            .stream()
            .filter(r -> r.getType() != PageRuleType.FINISH)
            .map(FormPageRule::getPageId)
            .distinct()
            .map(this::_getFormPage)
            .map(FormPage::getTitle)
            .toList();
    }
    
    private FormPage _getFormPage(String pageId)
    {
        return _resolver.resolveById(pageId);
    }
    
    /**
     * Get view for question type
     * @param typeID id of the question type
     * @param formId id of the form
     * @return the view parsed in json for configurableFormPanel 
     * @throws ProcessingException  error while parsing view to json
     */
    @Callable (rights = FormDAO.HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 1)
    public Map<String, Object> getQuestionParametersDefinitions(String typeID, String formId) throws ProcessingException
    {
        Map<String, Object> response = new HashMap<>();
        Form form = _resolver.resolveById(formId);
        FormQuestionType questionType = _formQuestionTypeExtensionPoint.getExtension(typeID);
        View view = questionType.getView(form);
        response.put("parameters", view.toJSON(DefinitionContext.newInstance().withEdition(true)));
        response.put("questionNames", form.getQuestionsNames());
        return response;
    }
    
    /**
     * Get questions parameters values
     * @param questionID id of current question
     * @return map of question parameters value 
     */
    @Callable (rights = FormDAO.HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getQuestionParametersValues(String questionID)
    {
        Map<String, Object> results = new HashMap<>();
        
        FormQuestion question = _resolver.resolveById(questionID);
        FormQuestionType type = question.getType();
        Collection< ? extends ModelItem> questionModelItems = type.getModel().getModelItems();
        Map<String, Object> parametersValues = _parametersManager.getParametersValues(questionModelItems, question, StringUtils.EMPTY);
        results.put("values", parametersValues);
        
        // Repeater values aren't handled by getParametersValues()
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> repeaters = _parametersManager.getRepeatersValues((Collection<ModelItem>) questionModelItems, question, StringUtils.EMPTY);
        results.put("repeaters", repeaters);
        results.put("fieldToDisable", _getFieldNameToDisable(question));
        
        return results;
    }
    
    private List<String> _getFieldNameToDisable(FormQuestion question)
    {
        Form form = question.getForm();
        if (form.getEntries().isEmpty())
        {
            return List.of();
        }
        
        return question.getType().getFieldToDisableIfFormPublished(question);
    }
    
    /**
     * Creates a {@link FormQuestion}.
     * @param pageId id of current page
     * @param typeId id of FormQuestionType
     * @return The id of the created form question, the id of the page and the id of the form
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> createQuestion(String pageId, String typeId)
    {
        Map<String, Object> result = new HashMap<>();
        FormPage page = _resolver.resolveById(pageId);

        _formDAO.checkHandleFormRight(page);
        
        FormQuestionType type = _formQuestionTypeExtensionPoint.getExtension(typeId);
        Form form = page.getForm();
        
        String defaultTitle = _i18nUtils.translate(type.getDefaultTitle());
        String nameForForm = form.findUniqueQuestionName(defaultTitle);
        
        FormQuestion question = page.createChild(nameForForm, "ametys:form-question");
        question.setNameForForm(nameForForm);
        question.setTypeId(typeId);
        
        Model model = question.getType().getModel();
        for (ModelItem modelItem : model.getModelItems())
        {
            if (modelItem instanceof ElementDefinition)
            {
                Object defaultValue = ((ElementDefinition) modelItem).getDefaultValue();
                if (defaultValue != null)
                {
                    question.setValue(modelItem.getPath(), defaultValue);
                }
            }
        }
        
        question.setTitle(form.findUniqueQuestionTitle(defaultTitle));
        
        page.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", page.getForm());
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", question.getId());
        result.put("pageId", page.getId());
        result.put("formId", question.getForm().getId());
        result.put("type", typeId);
        return result;
    }
    
    /**
     * Rename a {@link FormQuestion}
     * @param id The id of the question 
     * @param newName The new name of the question
     * @return A result map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, String> renameQuestion (String id, String newName)
    {
        Map<String, String> results = new HashMap<>();
        
        FormQuestion question = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(question);
        
        question.setTitle(newName);
        question.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", question.getForm());
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        results.put("id", id);
        results.put("newName", newName);
        results.put("formId", question.getForm().getId());
        
        return results;
    }
    
    /**
     * Edits a {@link FormQuestion}.
     * @param questionId id of current question
     * @param values The question's values
     * @return The id of the edited form question, the id of the page and the id of the form
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> editQuestion (String questionId, Map<String, Object> values)
    {
        Map<String, Object> result = new HashMap<>();
        Map<String, I18nizableText> errors = new HashMap<>();
        
        FormQuestion question = _resolver.resolveById(questionId);
        _formDAO.checkHandleFormRight(question);
        
        Form parentForm = question.getForm();
        String questionName = StringUtils.defaultString((String) values.get("name-for-form"));
        
        // if question can not be answered by user, id can't be changed and is unique by default
        if (!question.getType().canBeAnsweredByUser(question) || questionName.equals(question.getNameForForm()) || parentForm.isQuestionNameUnique(questionName))
        {
            FormQuestionType type = question.getType();
            type.validateQuestionValues(values, errors);

            if (!errors.isEmpty())
            {
                result.put("errors", errors);
                return result;
            }

            _parametersManager.setParameterValues(question.getDataHolder(), type.getModel().getModelItems(), values);
            type.doAdditionalOperations(question, values);
            
            question.saveChanges();
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put("form", parentForm);
            _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
            
            result.put("id", question.getId());
            result.put("pageId", question.getParent().getId());
            result.put("formId", parentForm.getId());
            result.put("type", question.getType().toString());
        }
        else
        {
            errors.put("duplicate_name", new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_SET_ID_ERROR"));
            result.put("errors", errors);
            getLogger().error("An error occurred creating the question. The identifier value '" + questionName + "' is already used.");
        }
        
        return result;
    }
    
    /**
     * Deletes a {@link FormQuestion}.
     * @param id The id of the form question to delete
     * @return The id of the form question, the id of the page and the id of the form
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, String> deleteQuestion (String id)
    {
        FormQuestion question = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(question);
        
        question.getForm().deleteQuestionsRule(question.getId());
        
        FormPage page = question.getParent();
        question.remove();
        
        page.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", page.getForm());
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        return Map.of("id", id);
    }

    /**
     * Copies and pastes a form question.
     * @param pageId The id of the page, target of the copy
     * @param questionId The id of the question to copy
     * @return The id of the created question, the id of the page and the id of the form
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, String> copyQuestion(String pageId, String questionId)
    {
        Map<String, String> result = new HashMap<>();
        
        FormQuestion originalQuestion = _resolver.resolveById(questionId);
        _formDAO.checkHandleFormRight(originalQuestion);
        
        FormPage parentPage = _resolver.resolveById(pageId);
        
        Form parentForm = parentPage.getForm();
        
        String uniqueName = parentForm.findUniqueQuestionName(originalQuestion.getNameForForm());
        FormQuestion questionCopy = parentPage.createChild(uniqueName, "ametys:form-question");
        originalQuestion.copyTo(questionCopy);
        
        String copyTitle = _i18nUtils.translate(new I18nizableText("plugin.forms", "PLUGIN_FORMS_TREE_COPY_NAME_PREFIX")) + originalQuestion.getTitle();
        questionCopy.setTitle(parentForm.findUniqueQuestionTitle(copyTitle));
        questionCopy.setTypeId(originalQuestion.getType().getId());
        questionCopy.setNameForForm(uniqueName);
        
        for (String epId : _copyFormEP.getExtensionsIds())
        {
            CopyFormUpdater copyFormUpdater = _copyFormEP.getExtension(epId);
            copyFormUpdater.updateFormQuestion(originalQuestion, questionCopy);
        }
        
        parentPage.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", parentForm);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", questionCopy.getId());
        result.put("pageId", parentPage.getId());
        result.put("formId", parentForm.getId());
        result.put("type", questionCopy.getType().getId());
        
        return result;
    }
    
    /**
     * Gets the page rules for a form question.
     * @param id The id of the form question.
     * @param number The question number
     * @return The rules
     * @throws Exception error while getting choice options
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getRules (String id, int number) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        
        FormQuestion question = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(question);
        
        FormQuestionType type = question.getType();
        if (type instanceof ChoicesListQuestionType cLType)
        {
            ChoiceSourceType sourceType = cLType.getSourceType(question);
          
            result.put("id", question.getId());
            result.put("number", String.valueOf(number));
            result.put("title", question.getTitle());
            
            List<Object> rules = new ArrayList<>();
            for (FormPageRule rule : question.getPageRules())
            {
                String option = rule.getOption();
                Map<String, Object> enumParam = new HashMap<>();
                enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
                I18nizableText label = sourceType.getEntry(new ChoiceOption(option), enumParam);
                
                Map<String, Object> resultRule = new HashMap<>();
                resultRule.put("option", option);
                resultRule.put("optionLabel", label);
                resultRule.put("type", rule.getType());
                String pageId = rule.getPageId();
                if (pageId != null)
                {
                    try
                    {
                        FormPage page = _resolver.resolveById(pageId);
                        resultRule.put("page", pageId);
                        resultRule.put("pageName", page.getTitle());
                    }
                    catch (UnknownAmetysObjectException e)
                    {
                        // Page does not exist anymore
                    }
                }
                        
                rules.add(resultRule);
            }
            
            result.put("rules", rules);
        }
        
        return result;
    }

    /**
     * Adds a new rule to a question.
     * @param id The question id
     * @param option The option
     * @param rule The rule type
     * @param page The page to jump or skip
     * @return An empty map, or an error
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> addPageRule (String id, String option, String rule, String page)
    {
        Map<String, Object> result = new HashMap<>();
        
        FormQuestion question = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(question);
        
        // Check if exists
        if (question.hasPageRule(option))
        {
            result.put("error", "already-exists");
            return result;
        }
        
        question.addPageRules(option, PageRuleType.valueOf(rule), page);
        question.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", question.getForm());
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", question.getId());
        result.put("pageId", question.getFormPage().getId());
        result.put("formId", question.getForm().getId());
        result.put("type", question.getType().getId());
        return result;
    }
    
    /**
     * Deletes a rule to a question.
     * @param id The question id
     * @param option The option to delete
     * @return An empty map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> deletePageRule (String id, String option)
    {
        FormQuestion question = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(question);
        
        question.deletePageRule(option);
        question.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", question.getForm());
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        return new HashMap<>();
    }
    
    /**
     * Record for entry values coming from input or from the entry
     * @param inputValues the inputValues. Can be null if the entry is not null
     * @param entry the form entry. Can be null if the input values is not null
     */
    public record FormEntryValues(Map<String, Object> inputValues, FormEntry entry) 
    {
        Object getValue(String attributeName)
        {
            if (inputValues != null)
            {
                return inputValues.get(attributeName);
            }
            else
            {
                return entry.getValue(attributeName);
            }
        }
    }
    
    /**
     * Get the list of active question depending of the form rules
     * @param form the form
     * @param entryValues the entry values to compute rules
     * @param currentStepId the current step id. Can be empty if the form has no workflow
     * @param onlyWritableQuestion <code>true</code> to have only writable question
     * @param onlyReadableQuestion <code>true</code> to have only readable question
     * @return the list of active question depending of the form rules
     */
    public List<FormQuestion> getRuleFilteredQuestions(Form form, FormEntryValues entryValues, Optional<Long> currentStepId, boolean onlyWritableQuestion, boolean onlyReadableQuestion)
    {
        List<FormQuestion> filteredQuestions = new ArrayList<>();
        for (FormQuestion activeQuestion : _getActiveQuestions(form, entryValues, currentStepId, onlyWritableQuestion, onlyReadableQuestion))
        {
            if (!activeQuestion.getType().onlyForDisplay(activeQuestion))
            {
                Optional<Rule> firstQuestionRule = activeQuestion.getFirstQuestionRule();
                if (firstQuestionRule.isPresent())
                {
                    Rule rule = firstQuestionRule.get();
                    FormQuestion sourceQuestion = _resolver.resolveById(rule.getSourceId());
                    List<String> ruleValues = _getRuleValues(entryValues, sourceQuestion.getNameForForm());
                    boolean equalsRuleOption = ruleValues.contains(rule.getOption());
                    QuestionRuleType ruleAction = rule.getAction();
                    
                    if (!equalsRuleOption && ruleAction.equals(QuestionRuleType.HIDE)
                            || equalsRuleOption && ruleAction.equals(QuestionRuleType.SHOW))
                    {
                        filteredQuestions.add(activeQuestion);
                    }
                }
                else
                {
                    filteredQuestions.add(activeQuestion);
                }
            }
        }
        
        return filteredQuestions;
    }
    
    /**
     * Get a list of the form questions not being hidden by a rule
     * @param form the current form
     * @param entryValues the entry values
     * @param currentStepId current step of the entry. Can be empty if the form has no workflow
     * @param onlyWritableQuestion <code>true</code> to have only writable question
     * @param onlyReadableQuestion <code>true</code> to have only readable question
     * @return a list of visible questions
     */
    protected List<FormQuestion> _getActiveQuestions(Form form, FormEntryValues entryValues, Optional<Long> currentStepId, boolean onlyWritableQuestion, boolean onlyReadableQuestion)
    {
        String nextActivePage = null;
        List<FormQuestion> activeQuestions = new ArrayList<>();
        for (FormPage page : form.getPages())
        {
            if (nextActivePage == null || page.getId().equals(nextActivePage))
            {
                nextActivePage = null;
                for (FormQuestion question : page.getQuestions())
                {
                    if (currentStepId.isEmpty() // no current step id, ignore rights access
                        || (!onlyReadableQuestion || question.canRead(currentStepId.get())) 
                            && 
                           (!onlyWritableQuestion || question.canWrite(currentStepId.get())))
                    {
                        activeQuestions.add(question);
                    }
                    
                    if (question.getType() instanceof ChoicesListQuestionType type && !type.getSourceType(question).remoteData())
                    {
                        List<String> ruleValues = _getRuleValues(entryValues, question.getNameForForm());
                        for (FormPageRule rule : question.getPageRules())
                        {
                            if (ruleValues.contains(rule.getOption()))
                            {
                                nextActivePage = _getNextActivePage(rule);
                            }
                        }
                    }
                }
            }
            
            FormPageRule rule = page.getRule();
            if (rule != null && nextActivePage == null)
            {
                nextActivePage = _getNextActivePage(rule);
            }
        }
        return activeQuestions;
    }

    private String _getNextActivePage(FormPageRule rule)
    {
        return rule.getType() == PageRuleType.FINISH
                ? "finish"
                : rule.getPageId();
    }

    private List<String> _getRuleValues(FormEntryValues entryValues, String nameForForm)
    {
        Object ruleValue =  entryValues.getValue(nameForForm);
        if (ruleValue == null)
        {
            return ListUtils.EMPTY_LIST;
        }
        
        if (ruleValue.getClass().isArray())
        {
            String[] stringArray = ArrayUtils.toStringArray((Object[]) ruleValue);
            return Arrays.asList(stringArray);
        }
        else
        {
            return List.of(ruleValue.toString());
        }
    }
}
