/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.menu;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.ClientSideElementHelper;
import org.ametys.core.ui.SimpleMenu;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.util.I18nizableTextKeyComparator;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.FormQuestionTypeExtensionPoint;
import org.ametys.plugins.forms.question.types.RichTextQuestionType;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * The form question button menu configuration class
 */
public class FormQuestionButtonMenu extends SimpleMenu 
{
    /** The form question type extension point. */
    protected FormQuestionTypeExtensionPoint _formQuestionTypeExtensionPoint;
    
    /** Repository content */
    protected AmetysObjectResolver _resolver;
    
    /** The observation manager */
    protected ObservationManager _observationManager;
    
    private boolean _galleryInitialized;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) serviceManager.lookup(ObservationManager.ROLE);
        _formQuestionTypeExtensionPoint = (FormQuestionTypeExtensionPoint) serviceManager.lookup(FormQuestionTypeExtensionPoint.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        try
        {
            _lazyInitializeContentViewGallery();
        }
        catch (Exception e)
        {
            throw new IllegalStateException("Unable to lookup client side element local components", e);
        }
        
        List<Script> scripts = new ArrayList<>();
        for (Script script : super.getScripts(ignoreRights, contextParameters))
        {
            Script cloneScript = ClientSideElementHelper.cloneScript(script);
            cloneScript.getScriptFiles().add(new ScriptFile("/plugins/forms/resources/js/Ametys/plugins/forms/controllers/FormController.js"));
            scripts.add(cloneScript);
        }
        
        return scripts;
    }
    
    
    private synchronized void _lazyInitializeContentViewGallery() throws ConfigurationException
    {
        if (!_galleryInitialized)
        {
            Map<I18nizableText, Set<FormQuestionType>> typesByGroup = _getQuestionTypeByGroup();
            
            if (typesByGroup.size() > 0)
            {
                GalleryItem galleryItem = new GalleryItem();
                
                for (I18nizableText groupLabel : typesByGroup.keySet())
                {
                    GalleryGroup galleryGroup = new GalleryGroup(groupLabel);
                    galleryItem.addGroup(galleryGroup);
                    
                    Set<FormQuestionType> cTypes = typesByGroup.get(groupLabel);
                    for (FormQuestionType questionType : cTypes)
                    {
                        String id = this.getId() + "-" + questionType.getId();
                        
                        Configuration conf = _getQuestionTypeItemConfiguration (id, questionType);
                        _getGalleryItemManager().addComponent(_pluginName, null, id, StaticClientSideElement.class, conf);
                        galleryGroup.addItem(new UnresolvedItem(id, true));
                    }
                }
                _galleryItems.add(galleryItem);
            }
            
            if (_galleryItems.size() > 0)
            {
                try
                {
                    _getGalleryItemManager().initialize();
                }
                catch (Exception e)
                {
                    throw new ConfigurationException("Unable to lookup parameter local components", e);
                }
            }
        }
        
        _galleryInitialized = true;
    }
    
    /**
     * Get the configuration for a question type item
     * @param itemId The item id
     * @param questionType the question type
     * @return The configuration
     */
    protected Configuration _getQuestionTypeItemConfiguration (String itemId, FormQuestionType questionType)
    {
        DefaultConfiguration conf = new DefaultConfiguration("extension");
        conf.setAttribute("id", itemId);
        
        DefaultConfiguration classConf = new DefaultConfiguration("class");
        classConf.setAttribute("name", "Ametys.plugins.forms.controllers.FormController");
        
        // Label
        DefaultConfiguration labelConf = new DefaultConfiguration("label");
        I18nizableText label = questionType.getLabel();
        labelConf.setAttribute("i18n", "true");
        labelConf.setValue(label.getCatalogue() + ":" + label.getKey());
        classConf.addChild(labelConf);
        
        // Description
        DefaultConfiguration descConf = new DefaultConfiguration("description");
        I18nizableText desc = questionType.getDescription();
        descConf.setAttribute("i18n", "true");
        descConf.setValue(desc.getCatalogue() + ":" + desc.getKey());
        classConf.addChild(descConf);

        // Icons or glyph
        DefaultConfiguration iconGlyphConf = new DefaultConfiguration("icon-glyph");
        iconGlyphConf.setValue(questionType.getIconGlyph());
        classConf.addChild(iconGlyphConf);
        
        // Common configuration
        @SuppressWarnings("unchecked")
        Map<String, Object> commonConfig = (Map<String, Object>) this._script.getParameters().get("items-config");
        for (String tagName : commonConfig.keySet())
        {
            DefaultConfiguration c = new DefaultConfiguration(tagName);
            Object value = commonConfig.get(tagName);
            if (value instanceof I18nizableText i18nText)
            {
                c.setAttribute("i18n", String.valueOf(i18nText.isI18n()));
                c.setValue(i18nText.isI18n() ? i18nText.getKey() : i18nText.getLabel());
            }
            else
            {
                c.setValue(String.valueOf(commonConfig.get(tagName)));
            }
            classConf.addChild(c);
        }
        
        if (!(questionType instanceof RichTextQuestionType))
        {
            DefaultConfiguration noEntryOnlyConf = new DefaultConfiguration("enable-on-noentry-only");
            noEntryOnlyConf.setValue("true");
            classConf.addChild(noEntryOnlyConf);
        }
        
        // Type Id
        DefaultConfiguration typeIdConf = new DefaultConfiguration("type-id");
        typeIdConf.setValue(questionType.getId());
        classConf.addChild(typeIdConf);
        
        conf.addChild(classConf);
        return conf;
    }
    
    /**
     * Get the list of question types classified by groups
     * @return The map of form question types
     */
    protected Map<I18nizableText, Set<FormQuestionType>> _getQuestionTypeByGroup ()
    {
        Map<I18nizableText, Set<FormQuestionType>> groups = new TreeMap<>(new I18nizableTextKeyComparator());
        
        Set<String> questionTypesIds = _formQuestionTypeExtensionPoint.getExtensionsIds();
        for (String questionTypeId : questionTypesIds)
        {
            FormQuestionType questionType = _formQuestionTypeExtensionPoint.getExtension(questionTypeId);
            _script.getScriptFiles().addAll(questionType.getScripts());
            addType (questionType, groups);
        }
        
        return groups;
    }
    
    /**
     * Add question type to groups
     * @param type The form question type
     * @param groups The groups
     */
    protected void addType (FormQuestionType type, Map<I18nizableText, Set<FormQuestionType>> groups)
    {
        I18nizableText group = type.getCategory();
        if (group.isI18n() && group.getKey().isEmpty()
            || !group.isI18n() && group.getLabel().isEmpty())
        {
            group = new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTION_TYPE_CATEGORY_90_OTHERS");
        }
        
        if (!groups.containsKey(group))
        {
            groups.put(group, new TreeSet<>(new QuestionTypeComparator()));
        }
        
        Set<FormQuestionType> types = groups.get(group);
        types.add(type);
    }
    
    class QuestionTypeComparator implements Comparator<FormQuestionType>
    {
        @Override
        public int compare(FormQuestionType type1, FormQuestionType type2)
        {
            Integer order1 = type1.getDisplayOrder();
            Integer order2 = type2.getDisplayOrder();
            
            int compareTo = order1.compareTo(order2);
            if (compareTo == 0)
            {
                I18nizableText t1 = type1.getLabel();
                I18nizableText t2 = type2.getLabel();
                
                String str1 = t1.isI18n() ? t1.getKey() : t1.getLabel();
                String str2 = t2.isI18n() ? t2.getKey() : t2.getLabel();
                compareTo = str1.toString().compareTo(str2.toString());
                if (compareTo == 0)
                {
                    // Content types have same keys but there are not equals, so do not return 0 to add it in TreeSet
                    // Indeed, in a TreeSet implementation two elements that are equal by the method compareTo are, from the standpoint of the set, equal 
                    return 1;
                }
            }
            return compareTo;
        }
    }
}
