/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.menu;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.StringUtils;

import org.ametys.core.ui.SimpleMenu;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.Workflow;

/**
 * The form workflow button menu configuration class
 */
public class FormWorkflowButtonMenu extends SimpleMenu 
{
    /** The button extension id */
    public static final String BUTTON_EXTENSION_ID = "org.ametys.plugins.forms.workflow.button.menu";
    
    private static final String _NO_WORKFLOW_ID = "ametys_no_workflow";
    
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    /** The Workflow provider */
    private WorkflowProvider _workflowProvider;
    
    private boolean _galleryInitialized;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _workflowProvider = (WorkflowProvider) serviceManager.lookup(WorkflowProvider.ROLE);
        _workflowHelper = (WorkflowHelper) serviceManager.lookup(WorkflowHelper.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        try
        {
            _lazyInitializeContentViewGallery();
        }
        catch (Exception e)
        {
            throw new IllegalStateException("Unable to lookup client side element local components", e);
        }
        
        return super.getScripts(ignoreRights, contextParameters);
    }
    
    private synchronized void _lazyInitializeContentViewGallery() throws ConfigurationException
    {
        if (!_galleryInitialized)
        {
            GalleryGroup galleryGroup = new GalleryGroup(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_BUTTON_MENU_GROUP_LABEL"));
            GalleryItem galleryItem = new GalleryItem();
            galleryItem.addGroup(galleryGroup);

            String noWorkflowId = this.getId() + "." + _NO_WORKFLOW_ID;
            Configuration confNoWorkflow = _getWorkflowItemConfiguration(noWorkflowId, null);
            _getGalleryItemManager().addComponent(_pluginName, null, noWorkflowId, StaticClientSideElement.class, confNoWorkflow);
            galleryGroup.addItem(new UnresolvedItem(noWorkflowId, true));
            
            Workflow workflow = _workflowProvider.getAmetysObjectWorkflow();
            String[] workflowNames = workflow.getWorkflowNames();
            for (String workflowName : workflowNames)
            {
                if (workflowName.startsWith("entry-form-"))
                {
                    String id = this.getId() + "." + workflowName;
                    Configuration conf = _getWorkflowItemConfiguration (id, workflowName);
                    _getGalleryItemManager().addComponent(_pluginName, null, id, StaticClientSideElement.class, conf);
                    galleryGroup.addItem(new UnresolvedItem(id, true));
                }
            }
            
            _galleryItems.add(galleryItem);
        
            try
            {
                _getGalleryItemManager().initialize();
            }
            catch (Exception e)
            {
                throw new ConfigurationException("Unable to lookup parameter local components", e);
            }
        }
        
        _galleryInitialized = true;
    }
    
    /**
     * Reset the workflow galery
     */
    public void reset()
    {
        _initializeGalleryItemManager();
        _galleryItems.clear();
        _galleryInitialized = false;
    }
    
    /**
     * Get the configuration for a workflow
     * @param id the id of the button
     * @param workflowName the workflow name. Can be null for no workflow configuration.
     * @return The configuration
     */
    protected Configuration _getWorkflowItemConfiguration (String id, String workflowName)
    {
        DefaultConfiguration conf = new DefaultConfiguration("extension");
        conf.setAttribute("id", id);
        
        DefaultConfiguration classConf = new DefaultConfiguration("class");
        classConf.setAttribute("name", "Ametys.ribbon.element.ui.ButtonController");
        
        // Workflow name
        DefaultConfiguration workflowConf = new DefaultConfiguration("workflowName");
        workflowConf.setValue(workflowName);
        classConf.addChild(workflowConf);
        
        // Label
        DefaultConfiguration labelConf = new DefaultConfiguration("label");
        I18nizableText label = StringUtils.isNotBlank(workflowName) ? _workflowHelper.getWorkflowLabel(workflowName) : new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORMS_EDITOR_WORKFLOW_NO_WORKFLOW");
        labelConf.setAttribute("i18n", "true");
        labelConf.setValue(label.getCatalogue() + ":" + label.getKey());
        classConf.addChild(labelConf);
        
        // Description
        DefaultConfiguration descConf = new DefaultConfiguration("description");
        descConf.setAttribute("i18n", "true");
        descConf.setValue(label.getCatalogue() + ":" + label.getKey());
        classConf.addChild(descConf);

        // Icons or glyph
        DefaultConfiguration iconGlyphConf = new DefaultConfiguration("icon-glyph");
        iconGlyphConf.setValue("ametysicon-workflow");
        classConf.addChild(iconGlyphConf);
        
        // Action
        DefaultConfiguration actionConf = new DefaultConfiguration("action");
        actionConf.setValue("Ametys.plugins.forms.actions.FormsActions.setWorkflow");
        classConf.addChild(actionConf);
        
        // Toggle button
        DefaultConfiguration toggleEnabledConf = new DefaultConfiguration("toggle-enabled");
        toggleEnabledConf.setValue("true");
        classConf.addChild(toggleEnabledConf);
        
        // Common configuration
        @SuppressWarnings("unchecked")
        Map<String, Object> commonConfig = (Map<String, Object>) this._script.getParameters().get("items-config");
        for (String tagName : commonConfig.keySet())
        {
            DefaultConfiguration c = new DefaultConfiguration(tagName);
            c.setValue(String.valueOf(commonConfig.get(tagName)));
            classConf.addChild(c);
        }
        
        conf.addChild(classConf);
        return conf;
    }
}
