/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.ui.ClientSideElement.ScriptFile;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;

import com.google.common.collect.Multimap;

/**
 * Interface for question's types
 */
public interface FormQuestionType
{
    /** 
     * Get id of the question type 
     * @return the id
     */
    public String getId();
    
    /** 
     * Get label of question type 
     * @return the label
     */
    public I18nizableText getLabel();
    
    /** 
     * Get description of question type 
     * @return the description 
     */
    public I18nizableText getDescription();
    
    /** 
     * Get display order of the question type 
     * @return the display order
     */
    public Integer getDisplayOrder();
    
    /** 
     * Get icon of question type button 
     * @return the icon-glyph name 
     */
    public String getIconGlyph();
    
    /**
     * Get the scripts to import
     * @return the list of scripts
     */
    public List<ScriptFile> getScripts();
    
    /**
     * Get the category of question type
     * @return the name of the category in questionTypeButtonMenu
     */
    public I18nizableText getCategory();
    
    /**
     * Get the display XSLT file
     * @return the display XSLT file
     */
    public String getDisplayXSLT();
    
    /**
     * Get the form question view
     * @param form the form
     * @return a view of question model
     */
    public View getView(Form form);
    
    /**
     * Get the form question model
     * @return the form question model
     */
    public Model getModel();

    /**
     * Get the form entry model for the question type
     * @param question the question
     * @return the form entry model
     */
    public Model getEntryModel(FormQuestion question);
    
    /**
     * Get the storage type of the question type
     * @param question the question
     * @return the storage type
     */
    public String getStorageType(FormQuestion question);
    
    /**
     * Get the default title depending on type 
     * @return the default title
     */
    public I18nizableText getDefaultTitle();
    
    /**
     * Sax additional informations for the question type
     * @param contentHandler the content handler
     * @param question the question
     * @throws SAXException if a saxing error occurred
     */
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException;

    /**
     * Validate values to create or edit the question
     * @param values the values
     * @param errors the errors
     */
    public void validateQuestionValues(Map<String, Object> values, Map<String, I18nizableText> errors);
    
    /**
     * Validate values to create a form entry
     * @param question the question
     * @param values the values
     * @param errors the errors
     * @param currentStepId the current step id of the entry. Can be empty if the form has no workflow
     * @param additionalParameters the additional parameters for validation
     */
    public void validateEntryValues(FormQuestion question, Map<String, Object> values, Multimap<String, I18nizableText> errors, Optional<Long> currentStepId, Map<String, Object> additionalParameters);
    
    /**
     * Do additional operations to create or edit the question
     * @param question the question
     * @param values the values
     */
    public void doAdditionalOperations(FormQuestion question, Map<String, Object> values);

    /**
     * Get field to disable if form is published
     * @param question the form question
     * @return the list of field name
     */
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question);
    
    /**
     * Sax the entry value depending on the question type
     * @param contentHandler the content handler
     * @param question the question getting saxed 
     * @param entry the entry
     * @throws SAXException exception while saxing entry
     */
    public void saxEntryValue(ContentHandler contentHandler, FormQuestion question, FormEntry entry) throws SAXException;
    
    /** 
     * Say if question needs more configuration to be used
     * @param question the current question
     * @return false if question needs more configuration, true by default
     */
    public boolean isQuestionConfigured(FormQuestion question);
    
    /**
     * Get the js renderer for the choice list
     * @param question the question
     * @return the js renderer
     */
    public default String getJSRenderer(FormQuestion question)
    {
        return null;
    }

    /**
     * Get the js converter for the choice list
     * @param question the question
     * @return the js converter
     */
    public default String getJSConverter(FormQuestion question)
    {
        return null;
    }
    
    /**
     * Convert the entry value into a JSON object to use client side
     * @param value the value to convert
     * @param question the form question
     * @param entry the form entry
     * @param modelItem the model item
     * @return The value as JSON
     * @throws Exception if an error occurred
     */
    public default Object valueToJSONForClient(Object value, FormQuestion question, FormEntry entry, ModelItem modelItem) throws Exception
    {
        DataContext context = RepositoryDataContext.newInstance()
                .withObject(entry)
                .withDataPath(modelItem.getPath());
        
        return entry.dataToJSON(modelItem.getName(), context);
    }
    
    /**
     * Return <code>true</code> if the type of question is only for display.
     * @param question the question
     * @return <code>true</code> if the type of question is only for display.
     */
    public default boolean onlyForDisplay(FormQuestion question)
    {
        return false;
    }
    
    /**
     * Return <code>true</code> if the type of question can be answered by user.
     * @param question the question
     * @return <code>true</code> if the type of question can be answered by user.
     */
    public default boolean canBeAnsweredByUser(FormQuestion question)
    {
        return true;
    }

    /**
     * Return <code>true</code> if the type of question can be cacheable.
     * @param question the question
     * @return <code>true</code> if the type of question can be cacheable.
     */
    public default boolean isCacheable(FormQuestion question)
    {
        return true;
    }
}
