/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.sources;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.core.search.UserAndGroupSearchManager;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;

/**
 * Class for creating users choice list from a population
 */
public class UsersSourceType extends AbstractSourceType 
{
    /** Constant for element definition name of the population id*/
    public static final String ATTRIBUTE_POPULATION_ID = "population-id";
    
    /** Map of ModelItems specific to UsersSourceType */
    protected Map<String, ModelItem> _usersSourceItems;
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The user and group search manager */
    protected UserAndGroupSearchManager _userAndGroupSearchManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _userAndGroupSearchManager = (UserAndGroupSearchManager) manager.lookup(UserAndGroupSearchManager.ROLE);
    }
    
    public Map<String, ModelItem> getModelItems()
    {
        _usersSourceItems = new HashMap<>();
        
        ElementDefinition<String> population = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_POPULATION_ID, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_USERS_POPULATION", "PLUGINS_FORMS_QUESTIONS_DIALOG_CHOICE_USERS_POPULATION_DESC", new DefaultValidator(null, true));
        population.setWidget("edition.userpopulation");
        _usersSourceItems.put(population.getName(), population);
       
        return _usersSourceItems;
    }

    public List<ViewItem> getViewItems()
    {
        List<ViewItem> viewElements = new ArrayList<>();
        
        ViewElement grid = new ViewElement();
        grid.setDefinition((ElementDefinition< ? >) _usersSourceItems.get(ATTRIBUTE_POPULATION_ID));
        viewElements.add(grid);
        
        return viewElements;
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished()
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished();
        fieldNames.add(ATTRIBUTE_POPULATION_ID);
        return fieldNames;
    }

    public boolean remoteData()
    {
        return true;
    }

    public I18nizableText getEntry(ChoiceOption value, Map<String, Object> contextParams) throws Exception
    {
        UserIdentity id = (UserIdentity) value.getValue();
        User user = _userManager.getUser(id);
        return new I18nizableText(user.getSortableName());
    }

    public Map<ChoiceOption, I18nizableText> getTypedEntries(Map<String, Object> contextParams) throws Exception
    {
        throw new UnsupportedOperationException("Method getTypedEntries can't be called for UsersSourceType");
    }

    public Map<ChoiceOption, I18nizableText> searchEntries(Map<String, Object> contextParams, int limit, Object paginationData) throws Exception
    {
        Map<ChoiceOption, I18nizableText> entries = new LinkedHashMap<>();
        
        FormQuestion question = _getQuestionFromParam(contextParams);
        String populationId = question.getValue(ATTRIBUTE_POPULATION_ID);
     
        Map<String, Object> paginationParam = new HashMap<>();
        int offset = paginationData != null ? ((int) paginationData - 1) * limit : 0;
        paginationParam.put("users", Map.of(populationId, offset));
        
        Map<String, Object> search = _userAndGroupSearchManager.searchUsersAndGroup(Set.of(populationId), Set.of(), limit, paginationParam, contextParams, true);
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> results = (List<Map<String, Object>>) search.get("results");
        for (Map<String, Object> member : results)
        {
            String id = member.get("login") + "#" + member.get("populationId");
            String label = (String) member.get("sortablename");
            ChoiceOption choice = new ChoiceOption(id);
            entries.put(choice, new I18nizableText(label));
        }

        return entries;
    }
    
    @Override
    public String getStorageType(FormQuestion question)
    {
        return org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID;
    }
    
    @Override
    public String getJSRenderer()
    {
        return "Ametys.plugins.forms.helper.SearchEntriesGridHelper.renderUserChoiceList";
    }
    
    @Override
    public Object removeEmptyOrOtherValue(Object value)
    {
        if (value == null)
        {
            return null;
        }
        
        if (value.getClass().isArray())
        {
            List<UserIdentity> newVal = new ArrayList<>();
            for (UserIdentity val : (UserIdentity[]) value)
            {
                if (val != null)
                {
                    newVal.add(val);
                }
            }
            
            if (newVal.isEmpty())
            {
                return null;
            }
            
            return newVal.toArray(new UserIdentity[newVal.size()]);
        }
        else
        {
            return value;
        }
    }

    @Override
    public Object valueToJSONForClient(Object value, FormQuestion question, FormEntry entry, ModelItem modelItem) throws Exception
    {
        return _getComputedComplexValue(value, question, entry);
    }
    
    @Override
    public String value2String(Object value)
    {
        if (value instanceof UserIdentity user)
        {
            return UserIdentity.userIdentityToString(user);
        }
        
        return super.value2String(value);
    }
}
