/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.forms.question.types;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.ui.ClientSideElement.ScriptFile;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.core.ui.util.ConfigurationHelper;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.question.FormQuestionDataTypeExtensionPoint;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.validators.NameForFormValidator;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.repository.type.AbstractRuleElementType;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.model.ModelViewItemGroup;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.ViewItemGroup;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.parameter.DefaultValidator;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

import com.google.common.collect.Multimap;

/**
 * Static class for creating {@link FormQuestionType} from xml config
 */
public abstract class AbstractStaticFormQuestionType extends AbstractLogEnabled implements FormQuestionType, Configurable, PluginAware, Serviceable
{
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    /** Id of question type */
    protected String _id;
    /** Label of question type */
    protected I18nizableText _label;
    /** Description of question type */
    protected I18nizableText _description;
    /** Icon glyph of question type */
    protected String _iconGlyph;
    /** Forms */
    protected String _pluginName;
    /** Scripts associated with question type */
    protected List<ScriptFile> _scripts;
    /** Xslt associated of question type */
    protected String _xslt;
    /** The form question model */
    protected Model _formQuestionModel;
    /** The type category in button menu */
    protected I18nizableText _category;
    /** The type priority in button menu */
    protected Integer _priority;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _id = configuration.getAttribute("id");
       
        Configuration childLabel = configuration.getChild("label");
        _label = I18nizableText.getI18nizableTextValue(childLabel, "plugin." + _pluginName, childLabel.getValue());
        
        Configuration childDesc = configuration.getChild("description");
        _description = I18nizableText.getI18nizableTextValue(childDesc, "plugin." + _pluginName, childDesc.getValue());
        
        Configuration childConfig = configuration.getChild("category");
        _category = I18nizableText.parseI18nizableText(childConfig, "plugin." + _pluginName, "");
        
        _priority = Integer.parseInt(configuration.getChild("order").getValue());
       
        _iconGlyph = configuration.getChild("icon-glyph").getValue();
        
        Configuration scriptsConf = configuration.getChild("scripts");
        _scripts = ConfigurationHelper.parsePluginResourceList(scriptsConf, _pluginName, getLogger());
        
        _xslt = configuration.getChild("xslt").getValue();
    }

    public String getId()
    {
        return _id;
    }

    public I18nizableText getLabel()
    {
        return _label;
    }

    public I18nizableText getDescription()
    {
        return _description;
    }
    
    public Integer getDisplayOrder()
    {
        return _priority;
    }
    public String getIconGlyph()
    {
        return _iconGlyph;
    }

    public List<ScriptFile> getScripts()
    {
        return _scripts;
    }
    
    public String getDisplayXSLT()
    {
        return _xslt;
    }
    
    public I18nizableText getCategory()
    {
        return _category;
    }
    
    public Model getModel()
    {
        if (_formQuestionModel == null)
        {
            try
            {
                List<ModelItem> items = _getModelItems();
                ModelItem[] modelItems = items.toArray(new ModelItem[items.size()]);
                _formQuestionModel = Model.of(
                        getId() + ".model.id", 
                        getId() + ".family.id",
                        modelItems
                    );
                        
            }
            catch (Exception e) 
            {
                getLogger().error("An error occurred getting the form question model", e);
                throw new RuntimeException("An error occurred getting the form question model", e);
            }
        }
        return _formQuestionModel;
    }
    
    public Model getEntryModel(FormQuestion question)
    {
        try
        {
            ModelItem item = _getEntryModelItem(question);
            String name = question.getNameForForm();
            return Model.of(
                    getId() + "." + name + ".model.entry.id", 
                    getId() + "." + name + ".family.entry.id",
                    item
                );
        }
        catch (Exception e) 
        {
            getLogger().error("An error occurred getting the form entry model", e);
            throw new RuntimeException("An error occurred getting the form entry model", e);
        }
    }
    
    /**
     * Get the list of ModelItems
     * @return a list of ModelItems
     */
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = new ArrayList<>();
        
        ElementDefinition<String> title = FormElementDefinitionHelper.getElementDefinition(FormQuestion.ATTRIBUTE_TITLE, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_QUESTION", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_QUESTION_DESC", new DefaultValidator(null, true));
        title.setDefaultValue(_i18nUtils.translate(getDefaultTitle()));
        modelItems.add(title);
        
        //TYPE
        ElementDefinition type = FormElementDefinitionHelper.getElementDefinition(FormQuestion.ATTRIBUTE_TYPE, ModelItemTypeConstants.STRING_TYPE_ID, null, null, new DefaultValidator(null, true));
        modelItems.add(type);
        
        //NAME
        ElementDefinition nameForForm = FormElementDefinitionHelper.getElementDefinition(FormQuestion.ATTRIBUTE_NAME_FOR_FORM, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_NAME_FOR_FORMS", "PLUGINS_FORMS_QUESTIONS_DIALOG_NAME_FOR_FORMS_DESC", new NameForFormValidator("^[a-zA-Z0-9_-]*$", true));
        modelItems.add(nameForForm);
        
        //RULE SOURCE
        ElementDefinition rule = FormElementDefinitionHelper.getElementDefinition(FormQuestion.ATTRIBUTE_RULE, AbstractRuleElementType.RULE_REPOSITORY_DATA_TYPE, "PLUGINS_FORMS_QUESTIONS_DIALOG_RULE_TITLE", "PLUGINS_FORMS_QUESTIONS_DIALOG_RULE_TITLE_DESC", null);
        rule.setWidget("edition.question-rules");
        RepeaterDefinition rulesRepeater = FormElementDefinitionHelper.getRepeaterDefinition(FormQuestion.ATTRIBUTE_RULES, new ElementDefinition[]{rule}, "PLUGINS_FORMS_QUESTIONS_DIALOG_RULES_TITLE", "PLUGINS_FORMS_QUESTIONS_DIALOG_RULES_TITLE_DESC", "PLUGINS_FORMS_QUESTIONS_DIALOG_RULES_ADD_LABEL", "PLUGINS_FORMS_QUESTIONS_DIALOG_RULES_DELETE_LABEL", 1);
        modelItems.add(rulesRepeater);
        
        return modelItems;
    }
    
    public View getView(Form form)
    {
        View view = new View();
        
        view.setLabel(getLabel());
        view.setDescription(getDescription());
        view.setIconGlyph(getIconGlyph());
        
        // iterate to avoid casting List<ViewItemGroup> to List<ViewItem>
        for (ViewItem item : _getTabs(form))
        {
            view.addViewItem(item);
        }
        
        return view;
    }
    
    /**
     * Get the list of tabs that will be added in the question view.
     * The list returned by this method should include the tab provided by _getTabs
     * @param form the form
     * @return the tabs
     */
    protected abstract List<ViewItemGroup> _getTabs(Form form);

    /**
     * Helper to create an empty tab to use as the main tab
     * @return an empty tab
     */
    protected SimpleViewItemGroup _createMainTab() 
    {
        SimpleViewItemGroup fieldset = new SimpleViewItemGroup();
        fieldset.setName("general");
        fieldset.setLabel(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_MAIN"));
        fieldset.setRole(ViewItemGroup.TAB_ROLE);
        
        return fieldset;
    }
    
    /**
     * Helper to create an empty tab to use as the advanced tab
     * @return an empty advanced tab
     */
    protected SimpleViewItemGroup _createAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = new SimpleViewItemGroup();
        advancedFieldset.setName("advanced");
        advancedFieldset.setLabel(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_ADVANCED"));
        advancedFieldset.setRole(ViewItemGroup.TAB_ROLE);

        return advancedFieldset;
    }
    
    /**
     * Get the view's common rules tab 
     * @return the rules tab as SimpleViewItemGroup
     */
    protected SimpleViewItemGroup _getRulesTab() 
    {
        SimpleViewItemGroup ruleFieldset = new SimpleViewItemGroup();
        ruleFieldset.setName("rules");
        ruleFieldset.setLabel(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RULE_FIELDSET"));
        ruleFieldset.setDescription(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_QUESTION_RULE_FIELDSET_DESC"));
        ruleFieldset.setRole(ViewItemGroup.TAB_ROLE);
        
        ModelItemGroup rulesModelItem = (ModelItemGroup) getModel().getModelItem(FormQuestion.ATTRIBUTE_RULES);
        ModelViewItemGroup rules = ModelViewItemGroup.of(rulesModelItem);
        ruleFieldset.addViewItem(rules);
        
        return ruleFieldset;
    }
    
    /**
     * Define the entry model items
     * @param question the question
     * @return the entry model item
     */
    protected ModelItem _getEntryModelItem(FormQuestion question)
    {
        try
        {
            String storageType = getStorageType(question);
            ElementDefinition item = DefaultElementDefinition.of(question.getNameForForm(), false, storageType, FormQuestionDataTypeExtensionPoint.ROLE);
            item.setLabel(new I18nizableText(question.getTitle()));
            return item;
        }
        catch (Exception e)
        {
            throw new RuntimeException("Can't have entry model item for question id '" + question.getId() + "'", e);
        }
    }
    
    public boolean isQuestionConfigured(FormQuestion question)
    {
        return true;
    }
    
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException
    {
        // Do nothing by default
    }

    public void doAdditionalOperations(FormQuestion question, Map<String, Object> values)
    {
        // Do nothing by default
    }
    
    public void validateQuestionValues(Map<String, Object> values, Map<String, I18nizableText> errors)
    {
        String nameForForm = (String) values.get(FormQuestion.ATTRIBUTE_NAME_FOR_FORM);
        if (StringUtils.startsWith(nameForForm, "ametys-"))
        {
            errors.put(FormQuestion.ATTRIBUTE_NAME_FOR_FORM, new I18nizableText("plugin.forms", "PLUGINS_FORMS_FORMS_RENDER_ERROR_NAME_FOR_FORM"));
        }
    }
    
    public void validateEntryValues(FormQuestion question, Map<String, Object> values, Multimap<String, I18nizableText> errors, Optional<Long> currentStepId, Map<String, Object> additionalParameters)
    {
        boolean ignoreWriteRestriction = (boolean) additionalParameters.getOrDefault("ignoreWriteRestriction", false);
        
        if (!ignoreWriteRestriction && currentStepId.isPresent() && !question.canWrite(currentStepId.get()))
        {
            String nameForForm = question.getNameForForm();
            Object value = values.get(nameForForm);
            
            // write without rights
            if (value != null)
            {
                errors.put(nameForForm, new I18nizableText("plugin.forms", "PLUGINS_FORMS_VALIDATOR_RESTICTED_IN_WRITING"));
            }
        }
    }
    
    public void saxEntryValue(ContentHandler contentHandler, FormQuestion question, FormEntry entry) throws SAXException
    {
        String dataName = question.getNameForForm();
        ModelItem definition = entry.getDefinition(dataName);
        
        DataContext context = RepositoryDataContext.newInstance()
                                                   .withObject(entry)
                                                   .withDataPath(dataName);
        
        if (entry.hasValue(dataName))
        {
            definition.getType().valueToSAX(contentHandler, "value", entry.getValue(dataName), context);
        }
    }
    
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames = new ArrayList<>();
        fieldNames.add(FormQuestion.ATTRIBUTE_NAME_FOR_FORM);
        fieldNames.add(FormQuestion.ATTRIBUTE_TYPE);
        fieldNames.add(FormQuestion.ATTRIBUTE_RULES);
        return fieldNames;
    }

} 
