/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.rights;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.plugins.core.impl.right.AbstractHierarchicalAccessController;
import org.ametys.plugins.forms.dao.FormDirectoryDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormDirectory;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.SiteAwareAmetysObject;

/**
 * {@link AccessController} for a {@link Form}
 */
public class FormAccessController extends AbstractHierarchicalAccessController<AmetysObject> implements Contextualizable
{
    /** the form context category */
    public static final I18nizableText FORM_CONTEXT_CATEGORY = new I18nizableText("plugin.forms", "PLUGINS_FORMS_RIGHTS_CATEGORY");
    /** The form directory DAO */
    protected FormDirectoryDAO _formDirectoryDAO;
    private Context _context;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _formDirectoryDAO = (FormDirectoryDAO) manager.lookup(FormDirectoryDAO.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public boolean isSupported(Object object)
    {
        return object instanceof Form || object instanceof FormDirectory;
    }
    
    @Override
    protected boolean _isSupportedStoredContext(Object storedObject)
    {
        if (isSupported(storedObject))
        {
            String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
            return siteName == null
                    || storedObject instanceof SiteAwareAmetysObject ao && StringUtils.equals(ao.getSiteName(), siteName);
        }
        return false;
    }
    
    @Override
    protected Set<AmetysObject> _getParents(AmetysObject object)
    {
        AmetysObject parent = object.getParent();
        if (isSupported(parent))
        {
            return Collections.singleton(parent);
        }
        else
        {
            return null;
        }
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        Request request = ContextHelper.getRequest(_context);
        String siteName = WebHelper.getSiteName(request);
        if (StringUtils.isNotBlank(siteName) && workspacesContexts.contains("/cms"))
        {
            return Collections.singleton(_formDirectoryDAO.getFormDirectoriesRootNode(siteName));
        }
        return null;
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        if (object instanceof Form)
        {
            Map<String, I18nizableTextParameter> params = Map.of("title", getObjectLabel(object));
            return new I18nizableText("plugin.forms", "PLUGINS_FORMS_ACCESS_CONTROLLER_FORM_CONTEXT_EXPLANATION_LABEL", params);
        }
        else if (object instanceof FormDirectory)
        {
            Request request = ContextHelper.getRequest(_context);
            String siteName = WebHelper.getSiteName(request);
            if (_formDirectoryDAO.getFormDirectoriesRootNode(siteName).equals(object))
            {
                return new I18nizableText("plugin.forms", "PLUGINS_FORMS_ACCESS_CONTROLLER_ROOT_CONTEXT_EXPLANATION_LABEL");
            }
            else
            {
                Map<String, I18nizableTextParameter> params = Map.of("title", getObjectLabel(object));
                return new I18nizableText("plugin.forms", "PLUGINS_FORMS_ACCESS_CONTROLLER_DIRECTORY_CONTEXT_EXPLANATION_LABEL", params);
            }
        }
        throw new RightsException("Unsupported context " + object.toString());
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Form form)
        {
            return new I18nizableText(getFormDirectoryPathLabel(form.getParent()) + form.getTitle());
        }
        else if (object instanceof FormDirectory directory)
        {
            Request request = ContextHelper.getRequest(_context);
            String siteName = WebHelper.getSiteName(request);
            if (_formDirectoryDAO.getFormDirectoriesRootNode(siteName).equals(object))
            {
                return new I18nizableText("plugin.forms", "PLUGINS_FORMS_ACCESS_CONTROLLER_ROOT_CONTEXT_LABEL");
            }
            else
            {
                return new I18nizableText(getFormDirectoryPathLabel(directory.getParent()) + directory.getTitle());
            }
        }
        throw new RightsException("Unsupported context " + object.toString());
    }
    
    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        return FORM_CONTEXT_CATEGORY;
    }
    
    @Override
    public int getObjectPriority(Object object)
    {
        if (object instanceof FormDirectory)
        {
            Request request = ContextHelper.getRequest(_context);
            String siteName = WebHelper.getSiteName(request);
            if (_formDirectoryDAO.getFormDirectoriesRootNode(siteName).equals(object))
            {
                return 10;
            }
        }
        return super.getObjectPriority(object);
    }
    
    /**
     * Get the path as a label for a form directory
     * If the directory is the root, null is returned instead
     * @param directory the directory
     * @return the label
     */
    public static String getFormDirectoryPathLabel(FormDirectory directory)
    {
        // Do not handle the root directory as any other. We don't want to display it for every Forms
        if (directory.getParent() instanceof FormDirectory parent)
        {
            return getFormDirectoryPathLabel(parent) + directory.getTitle() + " > ";
        }
        else
        {
            return "";
        }
    }
}
