/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.rights;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractRightBasedAccessController;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormFactory;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.SiteAwareAmetysObject;

/**
 * {@link AccessController} to allow read access and handle for author of a {@link Form}
 */
public class FormAuthorAccessController extends AbstractRightBasedAccessController implements Serviceable
{
    private static final List<String> __CREATOR_RIGHTS = List.of(FormDAO.HANDLE_FORMS_RIGHT_ID);
    /** the ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public boolean isSupported(Object object)
    {
        return object instanceof Form;
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (((Form) object).getAuthor().equals(user))
        {
            return __CREATOR_RIGHTS.contains(rightId) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
        }
        
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return ((Form) object).getAuthor().equals(user) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
    }

    /**
     * If creator, access to a list of rights
     */
    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Map<String, AccessResult> permissionByRight = new HashMap<>();
        
        if (((Form) object).getAuthor().equals(user))
        {
            for (String rightId : __CREATOR_RIGHTS)
            {
                permissionByRight.put(rightId, AccessResult.USER_ALLOWED);
            }
        }
        
        return permissionByRight;
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    /**
     * If right requested is in the list, the creator is added the list of USER_ALLOWED
     */
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        Map<UserIdentity, AccessResult> permissionByUser = new HashMap<>();
        
        if (__CREATOR_RIGHTS.contains(rightId))
        {
            permissionByUser.put(((Form) object).getAuthor(), AccessResult.USER_ALLOWED);
        }
        return permissionByUser;
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    public AccessExplanation getStandardAccessExplanation(AccessResult permission, Object object)
    {
        switch (permission)
        {
            case USER_ALLOWED:
            case UNKNOWN:
                return new AccessExplanation(getId(), permission,
                        new I18nizableText("plugin.forms", "PLUGINS_FORMS_CREATOR_ACCESS_CONTROLLER_" + permission.name() + "_EXPLANATION",
                                Map.of("title", getObjectLabel(object))));
            default:
                return super.getStandardAccessExplanation(permission, object);
        }
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Form form)
        {
            return new I18nizableText(FormAccessController.getFormDirectoryPathLabel(form.getParent()) + form.getTitle());
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return FormAccessController.FORM_CONTEXT_CATEGORY;
    }

    @Override
    protected Iterable< ? extends Object> getHandledObjects(UserIdentity identity, Set<GroupIdentity> groups)
    {
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        
        if (StringUtils.isNotBlank(siteName))
        {
            UserExpression authorExpression = new UserExpression(Form.AUTHOR, Operator.EQ, identity);
            Expression siteExpression = new StringExpression(SiteAwareAmetysObject.METADATA_SITE, Operator.EQ, siteName);
            String query = QueryHelper.getXPathQuery(null, FormFactory.FORM_NODETYPE, new AndExpression(authorExpression, siteExpression));
            return _resolver.query(query);
        }
        
        return List.of();
    }
    
    @Override
    protected Collection<String> getHandledRights(UserIdentity identity, Set<GroupIdentity> groups, Object object)
    {
        return __CREATOR_RIGHTS;
    }
}
