/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.data.UserEntry;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.store.JdbcWorkflowStore;
import org.ametys.runtime.authentication.AccessDeniedException;

import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.spi.Step;

/**
 * This class creates multiple toggle buttons each representing a workflow step on content form entries
 */
public class FormEntriesWorkflowStepsClientSideElement extends AbstractFormEntriesWorkflowStepsClientSideElement
{
    /** The manager for the forms JCR representation */
    protected FormPropertiesManager _formPropertiesManager;
    /** The manager for the forms SQL tables */
    protected FormTableManager _formTableManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _formPropertiesManager = (FormPropertiesManager) serviceManager.lookup(FormPropertiesManager.ROLE);
        _formTableManager = (FormTableManager) serviceManager.lookup(FormTableManager.ROLE);
    }
    
    @Override
    protected String _getDefaultMenuClassName()
    {
        return "Ametys.plugins.forms.workflow.FormEntriesWorkflowMenu";
    }

    @Override
    protected String _getDefaultActionClassName()
    {
        return "Ametys.plugins.forms.workflow.FormEntriesWorkflowAction.doAction";
    }

    @Override
    protected List<ScriptFile> _getAdditionalScriptFiles()
    {
        List<ScriptFile> scriptFiles = new ArrayList<>();
        scriptFiles.add(new ScriptFile("/plugins/forms/resources/js/Ametys/plugins/forms/workflow/AbstractFormEntriesWorkflowAction.js"));
        scriptFiles.add(new ScriptFile("/plugins/forms/resources/js/Ametys/plugins/forms/workflow/FormEntriesWorkflowAction.js"));
        scriptFiles.add(new ScriptFile("/plugins/forms/resources/js/Ametys/plugins/forms/workflow/AbstractFormEntriesWorkflowMenu.js"));
        scriptFiles.add(new ScriptFile("/plugins/forms/resources/js/Ametys/plugins/forms/workflow/FormEntriesWorkflowMenu.js"));
        return scriptFiles;
    }
    
    @Override
    protected String _getSelectionTargetId()
    {
        return "^(content-form)$";
    }
    
    @SuppressWarnings("unchecked")
    @Override
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getWorkflowState(List<Object> entryIds, String formId, String scriptId)
    {
        // Start by checking right on content
        // Store the content for later use
        Content formContent = _getContentIfAllowed(formId);
        
        List<Map<String, Object>> entries = new ArrayList<> ();
        
        try
        {
            Form form = _formPropertiesManager.getForm(formId);
           
            List<UserEntry> allEntries = new ArrayList<> ();
            Workflow workflow = _workflowProvider.getExternalWorkflow(JdbcWorkflowStore.ROLE);
            
            List<Integer> entryIdsAsInt = entryIds.stream()
                .filter(Integer.class::isInstance)
                .map(Integer.class::cast)
                .collect(Collectors.toList());
            
            allEntries = _formTableManager.getSubmissions(form, new HashMap<>(), 0, Integer.MAX_VALUE, entryIdsAsInt);

            for (UserEntry entry : allEntries)
            {
                for (Script script : _scripts)
                {
                    if (script.getId().equals(scriptId))
                    {
                        Map<String, Object> entryMap = new HashMap<> ();
                        
                        Integer workflowId = entry.getWorkflowId();
                        if (workflowId != null)
                        {
                            List<Step> steps = workflow.getCurrentSteps(workflowId);
                            List<Integer> stepIds = new ArrayList<>();
                            for (Step step : steps)
                            {
                                stepIds.add(step.getStepId());
                            }
                            
                            Integer workflowStepId = (Integer) script.getParameters().get("workflow-step");
                            if (stepIds.contains(workflowStepId))
                            {
                                Map<String, Object> transientVars = new HashMap<>();
                                
                                // Used to check the rights on the form's content
                                transientVars.put(AbstractContentWorkflowComponent.CONTENT_KEY, formContent);
                                transientVars.put(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY, new ArrayList<> ());
                                
                                int[] availableActions = workflow.getAvailableActions(workflowId, transientVars);
                                Arrays.sort(availableActions);
                                
                                List<Integer> activeActions = new ArrayList<>();
                                for (int actionId : (List<Integer>) script.getParameters().get("workflow-actions-ids"))
                                {
                                    if (Arrays.binarySearch(availableActions, actionId) >= 0)
                                    {
                                        activeActions.add(actionId);
                                    }
                                }
                                
                                entryMap.put("id", entry.getId());
                                entryMap.put("actions", activeActions);
                                entries.add(entryMap);
                            }
                        }
                    }
                }
            }
        }
        catch (FormsException e)
        {
            getLogger().error("Error getting forms for a content.", e);
        }
        
        Map<String, Object> results = new HashMap<> ();
        results.put("entries", entries);
        return results;
    }

    private Content _getContentIfAllowed(String formId)
    {
        UserIdentity currentUser = _currentUserProvider.getUser();
        try
        {
            Content content = _formPropertiesManager.getFormContent(formId);
            if (_rightManager.hasReadAccess(currentUser, content))
            {
                return content;
            }
        }
        catch (FormsException e)
        {
            getLogger().error("An error occured while trying to retrieve the content of form '{}'", formId, e);
        }
        throw new AccessDeniedException("The user " + currentUser + " tried to access the workflow state of content form '" + formId + "' without sufficient rights");
    }
}
