/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.sitemap.PatternException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.core.group.Group;
import org.ametys.core.group.GroupDirectoryDAO;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.directory.GroupDirectory;
import org.ametys.core.right.Right;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightsExtensionPoint;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder.UserInput;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.helper.FormMailHelper;
import org.ametys.plugins.forms.helper.MailVariableParser;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.plugins.workflow.component.WorkflowArgument;
import org.ametys.plugins.workflow.support.WorkflowElementDefinitionHelper;
import org.ametys.plugins.workflow.support.WorkflowHelper.WorkflowVisibility;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;

import com.google.javascript.jscomp.jarjar.com.google.re2j.Pattern;
import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

import jakarta.mail.MessagingException;

/**
 * OS workflow function to send mail after an action is triggered.
 */
public class WorkflowFormSendMailFunction extends AbstractLogEnabled implements Component, EnhancedFunction, Serviceable, Contextualizable
{
    /** The component's role */
    public static final String ROLE =  WorkflowFormSendMailFunction.class.getName();
    /** The prefix for this function's workflow arguments' name */
    public static final String FUNCTION_PARAM_PREFIX = ROLE + "-";
    /** Variable for dashboard admin */
    public static final String FORM_DASHBOARD_ADMIN = "form.dashboardAdmin";
    /** Variable for submissions */
    public static final String FORM_DASHBOARD = "form.dashboard";
    /** Variable for the entry's number */
    public static final String ENTRY_NUMBER = "entryNumber";
    /** Variable for the form's title */
    public static final String FORM_TITLE = "form.title";
    /** Variable for entry attribute prefix */
    public static final String ENTRY = "entry.";
    /** Variable for entry's submitter */
    public static final String USER = "user";
    /** Sender argument of the email */
    public static final String ARG_SENDER = "sender";
    /** Email's recipient selection argument */
    public static final String ARG_RECIPIENT = "recipient";
    /** Manual input argument for email recipient */
    public static final String ARG_MANUAL_RECIPIENT = "manual-recipient";
    /** Group argument for email recipient selection*/
    public static final String ARG_RIGHTS_RECIPIENT = "right-recipient";
    /** Right argument for email recipient selection*/
    public static final String ARG_GROUP_RECIPIENT = "group-recipient";
    /** Default name for the rights category of standards form workflows */
    public static final String FORMS_RIGHTS_CATEGORY = "plugin.forms:FORMS_DEFAULT_WORKFLOW_ENTRY_RIGHTS_CATEGORY";
    /** Subject argument of the email */
    public static final String ARG_SUBJECT = "subject";
    /** Body argument of the email */
    public static final String ARG_BODY = "body";
    /** Link argument of the email */
    public static final String ARG_LINK = "link";
    /** Boolean argument for adding entry detail at the end of the email */
    public static final String ARG_ENTRY_DETAIL = "entry-detail";
    /** Comment argument of the email */
    public static final String ARG_COMMENT = "comment";
    /** The request key to ignore right */
    public static final String IGNORE_RIGHT_KEY = "ignore-right";

    private static final String __USER_GROUP_RECIPIENT_ARG = "user-group";
    private static final String __MANUAL_RECIPIENT_ARG = "manual";
    private static final String __RIGHTS_RECIPIENT_ARG = "rights";
    private static final String __USER_RECIPIENT_ARG = "user";
    private static final String __REGEX_EMAIL = "^([a-zA-Z0-9_\\.\\-\\+])+\\@(([a-zA-Z0-9\\-])+\\.)+([a-zA-Z0-9]{2,4})+$";
    private static final String __SUBMISSIONS = "submissions";
    private static final String __DASHBOARD = "dashboard";

    private static final String __WITH_RESTRICTION = "with-restriction";
    private static final String __WITHOUT_RESTRICTION = "without-restriction";
    private static final String __NO_DATA = "no-data";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The user manager */
    protected UserManager _userManager;

    /** The source resolver. */
    protected SourceResolver _sourceResolver;

    /** The context */
    protected Context _context;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** I18nUtils */
    protected I18nUtils _i18nUtils; 
    
    /** The workflow form mail EP */
    protected WorkflowFormMailExtensionPoint _workflowFormMailEP;
    
    /** The mail variable parser */
    protected MailVariableParser _mailVariableParser;
    
    /** The form mail helper */
    protected FormMailHelper _formMailHelper;
    
    /** The rights extension point */ 
    protected RightsExtensionPoint _rightsExtensionPoint;
   
    /** The rights manager. */
    protected RightManager _rightManager;
    
    /** The DAO for group directories */
    protected GroupDirectoryDAO _groupDirectoryDAO;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _workflowFormMailEP = (WorkflowFormMailExtensionPoint) manager.lookup(WorkflowFormMailExtensionPoint.ROLE);
        _mailVariableParser = (MailVariableParser) manager.lookup(MailVariableParser.ROLE);
        _formMailHelper = (FormMailHelper) manager.lookup(FormMailHelper.ROLE);
        _rightsExtensionPoint = (RightsExtensionPoint) manager.lookup(RightsExtensionPoint.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _groupDirectoryDAO = (GroupDirectoryDAO) manager.lookup(GroupDirectoryDAO.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        FormEntry formEntry = (FormEntry) transientVars.get(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY);
        
        String sender = _getSender(formEntry, transientVars, args);
        List<String> recipients = _getRecipients(formEntry, transientVars, args);
        try
        {
            String subject = _getSubject(formEntry, transientVars, args);
            String body = _getBody(formEntry, transientVars, args, subject);
            for (String recipient : recipients)
            {
                try
                {
                    SendMailHelper.newMail()
                        .withAsync(true)
                        .withSubject(subject)
                        .withHTMLBody(body)
                        .withSender(sender)
                        .withRecipient(recipient)
                        .sendMail();
                }
                catch (MessagingException | IOException e)
                {
                    getLogger().warn("Could not send a workflow notification mail to " + recipient, e);
                }
            }
        }
        catch (IOException e)
        {
            getLogger().warn("Could not send a workflow notification mail", e);
        }
    }
    
    /**
     * Get the sender email address from args, check for variables to translate
     * @param formEntry the form entry
     * @param transientVars the workflow transient vars
     * @param args the workflow args 
     * @return the sender's email address
     */
    protected String _getSender(FormEntry formEntry, Map transientVars, Map args)
    {
        String sender = _getStringParam(args, ARG_SENDER);
        String senderMail = _replaceText(sender, var -> _getReplacedAttribute(formEntry, var, true));
        String stripedEmail = StringUtils.trim(senderMail);
        if (StringUtils.isBlank(stripedEmail) || !Pattern.matches(__REGEX_EMAIL, stripedEmail))
        {
            getLogger().error("An error occured because email sender is invalid: '" + stripedEmail + "'. The configuration for sender '" + sender + "' must match with an unique email address");
        }
        return stripedEmail;
    }
    
    /**
     * Get the list of the recipients email addresses
     * check in the arguments and replace optional variables
     * @param formEntry the form entry
     * @param transientVars the workflow transient vars
     * @param args the workflow args 
     * @return the recipient's email
     */
    protected List<String> _getRecipients(FormEntry formEntry, Map transientVars, Map args)
    {
        List<String> emailAdresses = new ArrayList<>();
        
        String recipient = _getStringParam(args, ARG_RECIPIENT);
        switch (recipient)
        {
            case __MANUAL_RECIPIENT_ARG:
                emailAdresses.addAll(_getManualEmails(formEntry, args));
                break;
            case __RIGHTS_RECIPIENT_ARG:
                emailAdresses.addAll(_getEmailsFromRights(formEntry, args));
                break;
            case __USER_GROUP_RECIPIENT_ARG:
                emailAdresses.addAll(_getEmailsFromGroup(args));
                break;
            case __USER_RECIPIENT_ARG:
            default:
                emailAdresses.add(_getSubmitterValue(formEntry, true));
                break;
        }
        
        return emailAdresses;
    }

    /**
     * Get the e-mails from given group
     * @param args the arguments
     * @return the list of e-mail
     */
    protected List<String> _getEmailsFromGroup(Map args)
    {
        List<String> emailAdresses = new ArrayList<>();
        
        String recipientGroupValue = _getStringParam(args, ARG_GROUP_RECIPIENT);
        GroupIdentity groupIdentity = GroupIdentity.stringToGroupIdentity(recipientGroupValue);
        GroupDirectory groupDirectory = _groupDirectoryDAO.getGroupDirectory(groupIdentity.getDirectoryId());
        if (groupDirectory != null)
        {
            Group group = groupDirectory.getGroup(groupIdentity.getId());
            for (UserIdentity userIdentity : group.getUsers())
            {
                User user = _userManager.getUser(userIdentity);
                if (user != null)
                {
                    emailAdresses.add(user.getEmail());
                }
            }
        }
        
        return emailAdresses;
    }

    /**
     * Get manual e-mails 
     * @param formEntry the form entry
     * @param args the arguments
     * @return the list of manual e-mails
     */
    protected List<String> _getManualEmails(FormEntry formEntry, Map args)
    {
        List<String> emailAdresses = new ArrayList<>();
        
        String recipientManualValue = _getStringParam(args, ARG_MANUAL_RECIPIENT);
        String email = _replaceText(recipientManualValue, var -> _getReplacedAttribute(formEntry, var, true));
        if (StringUtils.isNotBlank(email))
        {
            String[] emails =  StringUtils.split(email, ",");
            for (String mail : emails)
            {
                String stripedEmail = StringUtils.trim(mail);
                if (StringUtils.isBlank(stripedEmail) || !Pattern.matches(__REGEX_EMAIL, stripedEmail))
                {
                    getLogger().error("An error occured because email recipient value is invalid: '" + stripedEmail + "'. The configuration for recipient '" + recipientManualValue + "' must match with an unique email address");
                }
                else
                {
                    emailAdresses.add(stripedEmail);
                }
            }
        }
        else
        {
            getLogger().error("An error occured because email recipient value is invalid: '" + email + "'. The configuration for recipient '" + recipientManualValue + "' must match with an unique email address");
        }
        
        return emailAdresses;
    }
    
    /**
     * Get the recipients' email addresses when filtering recipients by rights
     * @param formEntry the form entry
     * @param args the args containing the rights to filter
     * @return a list of emails
     */
    protected List<String> _getEmailsFromRights(FormEntry formEntry, Map args)
    {
        Form form = formEntry.getForm();
        String[] rights = _getStringParam(args, ARG_RIGHTS_RECIPIENT).split(",");
        
        Set<UserIdentity> users = _getUsersFromRights(form, new HashSet<>(Arrays.asList(rights)));
        
        return users.stream()
            .map(_userManager::getUser)
            .filter(Objects::nonNull)
            .map(User::getEmail)
            .filter(StringUtils::isNotBlank)
            .collect(Collectors.toList());
    }
    
    /**
     * Get the user identities who have given rights
     * @param form the form
     * @param rights the set of rights to check.
     * @return the users.
     */
    protected Set<UserIdentity> _getUsersFromRights(Form form, Set<String> rights)
    {
        Set<UserIdentity> users = new HashSet<>();
        
        Iterator<String> rightIt = rights.iterator();
        
        // First right : add all the granted users.
        if (rightIt.hasNext())
        {
            users.addAll(_rightManager.getAllowedUsers(StringUtils.trim(rightIt.next()), form).resolveAllowedUsers(Config.getInstance().getValue("runtime.mail.massive.sending")));
        }
        
        // Next rights : retain all the granted users.
        while (rightIt.hasNext())
        {
            users.retainAll(_rightManager.getAllowedUsers(StringUtils.trim(rightIt.next()), form).resolveAllowedUsers(Config.getInstance().getValue("runtime.mail.massive.sending")));
        }
        
        return users;
    }
    
    /**
     * Get the mail subject
     * @param formEntry the form entry
     * @param transientVars the workflow transient vars
     * @param args the workflow args 
     * @return the mail subject as string
     */
    protected String _getSubject(FormEntry formEntry, Map transientVars, Map args)
    {
        String subject = _getStringParam(args, ARG_SUBJECT);
        return _replaceText(subject, var -> _getReplacedAttribute(formEntry, var, false));
    }
    
    /**
     * Get the mail body
     * @param formEntry the form entry
     * @param transientVars the workflow transient vars
     * @param args the workflow args 
     * @param subject the subject of the mail
     * @return the mail body as string
     * @throws IOException if an error occurred
     */
    protected String _getBody(FormEntry formEntry, Map transientVars, Map args, String subject) throws IOException
    {
        String lang = _formDAO.getFormLocale(formEntry.getForm());
        
        String comment = _getStringParam(transientVars, ARG_COMMENT);
        String link = _getStringParam(args, ARG_LINK);
        String entryDetail = _getStringParam(args, ARG_ENTRY_DETAIL);
        String body = _getStringParam(args, ARG_BODY);
        
        body = _replaceText(body, var -> _getReplacedAttribute(formEntry, var, false));
        MailBodyBuilder mailBody = StandardMailBodyHelper.newHTMLBody()
                .withLanguage(lang)
                .withTitle(subject)
                .withMessage(body);
        
        // Get the workflow comment
        if (StringUtils.isNotEmpty(comment))
        {
            List<String> params = new ArrayList<>();
            params.add(comment);
            
            UserIdentity userIdentity = _currentUserProvider.getUser();
            mailBody.withUserInput(new UserInput(_userManager.getUser(userIdentity), ZonedDateTime.now(), comment.replaceAll("\r?\n", "<br/>")));
        }
        
        //Get the detail of the entry
        if (!StringUtils.equals(entryDetail, __NO_DATA))
        {
            String entryDetails = _formMailHelper.getMail("entry.html", formEntry, Map.of(), StringUtils.equals(entryDetail, __WITH_RESTRICTION));
            mailBody.withDetails(new I18nizableText("plugin.forms", "PLUGINS_FORMS_MAIL_RESULTS_DETAILS_TITLE"), entryDetails, false);
        }
        
        //Get the link to dashboards
        if (StringUtils.isNotEmpty(link))
        {
            Form form = formEntry.getForm();
            String siteName = form.getSiteName();
            String linkUrl = link.equals(__DASHBOARD) ? _formDAO.getAdminDashboardUri(siteName) : _formDAO.getDashboardUri(siteName);
            I18nizableText linkLabel = link.equals(__DASHBOARD) 
                    ? new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_SUBMISSIONS_LABEL")
                    : new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_DASHBOARD_LABEL");
            mailBody.withLink(linkUrl, linkLabel);
        }
        
        return mailBody.build();
    }

    private String _replaceText(String expression, Function<String, String> function)
    {
        String replacedText = "";
        try
        {
            replacedText = _mailVariableParser.replaceText(expression, function);
        }
        catch (PatternException e)
        {
            getLogger().error("Couldn't use arguments for sending email, variable  was not correctly written", e);
        }
        return StringUtils.isNotBlank(replacedText) ? replacedText : StringUtils.EMPTY;
    }
    
    /**
     * Replace variable by it's corresponding attribute 
     * @param formEntry the form entry
     * @param attribute name of variable to replace
     * @param isMailAddress true if the attribute must return an email address
     * @return the replaced attribute as text, "" if attribute is empty or null if it doesn't exist
     */
    protected String _getReplacedAttribute(FormEntry formEntry, String attribute, Boolean isMailAddress)
    {
        Form form = formEntry.getForm();
        String siteName = form.getSiteName();
        
        return switch (attribute)
        {
            case USER -> _getSubmitterValue(formEntry, isMailAddress);
            case FORM_TITLE -> _getFormTitle(form);
            case FORM_DASHBOARD -> _getDashboardUri(siteName);
            case FORM_DASHBOARD_ADMIN -> _getAdminDashboardUri(siteName);
            case ENTRY_NUMBER -> _getEntryNumber(formEntry);
            default -> attribute.startsWith(ENTRY) ? _getFormEntryAttributeValue(formEntry, attribute, isMailAddress) : null;
        };
    }

    /**
     * Get the number of the form entry
     * @param formEntry the form entry
     * @return the form entry number
     */
    protected String _getEntryNumber(FormEntry formEntry)
    {
        return String.valueOf(formEntry.getEntryId());
    }

    /**
     * Replace attribute with matching entry value, or null if attribute is invalid
     * @param formEntry the form entry
     * @param attribute name of variable to replace
     * @param isMailAddress true if the attribute must return an email address
     * @return the replaced attribute as text, "" if attribute is empty or null if it doesn't exist
     */
    protected String _getFormEntryAttributeValue(FormEntry formEntry, String attribute, Boolean isMailAddress)
    {
        String replacedAttribute = null;
        String attributeCloned = attribute.replace(ENTRY, "");
        
        String[] attributes = attributeCloned.split("\\.");
        String questionName = attributes[0];
        if (StringUtils.isNotBlank(questionName) && formEntry.hasDefinition(questionName))
        {
            if (attributes.length == 1)
            {
                if (isMailAddress && formEntry.getType(questionName).getId().equals(ModelItemTypeConstants.USER_ELEMENT_TYPE_ID))
                {
                    UserIdentity userId = (UserIdentity) formEntry.getValue(questionName);
                    User contentUser = userId != null ? _userManager.getUser(userId) : null;
                    replacedAttribute = contentUser != null ? contentUser.getEmail() : StringUtils.EMPTY;
                }
                else
                {
                    replacedAttribute = getFormQuestionMailValue(formEntry, questionName);
                }
            }
            else if (attributes.length > 1)
            {
                //If path segment is multiple we only process string content values 
                replacedAttribute = _getFormStringContentValue(formEntry, attributes, questionName);
            }
        }
        return replacedAttribute;
    }

    /**
     * Get the content value as string for the attribute
     * @param formEntry the form entry
     * @param attributes array of the path segments to the content value
     * @param questionName name of the question to find attribute in
     * @return the replaced attribute as text, "" if attribute is empty or null if it doesn't exist
     */
    protected String _getFormStringContentValue(FormEntry formEntry, String[] attributes, String questionName)
    {
        String replacedAttribute = "";
        if (formEntry.getType(questionName).getId().equals(ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID))
        {
            String path = String.join(ModelItem.ITEM_PATH_SEPARATOR, Arrays.copyOfRange(attributes, 1, attributes.length));
            
            if (formEntry.isMultiple(questionName))
            {
                ContentValue[] values = formEntry.getValue(questionName, false, new ContentValue[0]);
                List<String> valuesToJoin = new ArrayList<>();
                for (ContentValue value : values)
                {
                    String attributeValue = _getStringAttributeValue(path, value);
                    if (StringUtils.isNotBlank(attributeValue))
                    {
                        valuesToJoin.add(attributeValue);
                    }
                }
                replacedAttribute = String.join(", ", valuesToJoin);
            }
            else
            {
                ContentValue value = (ContentValue) formEntry.getValue(questionName);
                replacedAttribute = _getStringAttributeValue(path, value);
            }
        }
        return replacedAttribute;
    }

    /**
     * Get the admin dashboard uri
     * @param siteName the form's sitename
     * @return the uri of the admin dashboard
     */
    protected String _getAdminDashboardUri(String siteName)
    {
        return _formDAO.getAdminDashboardUri(siteName);
    }

    /**
     * Get the dashboard uri
     * @param siteName the form's sitename
     * @return the uri of the dashboard
     */
    protected String _getDashboardUri(String siteName)
    {
        return _formDAO.getDashboardUri(siteName);
    }

    /**
     * Get the form's title
     * @param form the current form
     * @return the form's title
     */
    protected String _getFormTitle(Form form)
    {
        return form.getTitle();
    }

    /**
     * Get the name of email address of the entry's submitter
     * @param formEntry the form entry 
     * @param isMailAddress true to get an email address
     * @return the submitter's value
     */
    protected String _getSubmitterValue(FormEntry formEntry, Boolean isMailAddress)
    {
        UserIdentity userIdentity = formEntry.getUser();
        User user = userIdentity != null ? _userManager.getUser(userIdentity) : null;
        return user != null
            ? isMailAddress ? user.getEmail() : user.getFullName() 
            : StringUtils.EMPTY;
    }

    /**
     * Get the entry's content value at path
     * @param path the path to the attribute
     * @param value the entry's value for current question
     * @return the content's value or empty string if the attribute is not defined
     */
    protected String _getStringAttributeValue(String path, ContentValue value)
    {
        if (value.hasDefinition(path) && value.getType(path).getId().equals(org.ametys.runtime.model.type.ModelItemTypeConstants.STRING_TYPE_ID))
        {
            return !value.isMultiple(path)
                    ? value.getValue(path)
                    : _getMultipleValue(value, path);
        }
        return StringUtils.EMPTY;
    }
    
    /**
     * Get the entry's content value for multiple attribute
     * @param path the path to the attribute
     * @param value the entry's value for current question
     * @return the content's value or empty string if the attribute is not defined
     */
    protected String _getMultipleValue(ContentValue value, String path)
    {
        String[] values = value.getValue(path, true);
        return values != null ? String.join(", ", values) : StringUtils.EMPTY;
    }
    
    /**
     * Sax question value 
     * @param entry the form entry
     * @param questionName the name of the question to sax
     * @return the html rendering of the question
     */
    public String getFormQuestionMailValue(FormEntry entry, String questionName)
    {
        Source src = null;
        Request request = ContextHelper.getRequest(_context);
        Form form = entry.getForm();
        
        try
        {
            request.setAttribute(IGNORE_RIGHT_KEY, true);
            String uri = "cocoon:/mail/entry/render-mail.html";
            Map<String, Object> parameters = new HashMap<>();
            parameters.put("formId", form.getId());
            parameters.put("entryId", entry.getId());
            parameters.put("questionName", questionName);

            
            src = _sourceResolver.resolveURI(uri, null, parameters);
            Reader reader = new InputStreamReader(src.getInputStream(), "UTF-8");
            return IOUtils.toString(reader);
        }
        catch (Exception e) 
        {
            getLogger().error("An error occured while saxxing question " + questionName + " in current entry", e);
            return null;
        }
        finally
        {
            request.setAttribute(IGNORE_RIGHT_KEY, false);
            _sourceResolver.release(src);
        }
    }
    
    /**
     * Get string param 
     * @param args the arguments
     * @param key the key
     * @return the string param value
     */
    protected String _getStringParam(Map args, String key)
    {
        Object object = args.get(key);
        return object != null ? (String) object : "";
    }
    
    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.POST;
    }
    
    @Override
    public List<WorkflowArgument> getArguments()
    {
        return List.of(
            _getSenderArgument(),
            _getRecipientTypeArgument(),
            _getManualRecipientArgument(),
            _getGroupRecipient(),
            _getRightsRecipientArgument(),
            _getSubjectArgument(),
            _getBodyArgument(),
            _getEntryDetailArgument(),
            _getLinkArgument()
        );
    }

    @SuppressWarnings("unchecked")
    private WorkflowArgument _getGroupRecipient()
    {
        WorkflowArgument groupRecipient = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_GROUP_RECIPIENT,
                new I18nizableText(""),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_GROUP_FIELD_DESCRIPTION"),
                true,
                false
            );
        StaticEnumerator<String> groupStaticEnumerator = new StaticEnumerator<>();
        for (GroupDirectory groupDirectory : _groupDirectoryDAO.getGroupDirectories())
        {
            I18nizableText groupDirectoryLabel = groupDirectory.getLabel();
            for (Group group : groupDirectory.getGroups())
            {
                String groupLabel = group.getLabel();
                String groupIdentity = GroupIdentity.groupIdentityToString(group.getIdentity());
                Map<String, I18nizableTextParameter> params = Map.of("directory", groupDirectoryLabel, "group", new I18nizableText(groupLabel));
                groupStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_GROUP_KEY_PARAMS_LABEL", params), groupIdentity);
            }
        }
        
        Map<String, I18nizableText> widgetParameters  = new HashMap<>();
        widgetParameters.put("emptyText", new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_GROUP_FIELD_PLACEHOLDER"));
        groupRecipient.setWidgetParameters(widgetParameters);
        
        groupRecipient.setEnumerator(groupStaticEnumerator);
        DisableConditions groupDisableConditions = new DisableConditions();
        //in workflow plugin functions arguments are prefixed with the function role to prevent duplicate
        DisableCondition groupCondition = new DisableCondition(_getFunctionArgumentPrefix() + ARG_RECIPIENT, OPERATOR.NEQ, __USER_GROUP_RECIPIENT_ARG); 
        groupDisableConditions.getConditions().add(groupCondition);
        groupRecipient.setDisableConditions(groupDisableConditions);
        return groupRecipient;
    }

    @SuppressWarnings("unchecked")
    private WorkflowArgument _getRightsRecipientArgument()
    {
        WorkflowArgument rightsRecipient = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_RIGHTS_RECIPIENT,
                new I18nizableText(""),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_RIGHT_FIELD_DESCRIPTION"),
                true,
                false
                );
        StaticEnumerator<String> rightsStaticEnumerator = new StaticEnumerator<>();
        for (String rightId : _rightsExtensionPoint.getExtensionsIds())
        {
            Right right = _rightsExtensionPoint.getExtension(rightId);
            if (FORMS_RIGHTS_CATEGORY.equals(right.getCategory().toString()))
            {
                rightsStaticEnumerator.add(right.getLabel(), right.getId());
            }
        }
        
        Map<String, I18nizableText> widgetParameters  = new HashMap<>();
        widgetParameters.put("emptyText", new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_RIGHT_FIELD_PLACEHOLDER"));
        rightsRecipient.setWidgetParameters(widgetParameters);
        
        rightsRecipient.setEnumerator(rightsStaticEnumerator);
        DisableConditions rightDisableConditions = new DisableConditions();
        //in workflow plugin functions arguments are prefixed with the function role to prevent duplicate
        DisableCondition rightCondition = new DisableCondition(_getFunctionArgumentPrefix() + ARG_RECIPIENT, OPERATOR.NEQ, __RIGHTS_RECIPIENT_ARG); 
        rightDisableConditions.getConditions().add(rightCondition);
        rightsRecipient.setDisableConditions(rightDisableConditions);
        rightsRecipient.setMultiple(true);
        return rightsRecipient;
    }

    @SuppressWarnings("unchecked")
    private WorkflowArgument _getManualRecipientArgument()
    {
        WorkflowArgument manualRecipient = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_MANUAL_RECIPIENT,
                new I18nizableText(""),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_MANUAL_FIELD_DESCRIPTION"),
                true,
                false
            );
        manualRecipient.setDefaultValue("{user}");
        DisableConditions manualDisableConditions = new DisableConditions();
        //in workflow plugin functions arguments are prefixed with the function role to prevent duplicate
        DisableCondition manualCondition = new DisableCondition(_getFunctionArgumentPrefix() + ARG_RECIPIENT, OPERATOR.NEQ, __MANUAL_RECIPIENT_ARG); 
        manualDisableConditions.getConditions().add(manualCondition);
        manualRecipient.setDisableConditions(manualDisableConditions);
        
        Map<String, I18nizableText> widgetParameters  = new HashMap<>();
        widgetParameters.put("emptyText", new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_MANUAL_FIELD_PLACEHOLDER"));
        manualRecipient.setWidgetParameters(widgetParameters);
        
        return manualRecipient;
    }

    /**
     * Get the prefix of arguments
     * @return the prefix
     */
    protected String _getFunctionArgumentPrefix()
    {
        return FUNCTION_PARAM_PREFIX;
    }
    
    @SuppressWarnings("unchecked")
    private WorkflowArgument _getRecipientTypeArgument()
    {
        StaticEnumerator<String> recipientStaticEnumerator = new StaticEnumerator<>();
        recipientStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_USER_LABEL"), __USER_RECIPIENT_ARG);
        recipientStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_GROUP_RIGHTS_LABEL"), __RIGHTS_RECIPIENT_ARG);
        recipientStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_USER_GROUP_LABEL"), __USER_GROUP_RECIPIENT_ARG);
        recipientStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_MANUAL_LABEL"), __MANUAL_RECIPIENT_ARG);
        WorkflowArgument recipient = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_RECIPIENT,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_RECIPIENT_DESCRIPTION"),
                true,
                false
            );
        recipient.setEnumerator(recipientStaticEnumerator);
        recipient.setDefaultValue(__USER_RECIPIENT_ARG);
        return recipient;
    }

    @SuppressWarnings("unchecked")
    private WorkflowArgument _getSenderArgument()
    {
        WorkflowArgument sender = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_SENDER,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_SENDER_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_SENDER_DESCRIPTION"),
                true,
                false
            );
        sender.setDefaultValue(Config.getInstance().getValue("smtp.mail.from"));
        return sender;
    }

    @SuppressWarnings("unchecked")
    private WorkflowArgument _getSubjectArgument()
    {
        WorkflowArgument subjectField = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_SUBJECT,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_SUBJECT_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_SUBJECT_DESCRIPTION"),
                true,
                false
            );
        subjectField.setDefaultValue(_i18nUtils.translate(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_FUNCTION_DEFAULT_SUBJECT")));
        return subjectField;
    }

    @SuppressWarnings("unchecked")
    private WorkflowArgument _getBodyArgument()
    {
        WorkflowArgument bodyField = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_BODY,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_BODY_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_BODY_DESCRIPTION"),
                true,
                false
            );
        bodyField.setWidget("edition.textarea");
        Map<String, I18nizableText> widgetParameters  = new HashMap<>();
        widgetParameters.put("height", new I18nizableText("250"));
        bodyField.setWidgetParameters(widgetParameters);
        bodyField.setDefaultValue(_i18nUtils.translate(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_FUNCTION_DEFAULT_BODY")));
        return bodyField;
    }

    @SuppressWarnings("unchecked")
    private WorkflowArgument _getEntryDetailArgument()
    {
        StaticEnumerator<String> dataStaticEnumerator = new StaticEnumerator<>();
        dataStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_NO_DATA_LABEL"), __NO_DATA);
        dataStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_WITH_RESTRICTION_LABEL"), __WITH_RESTRICTION);
        dataStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_WITHOUT_RESTRICTION_LABEL"), __WITHOUT_RESTRICTION);
        WorkflowArgument entryDetail = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_ENTRY_DETAIL,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_ENTRY_DETAILS_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_ENTRY_DETAILS_DESCRIPTION"),
                false,
                false
            );
        entryDetail.setEnumerator(dataStaticEnumerator);
        entryDetail.setDefaultValue(__NO_DATA);
        Map<String, I18nizableText> extWidgetParameters  = new HashMap<>();
        extWidgetParameters.put("naturalOrder", new I18nizableText("true"));
        entryDetail.setWidgetParameters(extWidgetParameters);
        return entryDetail;
    }

    @SuppressWarnings("unchecked")
    private WorkflowArgument _getLinkArgument()
    {
        StaticEnumerator<String> linkStaticEnumerator = new StaticEnumerator<>();
        linkStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_EMPTY_LABEL"), "");
        linkStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_DASHBOARD_LABEL"), __DASHBOARD);
        linkStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_SUBMISSIONS_LABEL"), __SUBMISSIONS);
        WorkflowArgument link = WorkflowElementDefinitionHelper.getElementDefinition(
                ARG_LINK,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_LINK_LABEL"),
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_ARGUMENT_LINK_DESCRIPTION"),
                false,
                false
            );
        link.setEnumerator(linkStaticEnumerator);
        link.setDefaultValue("");
        return link;
    }

    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_WORKFLOW_FORM_SEND_MAIL_FUNCTION_LABEL");
    }
    
    @Override
    public List<WorkflowVisibility> getVisibilities()
    {
        List<WorkflowVisibility> visibilities = EnhancedFunction.super.getVisibilities();
        visibilities.add(WorkflowVisibility.USER);
        return visibilities;
    }
}
