/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This tool does display the properties of the current selected entry
 * @private
 */
Ext.define('Ametys.plugins.forms.tool.EntryPropertiesTool', {
	extend: "Ametys.tool.SelectionTool",
	
	/**
	 * @property {String} [_entryUrl=entry/properties/view.xml] The url used to get the entry properties. 
	 */
	
	constructor: function(config)
	{
		this.callParent(arguments);
		
		Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onEntryChanged, this);
		Ametys.message.MessageBus.on(Ametys.message.Message.WORKFLOW_CHANGED, this._onEntryChanged, this);
		Ametys.message.MessageBus.on(Ametys.message.Message.DELETED, this._onEntryDeleted, this);
	},
	
	createPanel: function ()
	{
        return Ext.create('Ext.panel.Panel', {
            scrollable: false, 
            border: false,
            layout: 'card',
            activeItem: 0,
            
            items: [{
                    xtype: 'component',
                    cls: 'a-panel-text-empty',
                    border: false,
                    html: ''
                }, 
                {
		            scrollable: true, 
		            border: false,
		            html : '',
		            id : 'entry-details-view',
		            cls : 'details-view'
		        }
            ]
        });
	},
	
	setParams: function (params)
	{
		this._entryUrl = params['entryUrl'] || 'entry/properties/view.xml';
		this.callParent(arguments);
	},
	
	refresh: function ()
	{
		this.showRefreshing();
		
		var entryTarget = this.getCurrentSelectionTargets()[0];
		this._entryId = entryTarget.getParameters().id;
		
		var params = {entryId: this._entryId};
		
		Ametys.data.ServerComm.send({
			plugin: "forms",
			url: this._entryUrl,
			parameters: params, 
			priority: Ametys.data.ServerComm.PRIORITY_MAJOR, 
			callback: {
				handler: this._updateEntryProperties,
				scope: this,
				arguments: [this._entryId]
			},
			responseType: 'xml2text'
		});
	},
	
	/**
	 * @private
	 * Callback function called after #refresh to update the entry's properties
	 * @param {HTMLElement} response The XML document.
	 * @param {Array} args The callback arguments.
	 */
	_updateEntryProperties: function (response, args)
	{
		var entryId = args[0];
		
		if (this.getCurrentSelectionTargets() == null || this.getCurrentSelectionTargets().length == 0 ||this.getCurrentSelectionTargets()[0].getParameters().id != entryId)
		{
			// too late => discard (another entry has been selected)
			return;
		}
		
		if (Ametys.data.ServerComm.handleBadResponse("{{i18n PLUGINS_FORMS_ENTRY_PROPERTIES_ERROR}} '" + entryId + "'", response, 'Ametys.plugins.forms.tool.EntryPropertiesTool'))
        {
            return;
        }
		
        
		this.getContentPanel().items.get(1).update(Ext.dom.Query.selectValue("", response));
        this.getContentPanel().getLayout().setActiveItem(1);
		this.showRefreshed();
	},
    
	setNoSelectionMatchState: function (message)
	{
        this.callParent(arguments);
        
        var panel = this.getContentPanel().items.get(0);
        panel.update(message);
        this.getContentPanel().getLayout().setActiveItem(0);
        this._entryId = null;
	},
	
	/**
	 * Listener on {@link Ametys.message.Message#WORKFLOW_CHANGED} and {@link Ametys.message.Message#MODIFIED} messages. If the current
	 * entry is concerned, the tool will be out-of-date.
	 * 
	 * @param {Ametys.message.Message} message The bus message.
	 * @protected
	 */
	_onEntryChanged: function (message)
	{
		if (this.getTargetsInCurrentSelectionTargets(message).length > 0)
		{
			this.showOutOfDate();
		}
	},
    
    /**
     * Listener on {@link Ametys.message.Message#DELETED}messages. If the current
     * entry is concerned, clear the selection.
     * 
     * @param {Ametys.message.Message} message The bus message.
     * @protected
     */
    _onEntryDeleted: function (message)
    {
        if (this.getTargetsInCurrentSelectionTargets(message).length > 0)
        {
            this.setNoSelectionMatchState();
        }
    }
	
});