/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.actions;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.Request;

import org.ametys.core.right.RightManager;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.dao.FormQuestionDAO;
import org.ametys.plugins.forms.dao.FormQuestionDAO.FormEntryValues;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.sources.ChoiceSourceType;
import org.ametys.plugins.forms.question.types.ChoicesListQuestionType;
import org.ametys.plugins.forms.question.types.ComputedQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItem;
import org.ametys.web.FOAmetysObjectCreationHelper;

/**
 * Abstract action to process form entry inputs
 */
public abstract class AbstractProcessFormAction extends ServiceableAction
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The FO ametys object creation helper */
    protected FOAmetysObjectCreationHelper _foAmetysObjectCreationHelper;

    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** The form entry DAO */
    protected FormEntryDAO _entryDAO;
    
    /** The form question DAO */
    protected FormQuestionDAO _formQuestionDAO;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _foAmetysObjectCreationHelper = (FOAmetysObjectCreationHelper) smanager.lookup(FOAmetysObjectCreationHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _formDAO = (FormDAO) smanager.lookup(FormDAO.ROLE);
        _entryDAO = (FormEntryDAO) smanager.lookup(FormEntryDAO.ROLE);
        _formQuestionDAO = (FormQuestionDAO) smanager.lookup(FormQuestionDAO.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);        
    }
    
    /**
     * Remove empty value for choice list because empty value is an other value
     * @param form the form
     * @param formInputValues the form inputs to change
     */
    protected void _adaptFormValuesForChoiceList(Form form, Map<String, Object> formInputValues)
    {
        List<FormQuestion> choiceListQuestions = form.getQuestions()
            .stream()
            .filter(q -> q.getType() instanceof ChoicesListQuestionType)
            .toList();
        
        for (FormQuestion question : choiceListQuestions)
        {
            ChoicesListQuestionType type = (ChoicesListQuestionType) question.getType();
            ChoiceSourceType sourceType = type.getSourceType(question);
            
            String nameForForm = question.getNameForForm();
            if (formInputValues.containsKey(nameForForm))
            {
                Object object = formInputValues.get(nameForForm);
                Object newValue = sourceType.removeEmptyOrOtherValue(object);
                if (newValue == null)
                {
                    formInputValues.remove(nameForForm);
                }
                else
                {
                    formInputValues.put(nameForForm, newValue);
                }
                
            }
        }
    }
    
    /**
     * Get a view without elements hidden by a rule
     * @param request the request
     * @param form The current form
     * @param entryView The entry view with possibly unwanted viewItems
     * @param entryValues The entry values 
     * @param currentStepId current step of the entry. Can be empty if the form has no workflow
     * @return a view with filtered items
     */
    protected View _getRuleFilteredEntryView(Request request, Form form, View entryView, FormEntryValues entryValues, Optional<Long> currentStepId)
    {
        View filteredEntryView = new View();
        for (FormQuestion target : _getRuleFilteredQuestions(request, form, entryValues, currentStepId))
        {
            ViewItem viewItem = entryView.getViewItem(target.getNameForForm());
            filteredEntryView.addViewItem(viewItem);
            _manageOtherOption(entryView, filteredEntryView, target);
        }
        return filteredEntryView;
    }
    
    /**
     * Get the list of active question depending of the form rules
     * @param request the request
     * @param form the form
     * @param entryValues the entry values to compute rules
     * @param currentStepId the current step id. Can be empty if the form has no workflow
     * @return the list of active question depending of the form rules
     */
    protected abstract List<FormQuestion> _getRuleFilteredQuestions(Request request, Form form, FormEntryValues entryValues, Optional<Long> currentStepId);
    
    private void _manageOtherOption(View entryView, View filteredEntryView, FormQuestion target)
    {
        if (target.getType() instanceof ChoicesListQuestionType type && type.hasOtherOption(target))
        {
            ViewItem viewOtherItem = entryView.getViewItem(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + target.getNameForForm());
            filteredEntryView.addViewItem(viewOtherItem);
        }
    }
    
    /**
     * Handle computed values
     * @param questions the form questions
     * @param entry the entry
     * @param forEdition <code>true</code> to handle edition
     */
    protected void _handleComputedValues(List<FormQuestion> questions, FormEntry entry, boolean forEdition)
    {
        for (FormQuestion question : questions)
        {
            FormQuestionType questionType = question.getType();
            if (questionType instanceof ComputedQuestionType type)
            {
                if (!forEdition || type.getComputingType(question).canEdit())
                {
                    Object computedValue = type.getComputingType(question).getComputedValue(question, entry);
                    if (computedValue != null)
                    {
                        entry.setValue(question.getNameForForm(), computedValue);
                    }
                }
            }
        }
    }
}
