/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.cms.search.cocoon.SearchAction;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.plugins.forms.enumerators.LazyEnumerator;
import org.ametys.plugins.forms.helper.FormStatisticsHelper;
import org.ametys.plugins.forms.helper.LimitedEntriesHelper;
import org.ametys.plugins.forms.helper.ScheduleOpeningHelper;
import org.ametys.plugins.forms.question.sources.AbstractSourceType;
import org.ametys.plugins.forms.question.sources.ChoiceOption;
import org.ametys.plugins.forms.question.sources.ChoiceSourceType;
import org.ametys.plugins.forms.question.sources.ChoiceSourceTypeExtensionPoint;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Action to get form choice list options
 */
public class GetFormChoiceListOptions extends ServiceableAction
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The choice source type extension point */
    protected ChoiceSourceTypeExtensionPoint _choiceSourceTypeEP;

    /** The limited entries helper */
    protected LimitedEntriesHelper _limitedEntriesHelper;
    
    /** The form statistics helper */
    protected FormStatisticsHelper _formStatsHelper;
    
    /** The schedule opening helper */
    protected ScheduleOpeningHelper _scheduleOpeningHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _choiceSourceTypeEP = (ChoiceSourceTypeExtensionPoint) smanager.lookup(ChoiceSourceTypeExtensionPoint.ROLE);
        _limitedEntriesHelper = (LimitedEntriesHelper) smanager.lookup(LimitedEntriesHelper.ROLE);
        _formStatsHelper = (FormStatisticsHelper) smanager.lookup(FormStatisticsHelper.ROLE);
        _scheduleOpeningHelper = (ScheduleOpeningHelper) smanager.lookup(ScheduleOpeningHelper.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        List<Map<String, Object>> json = new ArrayList<>();

        Request request = ObjectModelHelper.getRequest(objectModel);
        String lang = request.getParameter("lang");
        request.setAttribute(SearchAction.SEARCH_LOCALE, new Locale(lang));
        
        String sourceType = request.getParameter("sourceType");
        ChoiceSourceType choiceSourceType = _choiceSourceTypeEP.getExtension(sourceType);
        
        String questionId = request.getParameter("questionId");
        FormQuestion question = _resolver.resolveById(questionId);
        
        String search = request.getParameter("search");
        int limit = Integer.parseInt(request.getParameter("limit")); 
        int page = request.getParameter("page") != null 
                ? Integer.parseInt(request.getParameter("page"))
                : 1;
        
        Map<String, Object> enumParam = new HashMap<>();
        enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
        enumParam.put(LazyEnumerator.PATTERN_PARAMETER_KEY, search);
        enumParam.put(LazyEnumerator.SORT_PARAMETER_KEY, true);
        enumParam.put(LazyEnumerator.LANG_PARAMETER_KEY, lang);
        
        Map<ChoiceOption, I18nizableText> typedEntries = choiceSourceType.searchEntries(enumParam, limit, page);
        for (ChoiceOption option : typedEntries.keySet())
        {
            json.add(Map.of("text", typedEntries.get(option), "id", option.getValue()));
        }
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, json);
        
        return EMPTY_MAP;
    }
}
