/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.forms.FormXpathUtils;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormDirectory;
import org.ametys.plugins.forms.repository.FormDirectoryFactory;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.MovableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.SiteManager;

import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;

/**
 * DAO for manipulating form directories
 */
public class FormDirectoryDAO extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = FormDirectoryDAO.class.getName();
    
    /** The alias id of the root {@link FormDirectory} */
    public static final String ROOT_FORM_DIRECTORY_ID = "root";

    /** The right id to handle forms */
    public static final String HANDLE_FORM_DIRECTORIES_RIGHT_ID = "FormsDirectory_Rights_Directories";
    
    private static final String __ROOT_NODE_NAME = "ametys:forms";
    
    private static final String __PLUGIN_NODE_NAME = "forms";

    private static final String __FORMDIRECTORY_NAME_PREFIX = "formdirectory-";
    
    /** Observer manager. */
    protected ObservationManager _observationManager;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _userProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
    }
    
    /**
     * Check if a user have write rights on a form directory
     * @param userIdentity the user
     * @param formDirectory the form directory
     * @return true if the user have write rights on a form directory
     */
    public boolean hasWriteRightOnFormDirectory(UserIdentity userIdentity, FormDirectory formDirectory)
    {      
        return _rightManager.hasRight(userIdentity, HANDLE_FORM_DIRECTORIES_RIGHT_ID, formDirectory) == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Check rights for a form element as ametys object
     * @param formElement the form element as ametys object
     */
    public void checkHandleFormDirectoriesRight(AmetysObject formElement)
    {
        UserIdentity user = _userProvider.getUser();
        if (!hasWriteRightOnFormDirectory(user, null))
        {
            throw new IllegalAccessError("User '" + user + "' tried to handle form directories without convenient right [" + HANDLE_FORM_DIRECTORIES_RIGHT_ID + "]");
        }
    }
    
    /**
     * Get the root plugin storage object.
     * @param siteName the site's name
     * @return the root plugin storage object.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    public FormDirectory getFormDirectoriesRootNode(String siteName) throws AmetysRepositoryException
    {
        try
        {
            return _getOrCreateRootNode(siteName);
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the forms root node", e);
        }
    }
    
    private FormDirectory _getOrCreateRootNode(String siteName) throws AmetysRepositoryException
    {
        ModifiableTraversableAmetysObject pluginsNode = _siteManager.getSite(siteName).getRootPlugins();
        
        ModifiableTraversableAmetysObject pluginNode = (ModifiableTraversableAmetysObject) _getOrCreateNode(pluginsNode, __PLUGIN_NODE_NAME, "ametys:unstructured");
        
        return (FormDirectory) _getOrCreateNode(pluginNode, __ROOT_NODE_NAME, FormDirectoryFactory.FORM_DIRECTORY_NODETYPE);
    }
    
    private AmetysObject _getOrCreateNode(ModifiableTraversableAmetysObject parentNode, String nodeName, String nodeType) throws AmetysRepositoryException
    {
        AmetysObject definitionsNode;
        if (parentNode.hasChild(nodeName))
        {
            definitionsNode = parentNode.getChild(nodeName);
        }
        else
        {
            definitionsNode = parentNode.createChild(nodeName, nodeType);
            parentNode.saveChanges();
        }
        return definitionsNode;
    }
    
    /**
     * Get the root directory properties
     * @param siteName the site's name
     * @return The root directory properties
     */
    @Callable (rights = {"FormsDirectory_Rights_Tool", "CMS_Rights_Delegate_Rights", "Runtime_Rights_Rights_Handle"})
    public Map<String, Object> getRootProperties(String siteName)
    {
        return getFormDirectoryProperties(getFormDirectoriesRootNode(siteName), false);
    }
    
    /**
     * Get the form directory properties
     * @param siteName the site's name
     * @param id The form directory id. Can be {@link #ROOT_FORM_DIRECTORY_ID} for the root directory.
     * @return The form directory properties
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getFormDirectoryProperties(String siteName, String id)
    {
        return getFormDirectoryProperties(getFormDirectory(siteName, id), true);
    }
    
    /**
     * Get the form directory properties
     * @param formDirectory The form directory
     * @param withRights <code>true</code> to add rights to the properties
     * @return The form directory properties
     */
    public Map<String, Object> getFormDirectoryProperties(FormDirectory formDirectory, boolean withRights)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("isForm", false);
        infos.put("id", formDirectory.getId());
        infos.put("title", formDirectory.getTitle());
        infos.put("fullPath", getFormDirectoryPath(formDirectory, " > "));
        
        FormDirectory formDirectoriesRoot = getFormDirectoriesRootNode(formDirectory.getSiteName());
        String parentId = formDirectory.getParent().getId().equals(formDirectoriesRoot.getId()) ? ROOT_FORM_DIRECTORY_ID : formDirectory.getParent().getId();
        infos.put("parentId", parentId);
        
        infos.put("hasChildren", formDirectory.getChildren().getSize() != 0);

        UserIdentity currentUser = _userProvider.getUser();
        boolean canWriteParent = formDirectory.getParent() instanceof FormDirectory && hasWriteRightOnFormDirectory(currentUser, (FormDirectory) formDirectory.getParent());
        
        // Used to check if a user have the right to limit access to this directory
        infos.put("canEditRight", hasRightAffectationRightOnFormDirectory(currentUser, formDirectory));
        
        if (withRights)
        {
            infos.put("canWriteParent", canWriteParent);
            infos.put("rights", _getUserRights(formDirectory));
        }
        else
        {
            boolean canRead = this.hasReadRightOnFormDirectory(currentUser, formDirectory);
            boolean canWrite = this.hasWriteRightOnFormDirectory(currentUser, formDirectory);

            // Used to check if a user have the right to rename this directory
            infos.put("canRename", canWrite && canWriteParent);
            
            // Used to check if a user have the right to add directories or forms inside this directory
            infos.put("canWrite", canWrite);
            
            // Used to check if a user have the right to delete and drag&drop this directory
            infos.put("canEdit", canWrite && _hasWriteRightOnEachDescendant(currentUser, formDirectory) && canWriteParent);
            
            // Used to filter directories
            infos.put("displayForRead", canRead || canWrite || hasAnyReadableDescendant(currentUser, formDirectory) || hasAnyWritableDescendant(currentUser, formDirectory));
            infos.put("displayForWrite", canWrite || _hasAnyWriteDescendantFolder(currentUser, formDirectory));
            infos.put("displayForRights", canWrite || hasAnyAssignableDescendant(currentUser, formDirectory));
        }
        
        return infos;
    }
    
    /**
     * Get user rights for the given form directory
     * @param formDirectory the form directory
     * @return the set of rights
     */
    protected Set<String> _getUserRights (FormDirectory formDirectory)
    {
        UserIdentity user = _userProvider.getUser();
        return _rightManager.getUserRights(user, formDirectory);
    }
    
   /**
    * Get form directory with given id
    * @param siteName the site name where to search in jcr
    * @param id the directory id
    * @return the form directory having given id
    */
    public FormDirectory getFormDirectory(String siteName, String id)
    {
        return ROOT_FORM_DIRECTORY_ID.equals(id) 
                ? getFormDirectoriesRootNode(siteName)
                : _resolver.resolveById(id);
    }
    
    /**
     * Get the path of a form directory
     * @param formDirectory the form directory
     * @param separator the seperator to use from path
     * @return the path in form's directory
     */
    public String getFormDirectoryPath(FormDirectory formDirectory, String separator)
    {
        List<String> fullPath = new ArrayList<>();
        fullPath.add(formDirectory.getTitle());

        AmetysObject parent = formDirectory.getParent();
        while (parent instanceof FormDirectory parentDirectory && !isRoot(parentDirectory)) 
        {
            fullPath.add(0, parentDirectory.getTitle()); 
            parent = parent.getParent();
        }
        return String.join(separator, fullPath);
    }
    
    /**
     * Determines if the form directory is the root directory
     * @param formDirectory the form directory. Cannot be null.
     * @return true if is the root directory
     */
    public boolean isRoot(FormDirectory formDirectory)
    {
        return __ROOT_NODE_NAME.equals(formDirectory.getName());
    }
    
    /**
     * Creates a new {@link FormDirectory}
     * @param siteName the site's name
     * @param parentId The id of the parent. Use {@link #ROOT_FORM_DIRECTORY_ID} for the root directory.
     * @param name The desired name for the new {@link FormDirectory}
     * @return A result map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> createFormDirectory(String siteName, String parentId, String name)
    {
        Map<String, Object> results = new HashMap<>();
        
        FormDirectory parent = getFormDirectory(siteName, parentId);
        checkHandleFormDirectoriesRight(parent);
        
        // Find unique and legal name for node
        String uniqueName = NameHelper.getUniqueAmetysObjectName(parent, __FORMDIRECTORY_NAME_PREFIX + name);
        
        FormDirectory formDirectory = parent.createChild(uniqueName, FormDirectoryFactory.FORM_DIRECTORY_NODETYPE);
        formDirectory.setTitle(name);
        parent.saveChanges();
        
        results.put("id", formDirectory.getId());
        results.put("title", formDirectory.getTitle());
        results.put("parentId", parent.getId());
        
        return results;
    }
    
    /**
     * Renames a {@link FormDirectory}
     * @param id The id of the form directory
     * @param newName The new name of the directory
     * @return A result map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> renameFormDirectory(String id, String newName)
    {
        FormDirectory directory = _resolver.resolveById(id);
        checkHandleFormDirectoriesRight(directory);
        
        Map<String, Object> results = new HashMap<>();
        
        AmetysObject parent = directory.getParent();
        
        if (parent instanceof FormDirectory parentDirectory)
        {
            UserIdentity currentUser = _userProvider.getUser();
            
            boolean canWrite = hasWriteRightOnFormDirectory(currentUser, directory);
            boolean canWriteParent = hasWriteRightOnFormDirectory(currentUser, parentDirectory);
            
            if (canWrite && canWriteParent)
            {
                // Find unique and legal name for node
                String uniqueName = NameHelper.getUniqueAmetysObjectName(parentDirectory, __FORMDIRECTORY_NAME_PREFIX + newName);
                
                Node node = directory.getNode();
                try
                {
                    node.getSession().move(node.getPath(), node.getParent().getPath() + '/' + uniqueName);
                    directory.setTitle(newName);
                    node.getSession().save();
                    
                    results.put("id", id);
                    results.put("title", directory.getTitle());
                }
                catch (RepositoryException e)
                {
                    getLogger().warn("Form directory renaming failed.", e);
                    results.put("message", "cannot-rename");
                }
            }
            else
            {
                results.put("message", "not-allowed");
            }
        }
        else
        {
            results.put("message", "not-allowed");
        }
        
        return results;
    }
    
    /**
     * Deletes {@link FormDirectory}(s)
     * @param ids The ids of the form directories to delete
     * @return A result map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> deleteFormDirectory(List<String> ids)
    {
        Map<String, Object> results = new HashMap<>();
        
        List<Map<String, Object>> allDeleted = new ArrayList<>();
        List<Map<String, Object>> allUnknown = new ArrayList<>();

        if (!canDeleteAllFormDirectories(ids))
        {
            results.put("message", "not-allowed");
            return results;
        }
        
        for (String id : ids)
        {
            try
            {
                FormDirectory directory = _resolver.resolveById(id);
                String name = directory.getName();
                String title = directory.getTitle();
                directory.remove();
                directory.saveChanges();
                Map<String, Object> deleted = ImmutableMap.of("id", id, "name", name, "title", title);
                allDeleted.add(deleted);
            }
            catch (UnknownAmetysObjectException e)
            {
                Map<String, Object> unknown = ImmutableMap.of("id", id);
                allUnknown.add(unknown);
                getLogger().error("Unable to delete form directory. The directory of id '" + id + " doesn't exist", e);
            }
        }
        
        results.put("deleted", allDeleted);
        results.put("unknown", allUnknown);
        
        return results;
    }
    
    /**
     * Moves a {@link FormDirectory}
     * @param siteName name of the site
     * @param id The id of the form directory
     * @param newParentId The id of the new parent directory of the form directory. Use {@link #ROOT_FORM_DIRECTORY_ID} for the root directory.
     * @return A result map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> moveFormDirectory(String siteName, String id, String newParentId)
    {
        Map<String, Object> results = new HashMap<>();
        FormDirectory formDirectory = _resolver.resolveById(id);
        FormDirectory parentFormDirectory = getFormDirectory(siteName, newParentId);
        UserIdentity currentUser = _userProvider.getUser();
        
        if (_hasWriteRightOnEachDescendant(currentUser, formDirectory) 
                && hasWriteRightOnFormDirectory(currentUser, parentFormDirectory))
        {
            move(formDirectory, siteName, newParentId, results);
        }
        else
        {
            results.put("message", "not-allowed");
        }
        results.put("id", formDirectory.getId());
        results.put("title", formDirectory.getTitle());
        return results;
    }
    
    /**
     * Moves a {@link FormDirectory} or a {@link Form}
     * @param obj form or directory to move
     * @param siteName name of the site
     * @param newParentId id of directory where to drop obj
     * @param results a result map
     */
    public void move(MovableAmetysObject obj, String siteName, String newParentId, Map<String, Object> results)
    {
        FormDirectory newParent = getFormDirectory(siteName, newParentId);
        if (obj.canMoveTo(newParent))
        {
            try
            {
                obj.moveTo(newParent, false);
            }
            catch (AmetysRepositoryException e)
            {
                getLogger().warn("Form moving failed.", e);
                results.put("message", "cannot-move");
            }
        }
        else
        {
            results.put("message", "cannot-move");
        }
    }
    
    /**
     * Determines if application must warn before deleting the given {@link FormDirectory}s
     * @param ids The {@link FormDirectory} ids
     * @return <code>true</code> if application must warn
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public boolean mustWarnBeforeDeletion(List<String> ids)
    {
        return ids.stream()
                .anyMatch(LambdaUtils.wrapPredicate(this::_mustWarnBeforeDeletion));
    }
    
    private boolean _mustWarnBeforeDeletion(String id)
    {
        FormDirectory directory = _resolver.resolveById(id);
        AmetysObjectIterable<Form> allForms = getChildFormsForAdministrator(directory, false);
        return _containsNotOwnForms(allForms);
    }
    
    private boolean _containsNotOwnForms(AmetysObjectIterable<Form> allForms)
    {
        UserIdentity currentUser = _userProvider.getUser();
        return allForms.stream()
                .map(Form::getAuthor)
                .anyMatch(Predicates.not(currentUser::equals));
    }
    
    /**
     * Can the current user delete all the {@link FormDirectory}s from the list ?
     * @param ids The {@link FormDirectory} ids
     * @return <code>true</code> if he can
     */
    protected boolean canDeleteAllFormDirectories(List<String> ids)
    {
        return canDeleteFormDirectories(ids).values()
                .stream()
                .allMatch(Boolean.TRUE::equals);
    }
    
    /**
     * Determines if the current user can delete the given {@link FormDirectory}s
     * @param ids The {@link FormDirectory} ids
     * @return A map with <code>true</code> for each id if the current user can delete the associated {@link FormDirectory}
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Boolean> canDeleteFormDirectories(List<String> ids)
    {
        return ids.stream()
                .collect(Collectors.toMap(
                        Function.identity(), 
                        LambdaUtils.wrap(this::_canDeleteFormDirectory)));
    }
    
    private boolean _canDeleteFormDirectory(String id)
    {
        if (ROOT_FORM_DIRECTORY_ID.equals(id))
        {
            return false;
        }
        else
        {
            UserIdentity user = _userProvider.getUser();
            FormDirectory directory = _resolver.resolveById(id);
            
            return _hasWriteRightOnEachDescendant(user, directory) 
                        && !_hasPublishedForms(directory)
                        && directory.getParent() instanceof FormDirectory 
                        && hasWriteRightOnFormDirectory(user, (FormDirectory) directory.getParent());
        }
    }
    
    private boolean _hasPublishedForms(FormDirectory formDirectory)
    {
        boolean hasPublishedForms = false;
        for (AmetysObject child : formDirectory.getChildren())
        {
            if (child instanceof FormDirectory)
            {
                hasPublishedForms = hasPublishedForms || _hasPublishedForms((FormDirectory) child);
            }
            else if (child instanceof Form)
            {
                hasPublishedForms = hasPublishedForms || !_formDAO.getFormPage(child.getId(), formDirectory.getSiteName()).isEmpty();
            }
        }
        
        return hasPublishedForms;
    }
    
    /**
     * Check if a user have write rights on an form directory and each of his descendant
     * @param userIdentity the user
     * @param formDirectory the form directory
     * @return true if the user have write rights on an form directory and each of his descendant
     */
    private boolean _hasWriteRightOnEachDescendant(UserIdentity userIdentity, FormDirectory formDirectory)
    {
        boolean hasRight = hasWriteRightOnFormDirectory(userIdentity, formDirectory);
        if (!hasRight)
        {
            return false;
        }
        
        try (AmetysObjectIterable<AmetysObject> children = formDirectory.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof FormDirectory)
                {
                    hasRight = hasRight && _hasWriteRightOnEachDescendant(userIdentity, (FormDirectory) child);
                }
                else if (child instanceof Form)
                {
                    hasRight = hasRight && hasWriteRightOnFormAndParent(userIdentity, (Form) child, formDirectory);
                }
                
                if (!hasRight)
                {
                    return false;
                }
            }
            
            return hasRight;
        }
    }
    
    /**
     * Check if a user have write rights on a form
     * @param userIdentity the user
     * @param form the source for the form
     * @param formDirectory the form directory
     * @return true if the user have write rights on a form
     */
    public boolean hasWriteRightOnFormAndParent(UserIdentity userIdentity, Form form, FormDirectory formDirectory)
    {
        return form != null && userIdentity.equals(form.getAuthor()) || _formDAO.hasWriteRightOnForm(userIdentity, form) && hasWriteRightOnFormDirectory(userIdentity, formDirectory);
    }
    
    /**
     * Gets all forms in WRITE access for given parent
     * @param parent The {@link FormDirectory}, defining the context from which getting children
     * @param onlyDirect <code>true</code> in order to have only direct child forms from parent path, <code>false</code> otherwise to have all forms at any level underneath the parent path
     * @param user The user
     * @return all forms in WRITE access for given parent
     */
    public Stream<Form> getChildFormsInWriteAccess(FormDirectory parent, boolean onlyDirect, UserIdentity user)
    {
        return _resolver.query(FormXpathUtils.getXPathForForms(parent, onlyDirect))
                .stream()
                .filter(Form.class::isInstance)
                .map(obj -> (Form) obj)
                .filter(form -> _formDAO.hasWriteRightOnForm(user, form));
    }
    
    /**
     * Gets all forms in WRITE access for given parent
     * @param parent The {@link FormDirectory}, defining the context from which getting children
     * @param onlyDirect <code>true</code> in order to have only direct child forms from parent path, <code>false</code> otherwise to have all forms at any level underneath the parent path
     * @param user The user
     * @return all forms in WRITE access for given parent
     */
    public Stream<Form> getChildFormsInRightAccess(FormDirectory parent, boolean onlyDirect, UserIdentity user)
    {
        return _resolver.query(FormXpathUtils.getXPathForForms(parent, onlyDirect))
                .stream()
                .filter(Form.class::isInstance)
                .map(obj -> (Form) obj)
                .filter(form -> _formDAO.hasRightAffectationRightOnForm(user, form));
    }
    
    /**
     * Gets all forms in READ access for given parent
     * @param parent The {@link FormDirectory}, defining the context from which getting children
     * @param onlyDirect <code>true</code> in order to have only direct child forms from parent path, <code>false</code> otherwise to have all forms at any level underneath the parent path
     * @param user The user
     * @return all forms in READ access for given parent
     */
    public Stream<Form> getChildFormsInReadAccess(FormDirectory parent, boolean onlyDirect, UserIdentity user)
    {
        return _resolver.query(FormXpathUtils.getXPathForForms(parent, onlyDirect))
                .stream()
                .filter(Form.class::isInstance)
                .map(obj -> (Form) obj)
                        .filter(form -> _formDAO.hasReadRightOnForm(user, form));
    }

    /**
     * Gets all forms for administrator for given parent
     * @param parent The {@link FormDirectory}, defining the context from which getting children
     * @param onlyDirect <code>true</code> in order to have only direct child forms from parent path, <code>false</code> otherwise to have all forms at any level underneath the parent path
     * @return all forms for administrator for given parent
     */
    public AmetysObjectIterable<Form> getChildFormsForAdministrator(FormDirectory parent, boolean onlyDirect)
    {
        return _resolver.query(FormXpathUtils.getXPathForForms(parent, onlyDirect));
    }
    
    /**
     * Gets all form directories for given parent
     * @param parent The {@link FormDirectory}, defining the context from which getting children
     * @return all form directories for given parent
     */
    public AmetysObjectIterable<FormDirectory> getChildFormDirectories(FormDirectory parent)
    {
        return _resolver.query(FormXpathUtils.getXPathForFormDirectories(parent));
    }
    
    /**
     * Provides the current user.
     * @return the user which cannot be <code>null</code>.
     */
    protected UserIdentity _getCurrentUser()
    {      
        return _userProvider.getUser();
    }
    
    /**
     * Check if a user have write rights on a form directory
     * @param userIdentity the user
     * @param formDirectory the form directory
     * @return true if the user have write rights on a form
     */
    public boolean hasRightAffectationRightOnFormDirectory(UserIdentity userIdentity, FormDirectory formDirectory)
    {
        return hasWriteRightOnFormDirectory(userIdentity, formDirectory) || _rightManager.hasRight(userIdentity, "Runtime_Rights_Rights_Handle", "/cms") == RightResult.RIGHT_ALLOW;
    }
    
    
    /**
     * Check if a user have read rights on a form directory
     * @param userIdentity the user
     * @param directory the form directory
     * @return true if the user have read rights on a form directory
     */
    public boolean hasReadRightOnFormDirectory(UserIdentity userIdentity, FormDirectory directory)
    {
        return _rightManager.hasReadAccess(userIdentity, directory);
    }
    
    /**
     * Check if a directory have a descendant in read access for a given user
     * @param userIdentity the user
     * @param formDirectory the form directory
     * @return true if the user have read right for at least one child of this directory
     */
    public Boolean hasAnyReadableDescendant(UserIdentity userIdentity, FormDirectory formDirectory)
    {
        boolean hasDescendant = hasReadRightOnFormDirectory(userIdentity, formDirectory);
        if (hasDescendant)
        {
            return true;
        }
        
        try (AmetysObjectIterable<AmetysObject> children = formDirectory.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof FormDirectory)
                {
                    hasDescendant = hasDescendant || hasAnyReadableDescendant(userIdentity, (FormDirectory) child);
                }
                else if (child instanceof Form)
                {
                    hasDescendant = hasDescendant || _formDAO.hasReadRightOnForm(userIdentity, (Form) child);
                }
                
                if (hasDescendant)
                {
                    return true;
                }
            }
            
            return hasDescendant;
        }
    }

    /**
     * Check if a directory have descendant in write access for a given user
     * @param userIdentity the user
     * @param formDirectory the form directory
     * @return true if the user have write right for at least one child of this directory
     */
    public Boolean hasAnyWritableDescendant(UserIdentity userIdentity, FormDirectory formDirectory)
    {
        boolean hasDescendant = hasWriteRightOnFormDirectory(userIdentity, formDirectory);
        if (hasDescendant)
        {
            return true;
        }
        
        try (AmetysObjectIterable<AmetysObject> children = formDirectory.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof FormDirectory)
                {
                    hasDescendant = hasDescendant || hasAnyWritableDescendant(userIdentity, (FormDirectory) child);
                }
                else if (child instanceof Form)
                {
                    hasDescendant = hasDescendant || hasWriteRightOnFormAndParent(userIdentity, (Form) child, formDirectory);
                }
                
                if (hasDescendant)
                {
                    return true;
                }
            }
        }

        return hasDescendant;
    }
    
    private Boolean _hasAnyWriteDescendantFolder(UserIdentity userIdentity, FormDirectory formDirectory)
    {
        boolean hasDescendant = hasWriteRightOnFormDirectory(userIdentity, formDirectory);
        if (hasDescendant)
        {
            return true;
        }
        
        try (AmetysObjectIterable<AmetysObject> children = formDirectory.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof FormDirectory)
                {
                    hasDescendant = hasDescendant || _hasAnyWriteDescendantFolder(userIdentity, (FormDirectory) child);
                }
                
                if (hasDescendant)
                {
                    return true;
                }
            }
            
            return hasDescendant;
        }
    }
    
    /**
     * Check if a directory have descendant in right assignment access for a given user
     * @param userIdentity the user
     * @param formDirectory the form directory
     * @return true if the user have right assignment right for at least one child of this directory
     */
    public Boolean hasAnyAssignableDescendant(UserIdentity userIdentity, FormDirectory formDirectory)
    {
        boolean hasDescendant = hasRightAffectationRightOnFormDirectory(userIdentity, formDirectory);
        if (hasDescendant)
        {
            return true;
        }
        
        try (AmetysObjectIterable<AmetysObject> children = formDirectory.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof FormDirectory)
                {
                    hasDescendant = hasDescendant || hasAnyAssignableDescendant(userIdentity, (FormDirectory) child);
                }
                else if (child instanceof Form)
                {
                    hasDescendant = hasDescendant || _formDAO.hasRightAffectationRightOnForm(userIdentity, (Form) child);
                }
                
                if (hasDescendant)
                {
                    return true;
                }
            }
            
            return hasDescendant;
        }
    }
}
