/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.jackrabbit.util.Text;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.forms.FormEvents;
import org.ametys.plugins.forms.question.types.ChoicesListQuestionType;
import org.ametys.plugins.forms.repository.CopyFormUpdater;
import org.ametys.plugins.forms.repository.CopyFormUpdaterExtensionPoint;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormPage;
import org.ametys.plugins.forms.repository.FormPageFactory;
import org.ametys.plugins.forms.repository.FormPageRule;
import org.ametys.plugins.forms.repository.FormPageRule.PageRuleType;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * DAO for manipulating form pages
 */
public class FormPageDAO extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = FormPageDAO.class.getName();
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    /** The form question DAO */
    protected FormQuestionDAO _formQuestionDAO;
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    /** The form DAO */
    protected FormDAO _formDAO;
    /** The right manager */
    protected RightManager _rightManager;
    /** The copy form updater extension point */
    protected CopyFormUpdaterExtensionPoint _copyFormEP;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _formQuestionDAO = (FormQuestionDAO) manager.lookup(FormQuestionDAO.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _copyFormEP = (CopyFormUpdaterExtensionPoint) manager.lookup(CopyFormUpdaterExtensionPoint.ROLE);
    }

    /**
     * Creates a {@link FormPage}.
     * @param parentId The id of the parent.
     * @param name The desired name for the new {@link FormPage}
     * @return The id of the created form page
     * @throws Exception if an error occurs during the form creation process
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, String> createPage (String parentId, String name) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        Form rootNode = _resolver.resolveById(parentId);
        _formDAO.checkHandleFormRight(rootNode);
        
        // Find unique name
        String uniqueName = NameHelper.getUniqueAmetysObjectName(rootNode, name);
        FormPage page = rootNode.createChild(uniqueName, FormPageFactory.FORM_PAGE_NODETYPE);
        page.setTitle(uniqueName);
        
        rootNode.saveChanges();

        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", rootNode);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        
        result.put("id", page.getId());
        result.put("name", page.getTitle());
        
        return result;
    }
    
    /**
     * Rename a {@link FormPage}
     * @param id The id of the page 
     * @param newName The new name of the page
     * @return A result map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, String> renamePage (String id, String newName)
    {
        Map<String, String> results = new HashMap<>();
        
        FormPage page = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(page);
        
        String legalName = Text.escapeIllegalJcrChars(newName);
        Node node = page.getNode();
        try
        {
            page.setTitle(newName);
            
            node.getSession().move(node.getPath(), node.getParent().getPath() + '/' + legalName);
            node.getSession().save();
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put("form", page.getForm());
            _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
            
            results.put("id", id);
            results.put("newName", legalName);
            results.put("formId", page.getForm().getId());
        }
        catch (RepositoryException e)
        {
            getLogger().error("Form renaming failed.", e);
            results.put("message", "cannot-rename");
        }
        
        return results;
    }
    
    /**
     * Deletes a {@link FormPage}.
     * @param id The id of the page to delete
     * @return The id of the page
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, String> deletePage (String id)
    {
        Map<String, String> result = new HashMap<>();
        
        FormPage page = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(page);
        
        Form parent = page.getForm();
        
        //remove question rules references
        _removeReferencesFromQuestionsRules(page, parent);
        
        page.remove();
        
        // Remove page rules references
        _removeReferencesFromPages (id, parent);
        _removeReferencesFromQuestions(id, parent);
        parent.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", parent);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", id);
        result.put("formId", parent.getId());
        
        return result;
    }
    
    /**
     * Copies and pastes a form page.
     * @param formId The id of the form, target of the copy
     * @param pageId The id of the page to copy
     * @return The id of the created page
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, String> copyPage(String formId, String pageId)
    {
        Map<String, String> result = new HashMap<>();
        
        Form parentForm = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(parentForm);
        
        FormPage originalPage = _resolver.resolveById(pageId);
        
        // Find unique name
        String uniqueName = NameHelper.getUniqueAmetysObjectName(parentForm, originalPage.getName());
        FormPage cPage = originalPage.copyTo(parentForm, uniqueName);
        
        String copyTitle = _i18nUtils.translate(new I18nizableText("plugin.forms", "PLUGIN_FORMS_TREE_COPY_NAME_PREFIX")) + originalPage.getTitle();
        cPage.setTitle(copyTitle);
        cPage.deleteRule();
        
        parentForm.saveChanges();

        for (String epId : _copyFormEP.getExtensionsIds())
        {
            CopyFormUpdater copyFormUpdater = _copyFormEP.getExtension(epId);
            copyFormUpdater.updateFormPage(originalPage, cPage);
        }
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", parentForm);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", cPage.getId());
        result.put("formId", parentForm.getId());
        
        return result;
    }
    
    /**
     * Moves an element of the form.
     * @param id The id of the element to move.
     * @param oldParent The id of the element's parent.
     * @param newParent The id of the new element's parent.
     * @param index The index where to move. null to place the element at the end.
     * @return A map with the ids of the element, the old parent and the new parent
     * @throws Exception if an error occurs when moving an element of the form
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, String> moveObject (String id, String oldParent, String newParent, long index) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        JCRAmetysObject aoMoved = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(aoMoved);
        
        DefaultTraversableAmetysObject newParentAO = _resolver.resolveById(newParent);
        JCRAmetysObject  brother = null;
        long size = newParentAO.getChildren().getSize();
        if (index != -1 && index < size)
        {
            brother = newParentAO.getChildAt(index);
        }
        else if (index >= size)
        {
            brother = newParentAO.getChildAt(Math.toIntExact(size) - 1);
        }
        Form oldForm = getParentForm(aoMoved);
        if (oldForm != null)
        {
            result.put("oldFormId", oldForm.getId());
        }
        
        if (oldParent.equals(newParent) && brother != null)
        {
            Node node = aoMoved.getNode();
            String name = "";
            try
            {
                name = (index == size)
                            ? null 
                            : brother.getName();
                node.getParent().orderBefore(node.getName(), name);
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException(String.format("Unable to order AmetysOject '%s' before sibling '%s'", this, name), e);
            }
        }
        else
        {
            Node node = aoMoved.getNode();
            
            String name = node.getName();
            // Find unused name on new parent node
            int localIndex = 2;
            while (newParentAO.hasChild(name))
            {
                name = node.getName() + "-" + localIndex++;
            }
            
            node.getSession().move(node.getPath(), newParentAO.getNode().getPath() + "/" + name);
            
            if (brother != null)
            {
                node.getParent().orderBefore(node.getName(), brother.getName());
            }
        }
        
        if (newParentAO.needsSave())
        {
            newParentAO.saveChanges();
        }
        
        Form form = getParentForm(aoMoved);
        if (form != null)
        {
            result.put("newFormId", form.getId());
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put("form", form);
            _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        }
        
        result.put("id", id);
        
        if (aoMoved instanceof FormPage)
        {
            result.put("type", "page");
        }
        else if (aoMoved instanceof FormQuestion)
        {
            result.put("type", "question");
            result.put("questionType", ((FormQuestion) aoMoved).getType().getId());
        }
        
        result.put("newParentId", newParentAO.getId());
        result.put("oldParentId", oldParent);
        
        return result;
    }
    
    /**
     * Get the page's properties
     * @param pageId The form page's id
     * @return The page properties
     */
    @Callable  (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getPageProperties (String pageId)
    {
        try
        {
            FormPage page = _resolver.resolveById(pageId);
            return getPageProperties(page, true);
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().warn("Can't find page with id: {}. It probably has just been deleted", pageId, e);
            Map<String, Object> infos = new HashMap<>();
            infos.put("id", pageId);
            return infos;
        }
    }
    
    /**
     * Get the page's properties
     * @param page The form page
     * @param withRights <code>true</code> to have the rights in the properties
     * @return The page properties
     */
    public Map<String, Object> getPageProperties (FormPage page, boolean withRights)
    {
        Map<String, Object> infos = new HashMap<>();
        
        List<String> fullPath = new ArrayList<>();
        fullPath.add(page.getTitle());

        AmetysObject node = page.getParent();
        fullPath.add(0, node.getName()); 
        
        infos.put("type", "page");
        
        /** Use in the bus message */
        infos.put("id", page.getId());
        infos.put("title", page.getTitle());
        infos.put("formId", page.getForm().getId());
        infos.put("hasEntries", !page.getForm().getEntries().isEmpty());
        infos.put("hasChildren", page.getQuestions().size() > 0);
        
        boolean isConfigured = !page.getQuestions().stream().anyMatch(q -> !q.getType().isQuestionConfigured(q));
        infos.put("isConfigured", isConfigured);
        
        if (withRights)
        {
            infos.put("rights", _getUserRights(page));
        }
        else
        {
            infos.put("canWrite", _formDAO.hasWriteRightOnForm(_currentUserProvider.getUser(), page));
        }

        return infos;
    }
    
    /**
     * Get user rights for the given form page
     * @param page the form page
     * @return the set of rights
     */
    protected Set<String> _getUserRights (FormPage page)
    {
        UserIdentity user = _currentUserProvider.getUser();
        return _rightManager.getUserRights(user, page);
    }
    
    /**
     * Gets the ids of the path elements of a form component, i.e. the parent ids.
     * <br>For instance, if the page path is 'a/b/c', then the result list will be ["id-of-a", "id-of-b", "id-of-c"]
     * @param id The id of the form component
     * @return the ids of the path elements of a form
     */
    @Callable (rights = FormDAO.HANDLE_FORMS_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public List<String> getIdsOfPath(String id)
    {
        AmetysObject formComponent = _resolver.resolveById(id);
        
        if (!(formComponent instanceof FormPage) && !(formComponent instanceof FormQuestion))
        {
            throw new IllegalArgumentException("The given id is not a form component");
        }
        
        List<String> pathElements = new ArrayList<>();
        AmetysObject current = formComponent.getParent();
        while (!(current instanceof Form))
        {
            pathElements.add(0, current.getId());
            current = current.getParent();
        }
        
        return pathElements;
    }
    
    /**
     * Gets all pages for given parent
     * @param formId The id of the {@link Form}, defining the context from which getting children
     * @return all forms  for given parent
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public List<Map<String, Object>> getChildPages(String formId)
    {
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        return form.getPages()
                    .stream()
                    .map(p -> this.getPageProperties(p, false))
                    .toList();
    }
    
    /**
     * Get the form containing the given object.
     * @param obj the object.
     * @return the parent Form.
     */
    protected Form getParentForm(JCRAmetysObject obj)
    {
        try
        {
            JCRAmetysObject currentAo = obj.getParent();
            
            while (!(currentAo instanceof Form))
            {
                currentAo = currentAo.getParent();
            }
            
            if (currentAo instanceof Form)
            {
                return (Form) currentAo;
            }
        }
        catch (AmetysRepositoryException e)
        {
            // Ignore, just return null.
        }
        
        return null;
    }
    
    /**
     * Determines if a page is the last of form's pages.
     * @param id The page id
     * @return True if the page is the last one. 
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public boolean isLastPage (String id)
    {
        FormPage page = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(page);
        
        Form form = page.getForm();
        
        List<FormPage> pages = form.getPages();
        FormPage lastPage = pages.get(pages.size() - 1);
        
        return id.equals(lastPage.getId());
    }
    
    /**
     * Gets the branches for a form page.
     * @param pageId The id of the form page.
     * @return The branches
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getBranches (String pageId)
    {
        Map<String, Object> result = new HashMap<>();
        
        FormPage page = _resolver.resolveById(pageId);
        _formDAO.checkHandleFormRight(page);
        
        result.put("id", pageId);
        
        List<Object> questions = new ArrayList<>();
        List<FormQuestion> questionsAO = page.getQuestions();
        int index = 1;
        for (FormQuestion question : questionsAO)
        {
            if (question.getType() instanceof ChoicesListQuestionType type 
                && !type.getSourceType(question).remoteData()
                && !question.isModifiable())
            {
                try
                {
                    questions.add(_formQuestionDAO.getRules(question.getId(), index));
                }
                catch (Exception e)
                {
                    getLogger().error("an exception occured while getting rules for question " + question.getId());
                }
            }
            index++;
        }
        result.put("questions", questions);
        
        // SAX page rule
        result.put("rule", getRule(pageId));
        
        return result;
    }
    
    /**
     * Gets the rule for a form page.
     * @param id The id of the form page.
     * @return The rule, or null
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getRule (String id)
    {
        Map<String, Object> result = new HashMap<>();
        
        FormPage page = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(page);
        
        FormPageRule rule = page.getRule();
        
        if (rule != null)
        {
            result.put("type", rule.getType().name());
            String pageId = rule.getPageId();
            if (pageId != null)
            {
                try
                {
                    FormPage pageAO = _resolver.resolveById(pageId);
                    result.put("page", pageId);
                    result.put("pageName", pageAO.getTitle());
                }
                catch (UnknownAmetysObjectException e)
                {
                    // The page does not exist anymore
                }
            }
        }
        else
        {
            result = null;
        }
        
        return result;
    }
    
    /**
     * Adds a a new rule to a page.
     * @param id The id of the page
     * @param rule The rule type
     * @param page The page to jump or skip
     * @return An empty map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> addRule (String id, String rule, String page)
    {
        FormPage formPage = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(formPage);
        
        formPage.setRule(PageRuleType.valueOf(rule), page);
        formPage.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", formPage.getForm());
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        return new HashMap<>();
    }

    /**
     * Deletes a rule to a page
     * @param id The id of the page
     * @return An empty map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> deleteRule (String id)
    {
        FormPage formPage = _resolver.resolveById(id);
        _formDAO.checkHandleFormRight(formPage);
        
        formPage.deleteRule();
        formPage.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", formPage.getForm());
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _getCurrentUser(), eventParams));
        
        return new HashMap<>();
    }
    
    private void _removeReferencesFromPages (String pageId, Form parent)
    {
        List<FormPageRule> rulesWithPageId = parent.getPages().stream()
            .map(page -> page.getRule())
            .filter(rule -> rule != null && pageId.equals(rule.getPageId()))
            .collect(Collectors.toList());
        
        for (FormPageRule rule : rulesWithPageId)
        {
            rule.remove();
        }
        parent.saveChanges();
    }
    
    private void _removeReferencesFromQuestions (String pageId, Form parent)
    {
        List<FormPageRule> rulesWithPageId = parent.getQuestions().stream()
            .map(question -> question.getPageRules())
            .flatMap(List::stream)
            .filter(rule -> rule != null && pageId.equals(rule.getPageId()))
            .collect(Collectors.toList());
            
        for (FormPageRule rule : rulesWithPageId)
        {
            rule.remove();
        }
        parent.saveChanges();
    }
    
    private void _removeReferencesFromQuestionsRules (FormPage page, Form parent)
    {
        for (FormQuestion questionToDelete : page.getQuestions())
        {
            parent.deleteQuestionsRule(questionToDelete.getId());
        }
    }
    
    /**
     * Provides the current user.
     * @return the user which cannot be <code>null</code>.
     */
    protected UserIdentity _getCurrentUser()
    {      
        return _currentUserProvider.getUser();
    }
}
