/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;

import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;
import com.opensymphony.workflow.spi.Step;

/**
 * Generate the entry of a form
 */
public class FormEntryPropertiesGenerator extends FormAdminEntryInformationGenerator
{
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;

    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
    }
    
    @Override
    protected void _addAdditionalEntryAttributes(FormEntry entry, AttributesImpl attrs)
    {
        super._addAdditionalEntryAttributes(entry, attrs);
        
        try
        {
            attrs.addCDATAAttribute("access", Boolean.toString(_rightManager.currentUserHasRight("REPOSITORY_Rights_Access", "/repository") == RightResult.RIGHT_ALLOW));
            attrs.addCDATAAttribute("uuid", entry.getNode().getIdentifier());
            attrs.addCDATAAttribute("path", entry.getNode().getPath());
        }
        catch (Exception e) 
        {
            getLogger().error("Can get uuid from answer with id '" + entry.getId() + "'", e); 
        }
        
        if (entry.getForm().hasWorkflow())
        {
            AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(entry);
            WorkflowDescriptor workflowDescriptor = workflow.getWorkflowDescriptor(entry.getForm().getWorkflowName());
            Step currentStep = (Step) workflow.getCurrentSteps(entry.getWorkflowId()).iterator().next();
            
            StepDescriptor stepDescriptor = workflowDescriptor.getStep(currentStep.getStepId());
            I18nizableText workflowStepName = new I18nizableText("application", stepDescriptor.getName());
            if ("application".equals(workflowStepName.getCatalogue()))
            {
                attrs.addCDATAAttribute("icon", "/plugins/cms/resources_workflow/" + workflowStepName.getKey() + "-small.png");
            }
            else
            {
                String pluginName = workflowStepName.getCatalogue().substring("plugin.".length());
                attrs.addCDATAAttribute("icon", "/plugins/" + pluginName + "/resources/img/workflow/" + workflowStepName.getKey() + "-small.png");
            }
            attrs.addCDATAAttribute("workflow", _i18nUtils.translate(workflowStepName));
        }
    }
}
