/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.dao.FormQuestionDAO.FormEntryValues;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.plugins.workflow.support.WorkflowHelper;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Generate the entry for front edition
 */
public class FormFrontEditEntryInformationGenerator extends FormAdminEntryInformationGenerator
{
    /** The workflow helper component */
    protected WorkflowHelper _workflowHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _workflowHelper = (WorkflowHelper) smanager.lookup(WorkflowHelper.ROLE);
    }
    
    private boolean _isSubmitter()
    {
        return parameters.getParameterAsBoolean("isSubmitter", false);
    }
    
    @Override
    protected void _checkRights(FormEntry entry)
    {
        UserIdentity user = _currentUserProvider.getUser();
        boolean isSubmitter = _isSubmitter() && entry.getUser().equals(user); // Add the additional test that the entry user is the current user to be sure is the submitter
        if (!isSubmitter) 
        {
            super._checkRights(entry);
        }
    }
    
    @Override
    protected List<FormQuestion> _getQuestions(FormEntry entry, Optional<Long> currentStepId)
    {
        boolean onlyReadableQuestion = _isSubmitter(); // Get only readable question if this form is for submitter edition. Else get all question for admin edition.
        return _formQuestionDAO.getRuleFilteredQuestions(entry.getForm(), new FormEntryValues(null, entry), currentStepId, false, onlyReadableQuestion);
    }
    
    @Override
    protected void _saxQuestion(FormQuestion question, FormEntry entry) throws SAXException
    {
        Long currentStepId = _formEntryDAO.getCurrentStepId(entry);
        
        FormQuestionType questionType = question.getType();
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("type", questionType.getId());
        attrs.addCDATAAttribute("id", question.getId());
        attrs.addCDATAAttribute("name", question.getNameForForm());
        attrs.addCDATAAttribute("canWrite", String.valueOf(_isSubmitter() || question.canWrite(currentStepId))); // The question can be write if this form is for submitter edition.
        attrs.addCDATAAttribute("canRead", "true"); // Already filtered questions which can be read
        
        XMLUtils.startElement(contentHandler, "question", attrs);
        RepositoryDataContext context = RepositoryDataContext.newInstance().withObject(question);
        question.dataToSAX(contentHandler, context);
        
        XMLUtils.startElement(contentHandler, "additional-infos");
        questionType.saxAdditionalInfos(contentHandler, question);
        XMLUtils.endElement(contentHandler, "additional-infos");
        
        XMLUtils.startElement(contentHandler, "entry-value");
        questionType.saxEntryValue(contentHandler, question, entry);
        XMLUtils.endElement(contentHandler, "entry-value");
        XMLUtils.endElement(contentHandler, "question");
    }
    
    @Override
    protected void _addAdditionalEntryAttributes(FormEntry entry, AttributesImpl attrs)
    {
        super._addAdditionalEntryAttributes(entry, attrs);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        String actionId = (String) request.get("actionId");
        if (StringUtils.isNotBlank(actionId))
        {
            WorkflowDescriptor workflowDescriptor = _workflowHelper.getWorkflowDescriptor(entry.getForm().getWorkflowName());
            ActionDescriptor action = workflowDescriptor.getAction(Integer.parseInt(actionId));
            Map<String, Object> metaAttributes = action.getMetaAttributes();
            String actionType = (String) metaAttributes.get("action-type");
            if (StringUtils.isNotBlank(actionType))
            {
                attrs.addCDATAAttribute("actionType", actionType);
            }
            
            attrs.addCDATAAttribute("actionId", actionId);
        }
    }
}
