/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;

import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.User;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;

/**
 * The helper to schedule opening of form
 */
public class FormInvitationsHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon ROLE. */
    public static final String ROLE = FormInvitationsHelper.class.getName();
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The users manager */
    protected UserManager _userManager;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The limited entries helper */
    protected LimitedEntriesHelper _limitedEntriesHelper;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** The form mail helper */
    protected FormMailHelper _formMailHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _limitedEntriesHelper = (LimitedEntriesHelper) manager.lookup(LimitedEntriesHelper.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _formMailHelper = (FormMailHelper) manager.lookup(FormMailHelper.ROLE);
    }
    
    /**
     * Sends invitations emails.
     * @param formId The id of the form.
     * @param message The message content.
     * @param siteName The site name.
     * @param language the language
     * @return An empty map
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> sendInvitations (String formId, String message, String siteName, String language)
    {
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        List<Page> pages = _getFormPages(formId, siteName);
        if (pages.isEmpty())
        {
            throw new IllegalAccessError("Can't send invitations for form id '" + formId + "' because it is not published.");
        }
        
        _formMailHelper.sendInvitationMails(form, _getUsersToInvite(form), message);
        return new HashMap<>();
    }

    private List<Page> _getFormPages(String formId, String siteName)
    {
        List<Page> pages = _formDAO.getFormPage(formId, siteName)
            .stream()
            .filter(Page.class::isInstance)
            .map(Page.class::cast)
            .toList();
        return pages;
    }
    
    /**
     * Get users to invite to the form (user with mail and who are not answered yet to the form)
     * @param form the form
     * @return the users to invite
     */
    protected List<User> _getUsersToInvite(Form form)
    {
        // Can invite users if the form is on anonymous read access
        if (_rightManager.hasAnonymousReadAccess(form))
        {
            return List.of();
        }
        
        return _rightManager.getReadAccessAllowedUsers(form).resolveAllowedUsers(true)
                    .stream()
                    .filter(id -> !_limitedEntriesHelper.hasUserAlreadyAnswer(form, id, null)) // User who are not answered yet
                    .map(id -> _userManager.getUser(id)) // Get user object
                    .filter(Objects::nonNull) // User not null
                    .filter(u -> StringUtils.isNotEmpty(u.getEmail())) // User with a mail
                    .toList();
    }
    
    /**
     * Get users to invite to the form (user with mail and who are not answered yet to the form)
     * @param formId The id of the form.
     * @return The number of users to invite
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public int getNbOfUsersToInvite(String formId)
    {
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        return _getUsersToInvite(form).size();
    }
}
