/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.helper;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.forms.FormEvents;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The helper to schedule opening of form
 */
public class ScheduleOpeningHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon ROLE. */
    public static final String ROLE = ScheduleOpeningHelper.class.getName();
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** Observer manager. */
    protected ObservationManager _observationManager;
    
    /**
     * The form status
     */
    public enum FormStatus 
    {
        /** The form is not opened yet */
        COMING,
        /** The form is opened */
        OPEN,
        /** The form is closed */
        OVER;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
    }
    
    /**
     * Get the scheduled dates of a form
     * @param formId the id of the form
     * @return a map with start date and end date
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getScheduledDates(String formId)
    {
        Map<String, Object> dates = new HashMap<> ();
        
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        LocalDate startDate = form.getStartDate();
        if (startDate != null)
        {
            dates.put(Form.START_DATE, DateUtils.localDateToString(startDate));
        }
        
        LocalDate endDate = form.getEndDate();
        if (endDate != null)
        {
            dates.put(Form.END_DATE, DateUtils.localDateToString(endDate));
        }
        
        return dates;
    }
    
    /**
     * Set date of publication of form
     * @param formId The form id
     * @param startDateAsStr The start date. Can be null.
     * @param endDateAsStr The end date. Can be null.
     * @return true if operation has succeeded.
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public boolean setScheduledDate (String formId, String startDateAsStr, String endDateAsStr)
    {
        Form form = _resolver.resolveById(formId);
        _formDAO.checkHandleFormRight(form);
        
        LocalDate startDate = startDateAsStr != null ? LocalDate.parse(startDateAsStr) : null;
        LocalDate endDate = endDateAsStr != null ? LocalDate.parse(endDateAsStr) : null;
        
        form.setStartDate(startDate);
        form.setEndDate(endDate);
        form.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("form", form);
        _observationManager.notify(new Event(FormEvents.FORM_MODIFIED, _currentUserProvider.getUser(), eventParams));
        
        return true;
    }
    
    /**
     * Get the opening status of the form 
     * @param form the form
     * @return 'open', 'over', or 'coming' depending on today's date
     */
    public FormStatus getStatus(Form form)
    {
        LocalDate startDate = form.getStartDate();
        LocalDate endDate = form.getEndDate();
        
        LocalDate now = LocalDate.now();
        
        if (startDate != null && startDate.isAfter(now))
        {
            return FormStatus.COMING;
        }
        
        if (endDate != null && endDate.isBefore(now))
        {
            return FormStatus.OVER;
        }
        
        return FormStatus.OPEN;
    }
}
