/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.forms.helper.FormElementDefinitionHelper;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.forms.repository.type.AbstractMatrixElementType;
import org.ametys.plugins.forms.repository.type.Matrix;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.type.ModelItemTypeConstants;


/**
 * Class for creating matrix form question
 */
public class MatrixQuestionType extends AbstractFormQuestionType implements MultipleAwareFormQuestionType
{
    /** Constant for element definition name of the matrix grid*/
    public static final String ATTRIBUTE_GRID = "matrix-grid";
    
    /** Constant for default title */
    public static final String DEFAULT_TITLE = "PLUGIN_FORMS_QUESTION_DEFAULT_TITLE_MATRIX";
    
    /** JSON helper */
    protected JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    protected List<ModelItem> _getModelItems()
    {
        List<ModelItem> modelItems = super._getModelItems();
        
        ElementDefinition matrixWidget = FormElementDefinitionHelper.getElementDefinition(ATTRIBUTE_GRID, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_MATRIX_GRID", "PLUGINS_FORMS_QUESTIONS_DIALOG_MATRIX_GRID_DESC", null);
        matrixWidget.setWidget("edition.matrix-grid");
        modelItems.add(matrixWidget);
        
        modelItems.add(getMultipleModelItem());
        
        return modelItems;
    }
    
    @Override
    protected SimpleViewItemGroup _getMainTab()
    {
        SimpleViewItemGroup mainFieldset = super._getMainTab();
        
        ViewElement widgetMatrix = new ViewElement();
        widgetMatrix.setDefinition((ElementDefinition< ? >) getModel().getModelItem(ATTRIBUTE_GRID));
        mainFieldset.addViewItem(widgetMatrix);
        
        return mainFieldset;
    }
    
    @Override
    protected SimpleViewItemGroup _getAdvancedTab()
    {
        SimpleViewItemGroup advancedFieldset = super._getAdvancedTab();
        
        advancedFieldset.addViewItem(getMultipleViewElement(getModel()));
        
        return advancedFieldset;
    }
    
    @Override
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException
    {
        super.saxAdditionalInfos(contentHandler, question);
        
        Map<String, String> rows = getRows(question);
        if (rows != null)
        {
            XMLUtils.startElement(contentHandler, "options");
            for (String key : rows.keySet())
            {
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("value", key);
                XMLUtils.createElement(contentHandler, "option", attr, rows.get(key));
            }
            XMLUtils.endElement(contentHandler, "options");
            
            Map<String, String> columns = getColumns(question);
            XMLUtils.startElement(contentHandler, "columns");
            for (String key : columns.keySet())
            {
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("value", key);
                XMLUtils.createElement(contentHandler, "column", attr, columns.get(key));
            }
            XMLUtils.endElement(contentHandler, "columns");
        }
    }
    
    /**
     * Get the matrix columns
     * @param question the question
     * @return the columns of the matrix, null if empty
     */
    @SuppressWarnings("unchecked")
    public Map<String, String> getColumns(FormQuestion question)
    {
        Map<String, Object> values = _jsonUtils.convertJsonToMap(question.getValue(ATTRIBUTE_GRID));
        return (Map<String, String>) values.get("columns");
    }
    
    /**
     * Get the matrix rows
     * @param question the question
     * @return the rows of the matrix, null if empty
     */
    @SuppressWarnings("unchecked")
    public Map<String, String> getRows(FormQuestion question)
    {
        Map<String, Object> values = _jsonUtils.convertJsonToMap(question.getValue(ATTRIBUTE_GRID));
        return (Map<String, String>) values.get("options");
    }
    
    public String getStorageType(FormQuestion question)
    {
        return AbstractMatrixElementType.MATRIX_REPOSITORY_DATA_TYPE;
    }
    
    public I18nizableText getDefaultTitle()
    {
        return new I18nizableText("plugin.forms", DEFAULT_TITLE);
    }
    
    @Override
    public boolean isQuestionConfigured(FormQuestion question)
    {   
        Map<String, String> rows = getRows(question);
        Map<String, String> columns = getColumns(question);
        return rows != null && !rows.isEmpty() && columns != null && !columns.isEmpty();
    }
    
    @Override
    public void saxEntryValue(ContentHandler contentHandler, FormQuestion question, FormEntry entry) throws SAXException
    {
        super.saxEntryValue(contentHandler, question, entry);
        
        XMLUtils.startElement(contentHandler, "additional-infos");

        Matrix matrix = entry.getValue(question.getNameForForm());
        if (matrix != null)
        {
            Map<String, String> columns = getColumns(question);
            if (columns != null)
            {
                XMLUtils.startElement(contentHandler, "columns");
                for (String colValue : columns.keySet())
                {
                    AttributesImpl attr = new AttributesImpl();
                    attr.addCDATAAttribute("value", colValue);
                    XMLUtils.createElement(contentHandler, "column", attr, columns.get(colValue));
                }
                XMLUtils.endElement(contentHandler, "columns");

                Map<String, String> rows = getRows(question);
                if (rows != null)
                {
                    XMLUtils.startElement(contentHandler, "options");
                    for (String rowValue : rows.keySet())
                    {
                        List<String> opts = matrix.get(rowValue);
                        
                        AttributesImpl attr = new AttributesImpl();
                        attr.addCDATAAttribute("label", rows.get(rowValue));
                        attr.addCDATAAttribute("value", rowValue);
                        
                        XMLUtils.startElement(contentHandler, "option", attr);
                        for (String opt : columns.keySet())
                        {
                            AttributesImpl attrValue = new AttributesImpl();
                            attrValue.addCDATAAttribute("value", rowValue);
                            boolean isChecked = opts != null && opts.contains(opt);
                            attrValue.addCDATAAttribute("checked", String.valueOf(isChecked));
                            
                            XMLUtils.createElement(contentHandler, "value", attrValue);
                        }
                        XMLUtils.endElement(contentHandler, "option");
                    }
                    XMLUtils.endElement(contentHandler, "options");
                }
            }
        }
        
        XMLUtils.endElement(contentHandler, "additional-infos");
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished(FormQuestion question)
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished(question);
        fieldNames.add(ATTRIBUTE_GRID);
        fieldNames.add(ATTRIBUTE_MULTIPLE);
        return fieldNames;
    }
    
    @Override
    public String getJSRenderer(FormQuestion question)
    {
        return "Ametys.plugins.forms.helper.SearchEntriesGridHelper.renderMatrix";
    }
}
