/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.ametys.cms.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.plugins.forms.dao.FormQuestionDAO;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.types.MandatoryAwareFormQuestionType;
import org.ametys.plugins.forms.question.types.RestrictiveQuestionType;
import org.ametys.plugins.forms.repository.FormPageRule.PageRuleType;
import org.ametys.plugins.forms.repository.type.Rule;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.NotUniqueTypeException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.site.Site;

/**
 * {@link AmetysObject} for storing form
 */
public class FormQuestion extends DefaultTraversableAmetysObject<FormQuestionFactory> implements ModifiableModelAwareDataAwareAmetysObject, SiteAwareAmetysObject
{
    /** Constant for page id attribute. */
    public static final String ATTRIBUTE_PAGE_ID = "pageId";
    /** Constant for title attribute. */
    public static final String ATTRIBUTE_TITLE = "title";
    /** Constant for default identifier attribute. */
    public static final String ATTRIBUTE_NAME_FOR_FORM = "name-for-form";
    /** Constant for type attribute. */
    public static final String ATTRIBUTE_TYPE = "type";
    /** Constant for rules repeater. */
    public static final String ATTRIBUTE_RULES = "rules";
    /** Constant for rule attribute. */
    public static final String ATTRIBUTE_RULE = "rule";
    
    /**
     * Creates a {@link FormQuestion}.
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link FormFactory} which creates the AmetysObject.
     */
    public FormQuestion(Node node, String parentPath, FormQuestionFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Get the page to which this question belongs.
     * @return the page to which this question belongs.
     * @throws AmetysRepositoryException if a repository error occurs when retrieving the page attached to a form
     */
    public FormPage getFormPage() throws AmetysRepositoryException
    {
        return getParent();
    }
    
    /**
     * Get the form to which this question belongs.
     * @return the form to which this question belongs.
     * @throws AmetysRepositoryException if a repository error occurs when retrieving a form
     */
    public Form getForm() throws AmetysRepositoryException
    {
        return getFormPage().getForm();
    }
    
    /**
     * Retrieves the type.
     * @return the type.
     */
    public FormQuestionType getType() 
    {
        String typeId = getValue(ATTRIBUTE_TYPE);
        return _getFactory().getFormQuestionTypeExtensionPoint().getExtension(typeId);
    }
    
    /**
     * Set the type id.
     * @param typeId the type id.
     */
    public void setTypeId(String typeId) 
    {
        setValue(ATTRIBUTE_TYPE, typeId);
    }
    
    /**
     * Set the type id.
     * @param type the type.
     */
    public void setTypeId(FormQuestionType type) 
    {
        setValue(ATTRIBUTE_TYPE, type.getId());
    }
    
    /**
     * Retrieves the name for form.
     * @return the name for form.
     */
    public String getNameForForm()
    {
        return getValue(ATTRIBUTE_NAME_FOR_FORM);
    }
    
    /**
     * Set the name for form.
     * @param name the name for form.
     */
    public void setNameForForm(String name) 
    {
        setValue(ATTRIBUTE_NAME_FOR_FORM, name);
    }
    
    /**
     * Retrieves the title.
     * @return the title.
     */
    public String getTitle() 
    {
        return getValue(ATTRIBUTE_TITLE);
    }
    
    /**
     * Set the title.
     * @param title the title.
     */
    public void setTitle(String title) 
    {
        setValue(ATTRIBUTE_TITLE, title);
    }
    
    /**
     * Determines if the question is mandatory.
     * @return true if the question is mandatory.
     */
    public boolean isMandatory() 
    {
        return getType() instanceof MandatoryAwareFormQuestionType ? ((MandatoryAwareFormQuestionType) getType()).isMandatory(this) : false;
    }

    /**
     * Determines if the question is read restricted.
     * @return true if the question is read restricted.
     */
    public boolean isReadRestricted() 
    {
        return getType() instanceof RestrictiveQuestionType ? ((RestrictiveQuestionType) getType()).isReadRestricted(this) : false;
    }
    
    /**
     * Determines if we can read the question for the current step.
     * @param currentStep the current step of the form entry
     * @return <code>true</code> if we can read the question for the current step.
     */
    public boolean canRead(Long currentStep) 
    {
        if (getType() instanceof RestrictiveQuestionType type)
        {
            return !isReadRestricted() || type.getReadingSteps(this).contains(currentStep);
        }
        
        return true;
    }
    
    /**
     * Determines if the question is modifiable
     * @return true if the question is modifiable.
     */
    public boolean isModifiable() 
    {
        return getType() instanceof RestrictiveQuestionType ? ((RestrictiveQuestionType) getType()).isModifiable(this) : false;
    }
    
    /**
     * Determines if we can write the question for the current step.
     * @param currentStep the current step of the form entry
     * @return <code>true</code> if we can write the question for the current step.
     */
    public boolean canWrite(Long currentStep) 
    {
        if (currentStep.equals(RestrictiveQuestionType.INITIAL_WORKFLOW_ID))
        {
            return true;
        }
        
        if (getType() instanceof RestrictiveQuestionType type)
        {
            return isModifiable() && type.getWritingSteps(this).contains(currentStep);
        }
        
        return false;
    }
    
    /**
     * Get the question rules
     * @return A RuleTypeQuestion's name
     */
    public List<Rule> getQuestionRules()
    {
        List<Rule> rules = new ArrayList<>();
        ModelAwareRepeater value = getRepeater(ATTRIBUTE_RULES);
        if (value != null)
        {
            rules = value.getEntries()
                .stream()
                .map(e -> (Rule) e.getValue(ATTRIBUTE_RULE))
                .collect(Collectors.toList());
        }
        return rules;
    }
    
    /**
     * Get the first question rule
     * @return A RuleTypeQuestion's name
     */
    public Optional<Rule> getFirstQuestionRule()
    {
        List<Rule> questionRules = getQuestionRules();
        return questionRules.isEmpty() ? Optional.empty() : Optional.of(questionRules.get(0));
    }
    
    /**
     * Get the page rules
     * @return the page rules
     * @throws AmetysRepositoryException if an error occurs.
     */
    public List<FormPageRule> getPageRules () throws AmetysRepositoryException
    {
        List<FormPageRule> rules = new LinkedList<>();
        
        if (hasChild(FormQuestionDAO.RULES_ROOT))
        {
            ModifiableTraversableAmetysObject ruleRoot = getChild(FormQuestionDAO.RULES_ROOT);
            
            rules.addAll(ruleRoot.getChildren()
                    .stream()
                    .map(rule -> (FormPageRule) rule)
                    .collect(Collectors.toList()));
        }
        
        return rules;
    }
    
    /**
     * Determines if a rule with given option exists
     * @param option the option
     * @return true if the rule exists
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean hasPageRule (String option) throws AmetysRepositoryException
    {
        if (hasChild(FormQuestionDAO.RULES_ROOT))
        {
            return ((ModifiableTraversableAmetysObject) getChild(FormQuestionDAO.RULES_ROOT)).hasChild(option);
        }
        return false;
    }
    
    /**
     * Add a page rule for branching
     * @param option the chosen option
     * @param ruleType the page rule type
     * @param pageId Id of the page to jump or skip. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void addPageRules (String option, PageRuleType ruleType, String pageId) 
    {
        ModifiableTraversableAmetysObject rulesRoot;
        if (hasChild(FormQuestionDAO.RULES_ROOT))
        {
            rulesRoot = getChild(FormQuestionDAO.RULES_ROOT);
        }
        else
        {
            rulesRoot = createChild(FormQuestionDAO.RULES_ROOT, "ametys:collection");
        }
        
        FormPageRule ruleNode = rulesRoot.createChild(option, "ametys:form-page-rule");
        ruleNode.setType(ruleType);
        ruleNode.setOption(option);
        if (ruleType == PageRuleType.JUMP || ruleType == PageRuleType.SKIP)
        {
            ruleNode.setPageId(pageId);
        }
    }
    
    /**
     * Delete a rule
     * @param option the option to delete
     * @throws AmetysRepositoryException  if an error occurs.
     */
    public void deletePageRule (String option) throws AmetysRepositoryException
    {
        ModifiableTraversableAmetysObject rulesRoot = getChild(FormQuestionDAO.RULES_ROOT);
        
        if (rulesRoot.hasChild(option))
        {
            ((FormPageRule) rulesRoot.getChild(option)).remove();
        }
    }
    
    @Override
    public void copyTo(ModifiableDataHolder dataHolder) throws UndefinedItemPathException, BadItemTypeException, UnknownTypeException, NotUniqueTypeException
    {
        ModifiableModelAwareDataAwareAmetysObject.super.copyTo(dataHolder);
        dataHolder.removeValue(ATTRIBUTE_RULES);
    }
    
    public String getSiteName() throws AmetysRepositoryException
    {
        return getForm().getSiteName();
    }

    public Site getSite() throws AmetysRepositoryException
    {
        return getForm().getSite();
    }

    public ModifiableIndexableDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, _getFactory().getFormQuestionModel());
    }

    /**
     * Returns true if the question is cacheable.
     * @return true if the question is cacheable.
     */
    public boolean isCacheable()
    {
        return getType().isCacheable(this);
    }
}
