/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.repository.type;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.ametys.core.model.type.AbstractElementType;
import org.ametys.plugins.forms.repository.type.Rule.QuestionRuleType;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Abstract class for rule element types
 */
public abstract class AbstractRuleElementType extends AbstractElementType<Rule>
{
    /** Constant for type rule */
    public static final String RULE_REPOSITORY_DATA_TYPE = "rule";
    
    @Override
    public Rule convertValue(Object value)
    {
        if (value == null)
        {
            return null;
        }
        else if (value instanceof String)
        {
            return _string2Rule((String) value);
        }
        else
        {
            throw new BadItemTypeException("Try to convert the non " + getId() + " JSON object '" + value + "' into a " + getId());
        }
    }
    
    @Override
    public String toString(Rule value)
    {
        return _jsonUtils.convertObjectToJson(value.toJson());
    }

    @Override
    public Object fromJSONForClient(Object json, DataContext context)
    {
        if (json instanceof List)
        {
            List<Rule> ruleList = new ArrayList<>();
            for (Object rule : (List) json)
            {
                ruleList.add(convertValue(rule));
            }
            
            return ruleList.toArray(new Rule[ruleList.size()]);
        }
        else if (json instanceof Rule[])
        {
            List<Rule> ruleList = new ArrayList<>();
            for (Rule rule : (Rule[]) json)
            {
                ruleList.add(convertValue(rule));
            }
            
            return ruleList.toArray(new Rule[ruleList.size()]);
        }
        return convertValue(json);
    }
    
    @Override
    public boolean isSimple()
    {
        return false;
    }
    
    @Override
    public boolean isCompatible(Object value)
    {
        return super.isCompatible(value)
            || value instanceof String || value instanceof String[]
            || value instanceof Rule || value instanceof Rule[];
    }
    
    /**
     * Map json to rule
     * @param value the json value
     * @return the rule
     */
    protected Rule _string2Rule(String value)
    {
        Map<String, Object> castMap = _jsonUtils.convertJsonToMap(value);
        
        return new Rule(
            (String) castMap.get("question"),
            (String) castMap.get("option"), 
            QuestionRuleType.valueOf((String) castMap.get("action"))
        );
    }
}
