/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;

import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.spi.Step;

/**
 * This class creates multiple toggle buttons each representing a workflow step on form entries
 */
public class FormEntryWorkflowStepsClientSideElement extends AbstractFormEntriesWorkflowStepsClientSideElement
{
    @Override
    protected String _getDefaultMenuClassName()
    {
        return "Ametys.plugins.forms.workflow.FormEntryWorkflowMenu";
    }
    
    @Override
    protected String _getDefaultActionClassName()
    {
        return "Ametys.plugins.forms.workflow.FormEntryWorkflowAction.doAction";
    }

    @Override
    protected List<ScriptFile> _getAdditionalScriptFiles()
    {
        List<ScriptFile> scriptFiles = new ArrayList<>();
        scriptFiles.add(new ScriptFile("/plugins/forms/resources/js/Ametys/plugins/forms/workflow/AbstractFormEntriesWorkflowAction.js"));
        scriptFiles.add(new ScriptFile("/plugins/forms/resources/js/Ametys/plugins/forms/workflow/FormEntryWorkflowAction.js"));
        scriptFiles.add(new ScriptFile("/plugins/forms/resources/js/Ametys/plugins/forms/workflow/AbstractFormEntriesWorkflowMenu.js"));
        scriptFiles.add(new ScriptFile("/plugins/forms/resources/js/Ametys/plugins/forms/workflow/FormEntryWorkflowMenu.js"));
        return scriptFiles;
    }
    
    @Override
    protected String _getSelectionTargetId()
    {
        return "^(form-target)$";
    }
    
    @Callable (rights = FormEntryDAO.HANDLE_FORMS_ENTRIES_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 1)
    @SuppressWarnings("unchecked")
    @Override
    public Map<String, Object> getWorkflowState(List<Object> entryIds, String formId, String scriptId)
    {
        List<Map<String, Object>> entries = new ArrayList<> ();
        
        List<FormEntry> formEntries = entryIds.stream()
                .filter(String.class::isInstance)
                .map(String.class::cast)
                .map(this::_resolveEntry)
                .collect(Collectors.toList());
        
        for (FormEntry entry : formEntries)
        {
            Workflow workflow = _workflowProvider.getAmetysObjectWorkflow(entry);
            for (Script script : _scripts)
            {
                if (script.getId().equals(scriptId))
                {
                    Map<String, Object> entryMap = new HashMap<> ();
                    
                    long workflowId = entry.getWorkflowId();
                    List<Step> steps = workflow.getCurrentSteps(workflowId);
                    List<Integer> stepIds = new ArrayList<>();
                    for (Step step : steps)
                    {
                        stepIds.add(step.getStepId());
                    }
                    
                    Integer workflowStepId = (Integer) script.getParameters().get("workflow-step");
                    if (stepIds.contains(workflowStepId))
                    {
                        Map<String, Object> transientVars = new HashMap<>();
                        
                        // Used to check the rights on the form's content
                        transientVars.put(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY, entry);
                        transientVars.put(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY, new ArrayList<> ());
                        
                        int[] availableActions = workflow.getAvailableActions(workflowId, transientVars);
                        Arrays.sort(availableActions);
                        
                        List<Integer> activeActions = new ArrayList<>();
                        for (int actionId : (List<Integer>) script.getParameters().get("workflow-actions-ids"))
                        {
                            if (Arrays.binarySearch(availableActions, actionId) >= 0)
                            {
                                activeActions.add(actionId);
                            }
                        }
                        
                        entryMap.put("id", entry.getId());
                        entryMap.put("actions", activeActions);
                        entries.add(entryMap);
                    }
                }
            }
        }
        
        Map<String, Object> results = new HashMap<> ();
        results.put("entries", entries);
        return results;
    }
    
    private FormEntry _resolveEntry(String entryId)
    {
        return _resolver.resolveById(entryId);
    }
}
