/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

var DashboardHelper = {
    
    /** 
     * The recorded dashboards 
     */
    dashboards: [],
    
    /**
     * The site name
     */
    siteName: null,
    
    /**
     * The language
     */
    lang: null,
    
    /**
     * The context path
     */
    contextPath: null,
    
    /**
     * The number of entries by page
     */
    nbEntriesByPage: null,
    
    /**
     * The sort order for submited date
     */
    entriesSortOrder: null,
    
    /**
     * The name of the xsl for dashboard rendering
     */
    dashboardView: null,
    
    /**
     * Url for getting entry values. Default to "/_plugins/forms/[LANG]/dashboard/form-entry/view.html"
     */
    entryUrl: "/dashboard/form-entry/view.html",
    
    /**
     * The loading selector
     */
    LOADING_SELECTOR: '[data-ametys-dashboard-role=loading]',
    
    /**
     * The loading selector for dialog
     */
    DIALOG_LOADING_SELECTOR: '[data-ametys-dashboard-dialog-role=loading]', 
    
    /**
     * The results selector
     */
    RESULTS_SELECTOR: '[data-ametys-dashboard-role=results]',
    
    /**
     * The filters selector
     */
    FILTERS_SELECTOR: '[data-ametys-dashboard-role=filters]',
    
    /**
     * The minimum number of results required to display the search box on filters's input
     */
    FILTER_MIN_SEARCH_RESULTS: 20,
    
    /**
     * The default name of the xsl for dashboard rendering
     */
    DEFAULT_DASHBOARD_VIEW: "default",
    
    /**
     * The default name of the xsl for admin dashboard rendering
     */
    DEFAULT_ADMIN_DASHBOARD_VIEW: "default",
    
    /**
     * Function to call to show entry history
     */
    onShowHistory: function() {},
    
    /**
     * Function to call to show entry values
     */
    onShowEntry: function() {},
    
    /**
     * Function to call to show send mail dialog
     */
    onShowSendMailDialog: function() {},
    
    /**
     * Function to call to show comment dialog
     */
    onShowCommentDialog: function() {},
    
    /**
     * The current page
     * @private
     */
    _currentPage: 1,
    
    /**
     * Initialize the dashboard helper
     * @param {String} uniqueId the dashboard unique id
     * @param {Object} config the dashboard configuration
     * @param {Function} config.onShowEntry the function to call to show entry values
     * @param {Function} config.onShowHistory the function to call to show entry history
     * @param {Function} [config.onShowError] the function to call in case of error
     * @param {String} config.siteName The site name
     * @param {String} config.ctxPath the context path
     * @param {String} [config.entryUrl] the URL for getting entry values
     * @param {String} config.lang The language
     */
    initialize: function(uniqueId, config)
    {
        var dashboardCfg = DashboardHelper.dashboards[uniqueId] || {entryUrl: DashboardHelper.entryUrl, dashboardView: DashboardHelper.DEFAULT_DASHBOARD_VIEW};
        
        $j.extend(dashboardCfg, config);
        dashboardCfg.isAdminDashboard = false;
        
        DashboardHelper.dashboards[uniqueId] = dashboardCfg;
        
        // Show dashboard
        DashboardHelper.showDashboard(uniqueId);
    
        DashboardHelper._fixScrollForEdition();
    },
    
    /**
     * Initialize the admin dashboard helper
     * @param {String} uniqueId the dashboard unique id
     * @param {Object} config the dashboard configuration
     * @param {Function} config.onShowDashboard the function to call after loading dashboard
     * @param {Function} config.onShowActionDialog the function to call to show the action dialog
     * @param {Function} config.onHideActionDialog the function after workflow action is done
     * @param {Function} [config.onShowError] the function to call in case of error
     * @param {Integer} config.nbEntriesByPage the number of entries by page
     * @param {String} config.entriesSortOrder the order for entries sorting
     * @param {String} [config.dashboardView=default] the name of the xsl for admin dashboard rendering
     */
    initializeAdmin: function(uniqueId, config)
    {
        var dashboardCfg = DashboardHelper.dashboards[uniqueId] || {entryUrl: DashboardHelper.entryUrl, filterMinSearchResults: DashboardHelper.FILTER_MIN_SEARCH_RESULTS, dashboardView: DashboardHelper.DEFAULT_ADMIN_DASHBOARD_VIEW};
        
        $j.extend(dashboardCfg, config);
        dashboardCfg.isAdminDashboard = true;
        
        DashboardHelper.dashboards[uniqueId] = dashboardCfg;
        
        // Show forms
        DashboardHelper.showAdminDashboard(uniqueId);
        
        DashboardHelper._fixScrollForEdition();
    },
    
    /**
     * Initialize the dashboard helper for content's form
     * @param {String} uniqueId the dashboard unique id
     * @param {Object} config the dashboard configuration
     * @param {Function} config.onShowEntry the function to call to show entry values
     * @param {Function} config.onShowHistory the function to call to show entry history
     * @param {Function} [config.onShowError] the function to call in case of error
     * @param {String} config.siteName The site name
     * @param {String} config.ctxPath the context path
     * @param {String} [config.entryUrl] the URL for getting entry values
     * @param {String} config.lang The language
     */
    initializeContentForms: function(uniqueId, config)
    {
        var dashboardCfg = DashboardHelper.dashboards[uniqueId] || {entryUrl: DashboardHelper.entryUrl, dashboardView: DashboardHelper.DEFAULT_DASHBOARD_VIEW};
        
        $j.extend(dashboardCfg, config);
        dashboardCfg.isAdminDashboard = false;
        
        DashboardHelper.dashboards[uniqueId] = dashboardCfg;
    },
    
    _fixScrollForEdition: function()
    {
        // Listener to be able to scroll to top of the form when needed (show form errors) when form is display in dashboard popup (iframe)
        // Message is post by AmetysFormsHelper._showFormErrors
        window.addEventListener("message", function (event){
        
            if (event.isTrusted && event.data.type == "ametys-forms" && event.data.action == "scrollTop")
            {
                let $currentEditPopup = $j("[data-dashboard-edit-dialog]:visible").find('[data-dashboard-edit-values]');
                $currentEditPopup.scrollTop(0);
                
                // Force resize of iframe
                let iframe = $currentEditPopup.find('iframe');
                iframe[0].style.height = iframe[0].contentWindow.document.body.scrollHeight + 'px';
            }
        })
    },
    
    /**
     * Open the history dialog box
     * @param {String} uniqueId the dashboard unique id
     * @param {String} formId The form id
     * @param {String} entryId The entry id
     * @param {DOMElement} el the clicked element
     */
    showHistory: function(uniqueId, formId, entryId, el)
    {
        DashboardHelper.dashboards[uniqueId].onShowHistory(formId, entryId, el);
    },
    
    /**
     * Open the history dialog box
     * @param {String} uniqueId the dashboard unique id
     * @param {String} formId The form id
     * @param {String} entryId The entry id
     * @param {String} formLabel The form label
     * @param {DOMElement} el the clicked element
     */
    showEntry: function(uniqueId, formId, entryId, formLabel, el)
    {
       var dashboardCfg = DashboardHelper.dashboards[uniqueId];
       
       $j.ajax({ 
            url: dashboardCfg.contextPath + "/_plugins/forms/" + dashboardCfg.lang + dashboardCfg.entryUrl, 
            type: 'POST',
            dataType: 'html',
            data: {
                formId: formId,
                entryId: entryId,
                siteName: dashboardCfg.siteName
            },
            success: function(data) { 
                var document = new DOMParser().parseFromString(data, "text/html");
                var content = $j(document).find("body").html();
                
                var titleAttr = $j("*[data-dashboard-entry-values-title = '" + uniqueId + "']").attr('data-dashboard-title-attr');
                if (titleAttr && titleAttr != 'true')
                {
                    $j("*[data-dashboard-entry-values-title = '" + uniqueId + "']").attr(titleAttr, formLabel);
                }
                else
                {
                    $j("*[data-dashboard-entry-values-title = '" + uniqueId + "']").html(formLabel);
                }
                $j("*[data-dashboard-entry-values = '" + uniqueId + "']").html(content);
                
                dashboardCfg.onShowEntry(formId, entryId, formLabel, el);
            },
            error: function(xhr, status, error)
            {
                DashboardHelper._showError(dashboardCfg, null, "{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_DASHBOARD_HELPER_ERROR_MSG}}");
            }
        });
        
        return false;
    },
    
    /**
     * Open box to edit on front the entry values
     * @param {String} uniqueId the dashboard unique id
     * @param {String} formId The form id
     * @param {String} entryId The entry id
     * @param {String} formLabel The form label
     */
    showFrontEditEntryValues: function(uniqueId, formId, entryId, actionId)
    {
        var dashboardCfg = DashboardHelper.dashboards[uniqueId];
        var popupId = formId + "-" + entryId + "-" + actionId;
        
        var $popup = $j("*[data-dashboard-edit-dialog='" + popupId + "']");
        var $loading = $popup.find(this.DIALOG_LOADING_SELECTOR);
        var $values = $j("*[data-dashboard-edit-values='" + popupId + "']");
        
        var iframeid = popupId.replace(/:\/\//g, '');
        var iframe = $j("#iframe-" + iframeid);
        if (iframe.length != 0)
        {
            iframe.remove();
        }
    
        $loading.show();
        
        var iframeSrc= dashboardCfg.isAdminDashboard 
            ? `${dashboardCfg.contextPath}/_plugins/forms/${dashboardCfg.lang}/admin-dashboard/form-entry/edit.html?actionId=${actionId}&entryId=${entryId}&uniqueId=${uniqueId}`
            : `${dashboardCfg.contextPath}/_plugins/forms/${dashboardCfg.lang}/dashboard/form-entry/edit.html?actionId=${actionId}&entryId=${entryId}&uniqueId=${uniqueId}`;
        
        iframe = $j(`<iframe id="iframe-${iframeid}" class="admin-entry-iframe" width="100%" data-id="iframe-entry-${popupId}" frameborder="0" scrolling="no"></iframe>`).appendTo($values);
        iframe.attr('src', iframeSrc);
        iframe.on('load', function() {
            // Force iframe to fit height
            iframe[0].style.height = iframe[0].contentWindow.document.body.scrollHeight + 'px';
            $loading.hide();
        });
        
        return false;
    },
    
    /**
     * Submit edition of a form entry
     * @param {String} uniqueId the dashboard unique id
     * @param {String} formId The form id
     * @param {String} entryId The entry id
     * @param {String} formLabel The form label
     */
    editFormEntry: function(uniqueId, formId, entryId, actionId)
    {
        var dashboardCfg = DashboardHelper.dashboards[uniqueId];

        var popupId = formId + "-" + entryId + "-" + actionId;
        var submitbutton = $("*[data-id='iframe-entry-" + popupId + "']").contents().find("#submit-" + uniqueId);
        if (dashboardCfg.isAdminDashboard)
        {
            submitbutton.click();
        }
        else // ask confirmation only on my submission dashboard
        {
            AmetysFront.Utils.confirm("{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_DASHBOARD_EDIT_CONFIRM_TITLE}}", "{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_DASHBOARD_EDIT_CONFIRM_MSG}}", function () {
                submitbutton.click();
            });
        }
    },
    
    /**
     * Show the dashboard
     * @param {String} uniqueId the dashboard unique id
     * @param {Function} [callback] callback function after showing dashboard
     */
    showDashboard: function(uniqueId, callback)
    {
       var dashboardCfg = DashboardHelper.dashboards[uniqueId];
       
       var $dashboard = $j("*[data-dashboard='" + uniqueId + "']");
       var $loading = $dashboard.find(this.LOADING_SELECTOR);
       var $results = $dashboard.find(this.RESULTS_SELECTOR);
       
       $j.ajax({ 
            url: dashboardCfg.contextPath + "/_plugins/forms/" + dashboardCfg.lang + "/dashboard/" + dashboardCfg.dashboardView + ".html", 
            type: 'POST',
            dataType: 'html',
            data: {
                uniqueId: uniqueId,
                zoneItemId: dashboardCfg.zoneItemId
            },
            beforeSend: function() {
                $loading.show();
            },
            success: function(data) { 
                var document = new DOMParser().parseFromString(data, "text/html");
                var results = $j(document).find("[data-dashboard='results']").html();
                
                $results.html(results);
                
                dashboardCfg.onShowDashboard();
                
                if (typeof(callback) == 'function')
                {
                    callback(uniqueId);
                }
            },
            error: function(xhr, status, error)
            {
                DashboardHelper._showError(dashboardCfg, null, "{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_DASHBOARD_HELPER_ERROR_MSG}}");
            },
            complete: function() {
                $loading.hide();
            }
        });
        
        return false;
    },
    
    /**
     * Show the admin dashboard
     * @param {String} uniqueId the dashboard unique id
     * @param {String} [formId] The form id. Can be null to display the list of forms
     * @param {Integer} [page] The page number
     * @param {Function} [callback] callback function after showing dashboard
     */
    showAdminDashboard: function(uniqueId, formId, page, callback)
    {
       var dashboardCfg = DashboardHelper.dashboards[uniqueId];
       dashboardCfg._currentPage = page;
       
       var $dashboard = $j("*[data-dashboard='" + uniqueId + "']");
       var $loading = $dashboard.find(this.LOADING_SELECTOR);
       var $results = $dashboard.find(this.RESULTS_SELECTOR);
       var $filters = $dashboard.find(this.FILTERS_SELECTOR);
       
       var submitterFilter = $filters.find('select[name=submitter]').val() || '';
       submitterFilter = submitterFilter == "_ALL" ? "" : submitterFilter;
       var workflowFilter = $filters.find('select[name=workflow]').val() || '';
       workflowFilter = workflowFilter == "-1" ? "" : workflowFilter;
       
       $j.ajax({ 
            url: dashboardCfg.contextPath + "/_plugins/forms/" + dashboardCfg.lang + "/admin-dashboard/" + dashboardCfg.dashboardView + ".html", 
            type: 'POST',
            dataType: 'html',
            data: {
                uniqueId: uniqueId,
                formId: formId,
                page: page,
                nbEntriesByPage: dashboardCfg.nbEntriesByPage,
                entriesSortOrder: dashboardCfg.entriesSortOrder,
                workflowFilter: workflowFilter,
                submitterFilter: submitterFilter,
                zoneItemId: dashboardCfg.zoneItemId
            },
            beforeSend: function() {
                $loading.show();
            },
            success: function(data) { 
                var document = new DOMParser().parseFromString(data, "text/html");
                $j(document).find("[data-dashboard='page-number']").html(page);
                var results = $j(document).find("[data-dashboard='results']").html();
                
                $results.html(results);
                dashboardCfg.onShowDashboard();
                
                if (typeof(callback) == 'function')
                {
                    callback(uniqueId, formId, page);
                }
            },
            error: function(xhr, status, error)
            {
                DashboardHelper._showError(dashboardCfg, null, "{{i18n plugin.forms:PLUGINS_FORMS_SERVICE_DASHBOARD_HELPER_ERROR_MSG}}");
            },
            complete: function() {
                DashboardHelper.initSubmitterFilter(uniqueId, formId, submitterFilter);
                DashboardHelper.initWorkflowFilter(uniqueId, formId, workflowFilter);
                $loading.hide();
            }
        });
        
        return false;
    },
    
    /**
     * Load the submitter filter and set its default value
     * @param {String} uniqueId the dashboard unique id
     * @param {String} [formId] The form id. Can be null to display the list of forms
     * @param {String} [value] The filter's value
     */
    initSubmitterFilter: function(uniqueId, formId, value)
    {
        var dashboardCfg = DashboardHelper.dashboards[uniqueId];
        var $dashboard = $j("*[data-dashboard='" + uniqueId + "']");
        var $filters = $dashboard.find(this.FILTERS_SELECTOR);

        // Reset users filter cache if come back to forms selection
        if (!formId)
        {
            dashboardCfg.usersFilter = null;
        }
        
        var $select = $filters.find('select[name=submitter]')
        if ($select.length)
        {
            if (!dashboardCfg.usersFilter)
            {
                $select.select2({
                  placeholder: "{{i18n PLUGINS_FORMS_SERVICE_ADMIN_DASHBOARD_FILTER_AUTHOR_LOADING}}"
                });
                   
                // Get users list from server
                AmetysFront.ServerComm.callMethod({
		            role: 'org.ametys.plugins.forms.dao.FormEntryDAO',
		            methodName: 'getFormEntriesUsers',
		            parameters: [formId],
		            callback: {
		                handler: function(data) {
	                        data.unshift({id:"", text:'{{i18n PLUGINS_FORMS_SERVICE_ADMIN_DASHBOARD_FILTER_AUTHOR_EMPTY_OPTION}}'});
	                        
	                        dashboardCfg.usersFilter = data;
	                        
	                        $select.select2({
	                              minimumResultsForSearch: dashboardCfg.filterMinSearchResults,
	                              data: data,
                                  width: 'style'
	                        });
	                        
	                        $select.val(value || '');
                            $select.trigger('change');
                            $select.on("change", function () { DashboardHelper.showAdminDashboard(uniqueId, formId, 1) });
                        },
		                scope: this
		            }
		        });
            }
            else
            {
                // initialize the select with the values
                $select.select2({
                      minimumResultsForSearch: dashboardCfg.filterMinSearchResults,
                      data: dashboardCfg.usersFilter
                });
                
                $select.val(value || '');
                $select.trigger('change');
                $select.on("change", function () { DashboardHelper.showAdminDashboard(uniqueId, formId, 1) });
            }
        }
    },
    
    /**
     * Initialize the wrokflow filter
     * @param {String} uniqueId the dashboard unique id
     * @param {String} [formId] The form id. Can be null to display the list of forms
     * @param {String} [value] The filter's value.
     */
    initWorkflowFilter: function(uniqueId, formId, value)
    {
        var $dashboard = $j("*[data-dashboard='" + uniqueId + "']");
        
        var $select = $dashboard.find(this.FILTERS_SELECTOR).find('select[name=workflow]');
        $select.select2({
            minimumResultsForSearch: Infinity
        });
        $select.val(value || '');
        $select.trigger('change');
        $select.on("change", function () { DashboardHelper.showAdminDashboard(uniqueId, formId, 1) });
    },
    
    /**
     * Show dialog to configure the mail to be send
     * @param {String} uniqueId the dashboard unique id
     * @param {String} formId The form id
     * @param {String} entryId The entry id
     * @param {String} actionId The action id
     * @param {DOMElement} el The clicked element to open dialog
     */
    showSendMailDialog: function(uniqueId, formId, entryId, actionId, el)
    {
       DashboardHelper.dashboards[uniqueId].onShowActionDialog('send-mail', formId, entryId, actionId, el);
    },
    
    /**
     * Show dialog to configure the comment of the action
     * @param {String} uniqueId the dashboard unique id
     * @param {String} formId The form id
     * @param {String} entryId The entry id
     * @param {String} actionId The action id
     */
    showCommentDialog: function(uniqueId, formId, entryId, actionId)
    {
       
    },
    
    /**
     * Do workflow action
     * @param {String} uniqueId the dashboard unique id
     * @param {String} type the type of action
     * @param {String} formId The form id
     * @param {String} entryId The entry id
     * @param {String} actionId The action id
     * @param {DOMElement} el The clicked element to open dialog
     */
    doWorkflowAction: function(uniqueId, type, formId, entryId, actionId, el)
    {
        if (type == 'comment')
        {
            DashboardHelper.dashboards[uniqueId].onShowActionDialog('comment', formId, entryId, actionId, el);
        }
        else if (type == 'send-mail')
        {
            DashboardHelper.dashboards[uniqueId].onShowActionDialog('send-mail', formId, entryId, actionId, el);
        }
        else if (type == 'edit' || type == 'edit-by-submitter')
        {
            DashboardHelper.showFrontEditEntryValues(uniqueId, formId, entryId, actionId);
            DashboardHelper.dashboards[uniqueId].onShowActionDialog("edit", formId, entryId, actionId, el);
        }
        else
        {
            DashboardHelper.doAction(uniqueId, formId, entryId, actionId);
        }
    },
    
    /**
     * Do workflow action for given entry id
     * @param {String} uniqueId the dashboard unique id
     * @param {String} formId The form id
     * @param {String} entryId The entry id
     * @param {String} actionId The action id
     */
    doAction: function(uniqueId, formId, entryId, actionId)
    {
       var dashboardCfg = DashboardHelper.dashboards[uniqueId];
       
       var params = {};
       var sendMailForm = $j("*[data-dashboard-send-mail-form= '" + formId + "-" + entryId + "-" + actionId + "']")
       $j.each(sendMailForm.serializeArray(), function(i, field) {
           params[field.name] = field.value;
       });
       params["entryId"] = entryId;
       params["actionId"] = actionId;
    
       var commentForm = $j("*[data-dashboard-comment-form= '" + formId + "-" + entryId + "-" + actionId + "']")
       $j.each(commentForm.serializeArray(), function(i, field) {
           params[field.name] = field.value;
       });
    
       $j.ajax({ 
            url: dashboardCfg.contextPath + "/_plugins/forms/" + dashboardCfg.lang + "/do-form-entry-action", 
            type: 'POST',
            dataType: 'html',
            data: params,
            success: function(data) { 
                
                var document = new DOMParser().parseFromString(data, "text/html");
                if ($j(document).find("success").html() != undefined)
                {
                    if (dashboardCfg.isAdminDashboard)
                    {
                        DashboardHelper.showAdminDashboard(uniqueId, formId, dashboardCfg._currentPage, function() {
                            // Put focus in cell containing workflow actions
                            let $actionsCell = $j("*[data-dashboard-actions= '" + formId + "-" + entryId + "']");
                            $actionsCell.attr('tabindex', -1);
                            $actionsCell.focus();
                        });
                    }
                    else
                    {
                        DashboardHelper.showDashboard(uniqueId, function() {
                            // Put focus in cell containing workflow edition
                            let $actionsCell = $j("*[data-dashboard-actions= '" + formId + "-" + entryId + "']");
                            $actionsCell.attr('tabindex', -1);
                            $actionsCell.focus();
                        });
                    }
                    
                    if (dashboardCfg.onHideActionDialog)
                    {
                        dashboardCfg.onHideActionDialog(formId, entryId, actionId);
                    }
                }
                else
                {
                    var errorHtml = $j(document).find("error").html();
                    DashboardHelper._showError(dashboardCfg, "{{i18n plugin.forms:PLUGINS_FORMS_WORKFLOW_ERROR}}", errorHtml);
                }
            },
            error: function(xhr, status, error)
            {  
                DashboardHelper._showError(dashboardCfg, null, "{{i18n plugin.forms:PLUGINS_FORMS_WORKFLOW_ERROR}}");
            }
        });
        
        return false;
    },
    
    /**
     * Go back to forms selection
     * @param {String} uniqueId the dashboard unique id
     */
    goBackToForms: function(uniqueId)
    {
        DashboardHelper.showAdminDashboard(uniqueId, null, null, function () {
            // replace focus
            $j(`a[name="forms-${uniqueId}"]`).focus();
        });
    },
    
    _showError: function(dashboardCfg, title, msg)
    {
        if (dashboardCfg.onShowError)
        {
            dashboardCfg.onShowError(title, msg);
        }
        else
        {
            alert(msg || title);
        }
    }
}
