/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content;

import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.xml.dom.DOMParser;
import org.apache.excalibur.xml.xpath.XPathProcessor;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.ametys.cms.data.RichText;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.repository.Content;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.data.UserEntry;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.plugins.forms.content.workflow.FormParser;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.workflow.store.JdbcWorkflowStore;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.web.repository.content.WebContent;

import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.WorkflowException;

/**
 * Class to handle forms in contents
 */
public class FormManager extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = FormManager.class.getName();
    
    /** Constant for entry workflow reinitialization */
    public static final String ENTRY_WORKFLOW_REINITIALIZATION = "workflowEntryReinitialization";
    
    
    private static final String __FORM_TYPE_CMS = "//html:form[@type='cms']";
    
    private ServiceManager _manager;
    
    private FormPropertiesManager _formPropertiesManager;
    private DOMParser _parser;
    private FormParser _formParser;
    private FormTableManager _formTableManager;
    private WorkflowProvider _workflowProvider;
    private JdbcWorkflowStore _jdbcWorkflowStore;
    private WorkflowHelper _workflowHelper;
    private Context _context;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _manager = smanager;
        _parser = (DOMParser) smanager.lookup(DOMParser.ROLE);
        XPathProcessor processor = (XPathProcessor) smanager.lookup(XPathProcessor.ROLE);
        _formParser = new FormParser(processor);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    private FormPropertiesManager getFormPropertiesManager()
    {
        if (_formPropertiesManager == null)
        {
            try
            {
                _formPropertiesManager = (FormPropertiesManager) _manager.lookup(FormPropertiesManager.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        return _formPropertiesManager;
    }
    
    private FormTableManager getFormTableManager()
    {
        if (_formTableManager == null)
        {
            try
            {
                _formTableManager = (FormTableManager) _manager.lookup(FormTableManager.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        return _formTableManager;
    }
    
    private WorkflowProvider getWorkflowProvider()
    {
        if (_workflowProvider == null)
        {
            try
            {
                _workflowProvider = (WorkflowProvider) _manager.lookup(WorkflowProvider.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        return _workflowProvider;
    }
    
    private WorkflowHelper getWorkflowHelper()
    {
        if (_workflowHelper == null)
        {
            try
            {
                _workflowHelper = (WorkflowHelper) _manager.lookup(WorkflowHelper.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        return _workflowHelper;
    }
    
    private JdbcWorkflowStore getJdbcWorkflowStore()
    {
        if (_jdbcWorkflowStore == null)
        {
            try
            {
                _jdbcWorkflowStore = (JdbcWorkflowStore) _manager.lookup(JdbcWorkflowStore.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException(e);
            }
        }
        return _jdbcWorkflowStore;
    }
    
    /**
     * Find all the forms the content contains and process them.
     * @param content the content.
     * @throws SAXException if a SAX exception occurs during content parsing.
     * @throws IOException if an I/O exception occurs during content parsing.
     * @throws WorkflowException if an exception occurs while handling the form's workflow
     */
    public void processContentForms(Content content) throws SAXException, IOException, WorkflowException
    {
        String contentName = content.getName();
        String siteName = "";
        if (content instanceof WebContent)
        {
            siteName = ((WebContent) content).getSiteName();
        }
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Processing the forms for content '" + contentName + "'");
        }
        
        List<Form> forms = new ArrayList<>();
        
        Set<RichText> richTexts = _getRichTexts(content);
        for (RichText richText : richTexts)
        {
            InputStream contentStream = richText.getInputStream(); 

            Document document = _parser.parseDocument(new InputSource(contentStream));
            
            List<Node> formNodes = _formParser.getNodesAsList(document, __FORM_TYPE_CMS);

            for (Node formNode : formNodes)
            {
                _processForm(content, contentName, siteName, forms, formNode);
            }
        }
        
        _removeUnusedForms(content, contentName, forms);
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Forms processed for content '" + contentName + "'");
        }
    }

    private void _processForm(Content content, String contentName, String siteName, List<Form> forms, Node formNode) throws WorkflowException
    {
        try
        {
            // Parse the form node.
            Form form = _formParser.parseForm(formNode);
            forms.add(form);
            
            // Try to retrieve the old form properties from the repository.
            Form oldForm = getFormPropertiesManager().getForm(siteName, form.getId());
            
            String newWorkflowName = StringUtils.defaultString(form.getWorkflowName());
            if (oldForm != null)
            {
                String oldWorkflowName = oldForm.getWorkflowName();
                if (!newWorkflowName.equals(StringUtils.defaultString(oldWorkflowName)))
                {
                    // The workflow has switched
                    boolean dropColumn = StringUtils.isEmpty(newWorkflowName) && getFormTableManager().hasWorkflowIdColumn(form.getId());
                    boolean addColumn = StringUtils.isNotEmpty(newWorkflowName) && !getFormTableManager().hasWorkflowIdColumn(form.getId());
                    
                    // update/delete the concerned tables
                    _resetWorkflowTables(content, form, dropColumn, addColumn);
                }
            }
            
            // TODO Give oldForm to the form table manager to compare old/new state (in particular for radio buttons.)
            
            // Create the table only if it was given a label.
            // Otherwise, the results will be sent by e-mail and won't be browsable.
            if (StringUtils.isNotBlank(form.getLabel()))
            {
                if (!getFormTableManager().createTable(form))
                {
                    getLogger().error("The form " + form.getLabel() + " was not created in the database.");
                }
            }
            
            if (oldForm == null)
            {
                getFormPropertiesManager().createForm(siteName, form, content);
                if (StringUtils.isNotEmpty(newWorkflowName))
                {
                    getFormTableManager().addWorkflowIdColumn(form.getId());
                }
            }
            else
            {
                getFormPropertiesManager().updateForm(siteName, form, content);
            }
        }
        catch (FormsException e)
        {
            // Form error.
            getLogger().error("Error trying to store a form in the content " + contentName + " (" + content.getId() + ")", e);
        }
        catch (SQLException e)
        {
            // SQL error.
            getLogger().error("Error trying to store a form in the content " + contentName + " (" + content.getId() + ")", e);
        }
    }
    
    /**
     * Remove the workflow tables and the workflow id column if needed
     * @param content The content holding the form
     * @param form the form 
     * @param dropColumn true to drop the workflow id column
     * @param addColumn true to add the workflow id column
     * @throws FormsException if an error occurs while retrieving the workflow instances ids 
     * @throws WorkflowException if an error occurs during the reset of the workflow of the form entries 
     * @throws SQLException if an error occurs during the SQL queries
     */
    private void _resetWorkflowTables(Content content, Form form, boolean dropColumn, boolean addColumn) throws FormsException, WorkflowException, SQLException
    {
        // Add the column just now to get the proper submissions
        if (addColumn)
        {
            getFormTableManager().addWorkflowIdColumn(form.getId());
        }

        List<UserEntry> submissions = getFormTableManager().getSubmissions(form, new HashMap<>(), 0, Integer.MAX_VALUE, null);
        
        // Delete the corresponding workflow instances and their history
        Workflow workflow = getWorkflowProvider().getExternalWorkflow(JdbcWorkflowStore.ROLE);
        for (UserEntry submission : submissions)
        {
            Integer workflowId = submission.getWorkflowId();
            if (workflowId != null && workflowId != 0)
            {
                getJdbcWorkflowStore().clearHistory(workflowId);
                getJdbcWorkflowStore().deleteInstance(workflowId);
            }

            if (getFormTableManager().hasWorkflowIdColumn(form.getId()) && !dropColumn)
            {
                String workflowName = form.getWorkflowName();
                int initialActionId = getWorkflowHelper().getInitialAction(workflowName);
                
                Map<String, Object> inputs = new HashMap<>();
                inputs.put("formId", form.getId());
                inputs.put("entryId", String.valueOf(submission.getId()));
                inputs.put("contentId", content.getId());
                inputs.put(ENTRY_WORKFLOW_REINITIALIZATION, true);
                inputs.put(ObjectModelHelper.PARENT_CONTEXT, _context);
                inputs.put(ObjectModelHelper.REQUEST_OBJECT, ContextHelper.getRequest(_context));
                long newWorkflowId = workflow.initialize(workflowName, initialActionId, inputs);
                getFormTableManager().setWorkflowId(form, submission.getId(), newWorkflowId);
            }
        }
        
        if (dropColumn)
        {
            getFormTableManager().dropWorkflowIdColumn(form.getId());
        }
    }  
    
    /**
     * Remove the unused forms
     * @param content The content
     * @param contentName The name of the content
     * @param forms The forms submitted
     */
    private void _removeUnusedForms(Content content, String contentName, List<Form> forms)
    {
        try
        {
            for (Form form : getFormPropertiesManager().getForms(content))
            {
                boolean found = false;
                for (Form form2 : forms)
                {
                    if (form2.getId().equals(form.getId()))
                    {
                        found = true;
                        break;
                    }
                }
                
                if (!found)
                {
                    getFormPropertiesManager().remove(form, content);
                }
            }
        }
        catch (FormsException e)
        {
            // Form error.
            getLogger().error("Cannot iterate on existing forms to remove unused forms on content " + contentName + " (" + content.getId() + ")", e);
        }
    }
    
    /**
     * Get the rich texts
     * @param dataHolder the data holder
     * @return the rich texts in a Set
     */
    protected Set<RichText> _getRichTexts (ModelAwareDataHolder dataHolder)
    {
        Set<RichText> richTexts = new HashSet<>();
        
        Map<String, Object> richTextValues = DataHolderHelper.findItemsByType(dataHolder, ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID);
        for (Object richTextValue : richTextValues.values())
        {
            if (richTextValue instanceof RichText[])
            {
                Arrays.stream((RichText[]) richTextValue)
                    .forEach(richTexts::add);
            }
            else if (richTextValue != null) // Test that the rich text value is not empty
            {
                richTexts.add((RichText) richTextValue);
            }
        }
        
        return richTexts;
    }
}
