/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import java.io.IOException;
import java.util.List;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.question.types.ChoicesListQuestionType;
import org.ametys.plugins.forms.question.types.MatrixQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;
import com.opensymphony.workflow.spi.Step;

/**
 * Generate the entry of a form for xls export
 */
public class FormEntriesToExportGenerator extends FormAdminEntryInformationGenerator
{
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String formId = request.getParameter("id");
        
        contentHandler.startDocument();
        
        Form form = _resolver.resolveById(formId);
        _formEntryDAO.checkHandleDataRight(form);
        
        XMLUtils.startElement(contentHandler, "form-entries");
        
        for (FormEntry entry : form.getEntries())
        {
            _saxEntry(entry);
        }
        XMLUtils.endElement(contentHandler, "form-entries");
        
        contentHandler.endDocument();
    }
    
    @Override
    protected void _checkRights(FormEntry entry)
    {
        // Do nothing ... rights is checked only one time before
    }
    
    @Override
    protected List<FormQuestion> _getQuestions(FormEntry entry, Optional<Long> currentStepId)
    {
        Form form = entry.getForm();
        return form.getQuestions()
                    .stream()
                    .filter(q -> !q.getType().onlyForDisplay(q))
                    .toList();
    }
    
    @Override
    protected void _saxQuestion(FormQuestion question, FormEntry entry) throws SAXException
    {
        FormQuestionType type = question.getType();
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("type", type.getStorageType(question));
        attrs.addCDATAAttribute("typeId", type.getId());
        attrs.addCDATAAttribute("label", question.getTitle());
        attrs.addCDATAAttribute("name", question.getNameForForm());
        
        XMLUtils.startElement(contentHandler, "field", attrs);
        
        // Hack to gain some performance in XLS export
        // This code is supposed to be call by saxEntryValue but to avoid skin migration, this code is duplicated and arranged here
        // It will be fixed in issue FORMS-661
        if (type instanceof MatrixQuestionType)
        {
            type.saxEntryValue(contentHandler, question, entry);
        }
        else
        {
            entry.dataToSAX(contentHandler, question.getNameForForm());
            if (type instanceof ChoicesListQuestionType choicesListType)
            {
                XMLUtils.startElement(contentHandler, "additional-infos");
                
                String nameForForm = question.getNameForForm();
                boolean hasOtherValue = choicesListType.hasOtherOption(question);
                if (hasOtherValue)
                {
                    String value = entry.<String>getValue(ChoicesListQuestionType.OTHER_PREFIX_DATA_NAME + nameForForm);
                    if (StringUtils.isNotBlank(value))
                    {
                        XMLUtils.createElement(contentHandler, "other", value);
                    }
                }
                
                XMLUtils.endElement(contentHandler, "additional-infos");
            }
        }
        
        XMLUtils.endElement(contentHandler, "field");
    }
    
    @Override
    protected void _addAdditionalEntryAttributes(FormEntry entry, AttributesImpl attrs)
    {
        super._addAdditionalEntryAttributes(entry, attrs);
        
        Form form = entry.getForm();
        if (form.hasWorkflow())
        {
            AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(entry);
            WorkflowDescriptor workflowDescriptor = workflow.getWorkflowDescriptor(form.getWorkflowName());
            Step currentStep = (Step) workflow.getCurrentSteps(entry.getWorkflowId()).iterator().next();
            
            StepDescriptor stepDescriptor = workflowDescriptor.getStep(currentStep.getStepId());
            I18nizableText workflowStepName = new I18nizableText("application", stepDescriptor.getName());
            attrs.addCDATAAttribute("status", _i18nUtils.translate(workflowStepName));
        }
    }
}
